/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef URI_H_
#define URI_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "BString.h"
#include <string>
#include <iosfwd>

class URI
{
public:
	enum Style { S_DEFAULT, S_RELATIVE, S_START_WITH_HOST };
	
	explicit URI(BString const& str);
	
	URI(URI const& base, URI const& relative);
	
	static BString escape(BString const& str);
	
	static BString unescape(BString const& str, bool decode_plus=false);
	
	bool isGood() const { return testFlag(IS_GOOD); }
	
	bool isAbsolute() const { return testFlag(IS_ABSOLUTE); }
	
	bool isRelativePath() const { return testFlag(IS_RELATIVE_PATH); }
	
	bool hasAuthority() const { return testFlag(HAS_AUTHORITY); }
	
	bool hasUserInfo() const { return testFlag(HAS_USER_INFO); }
	
	bool hasQuery() const { return testFlag(HAS_QUERY); }
	
	bool hasFragment() const { return testFlag(HAS_FRAGMENT); }
	
	BString const& getScheme() const { return m_scheme; }
	
	void setScheme(BString const& scheme);
	
	BString const& getRawUserInfo() const { return m_rawUserInfo; }
	
	BString getDecodedUserInfo() const { return unescape(m_rawUserInfo); }
	
	void setRawUserInfo(BString const& user_info);
	
	BString const& getHost() const { return m_host; }
	
	void setHost(BString const& host);
	
	// returns -1 if the port wasn't specified
	int getPort() const { return m_port; }
	
	// guesses an unspecified port based on scheme
	int guessPort() const;
	
	void setPort(int port);
	
	BString const& getRawPath() const { return m_rawPath; }
	
	BString getDecodedPath() const { return unescape(m_rawPath); }
	
	void setAbsoluteRawPath(BString const& path);
	
	BString const& getRawQuery() const { return m_rawQuery; }
	
	BString getDecodedQuery() const { return unescape(m_rawQuery, true); }
	
	void setRawQuery(BString const& query);
	
	void removeQuery();
	
	BString const& getRawFragment() const { return m_rawFragment; }
	
	BString getDecodedFragment() const { return unescape(m_rawFragment); }
	
	void setRawFragment(BString const& fragment);
	
	void removeFragment();
	
	// removes scheme and authority
	void makeRelative();
	
	void swap(URI& other);
	
	void toStream(std::ostream& strm, Style style = S_DEFAULT) const;
	
	std::string toString(Style style = S_DEFAULT) const;
	
	BString toBString(Style style = S_DEFAULT) const;
private:
	enum Flag {
		IS_GOOD = 1,
		IS_ABSOLUTE = 2,
		IS_RELATIVE_PATH = 4,
		HAS_AUTHORITY = 8,
		HAS_USER_INFO = 16,
		HAS_QUERY = 32,
		HAS_FRAGMENT = 64
	};
	
	bool testFlag(Flag flag) const { return m_flags & flag; }
	
	void clearFlag(Flag flag) { m_flags &= ~flag; }
	
	void clearFlags(int flags) { m_flags &= ~flags; }
	
	void setFlag(Flag flag) { m_flags |= flag; }
	
	void setFlag(Flag flag, bool state) {
		if (state) setFlag(flag); else clearFlag(flag);
	}
	
	void parse(BString const& str);
	
	char const* parseLeadingScheme(BString const& str, char const* begin, char const* end);
	
	char const* parseLeadingAuthority(BString const& str, char const* begin, char const* end);
	
	char const* parseLeadingPath(BString const& str, char const* begin, char const* end);
	
	char const* parseLeadingQuery(BString const& str, char const* begin, char const* end);
	
	void parseLeadingFragment(BString const& str, char const* begin, char const* end);
	
	void parseAuthority(BString const& str, char const* begin, char const* end);
	
	void parseHostAndPort(BString const& str, char const* begin, char const* end);
	
	static BString mergePaths(BString const& abs, BString const& rel);
	
	static char const* skipLastPathComponent(
		char const* begin, char const* end);
	
	static bool validateScheme(BString const& str);
	
	static bool validateHost(BString const& str);
	
	int m_flags;
	BString m_scheme;
	BString m_rawAuthority;
	BString m_rawUserInfo;
	BString m_host;
	int m_port;
	BString m_rawPath;
	BString m_rawQuery;
	BString m_rawFragment;
};


inline void swap(URI& o1, URI& o2)
{
	o1.swap(o2);
}

inline std::ostream&
operator<<(std::ostream& strm, URI const& uri)
{
	uri.toStream(strm);
	return strm;
}

#endif
