/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpStateRequestLine.h"
#include "HttpRequestParser.h"
#include "SplittableBuffer.h"
#include "HttpRequestMetadata.h"
#include "HttpRequestLine.h"
#include "HttpVersion.h"
#include "StringUtils.h"
#include "BString.h"
#include "URI.h"

using namespace std;

HttpStateRequestLine::HttpStateRequestLine(HttpRequestParser& parser)
:	m_rParser(parser)
{
}

HttpStateRequestLine::~HttpStateRequestLine()
{
}

HttpState*
HttpStateRequestLine::processNewData(SplittableBuffer& input, bool eof,
                                     SplittableBuffer& body_data, bool& body_eof)
{
	SplittableBuffer::ByteIterator begin = input.begin();
	SplittableBuffer::ByteIterator const end = input.end();
	SplittableBuffer::ByteIterator nl = SplittableBuffer::find(begin, end, '\n');
	if (nl == end) {
		if (eof) {
			return m_rParser.activateStateError("unexpected end of data (in HttpStateRequestLine)");
		} else {
			return this;
		}
	}
	
	BString line(SplittableBuffer::toBString(begin, nl));
	HttpState* next_state = processRequestLine(line);
	
	if (next_state) {
		input.trimFront(++nl);
	} else {
		next_state = m_rParser.activateStateError("error parsing request line");
	}
	return next_state;
}

void
HttpStateRequestLine::cleanup()
{
}

HttpState*
HttpStateRequestLine::processRequestLine(BString const& line)
{
	char const* begin = line.begin();
	char const* end = line.end();
	if (end != begin && *(end-1) == '\r') {
		--end; // end points to '\r'
	}
	
	if (begin == end) {
		// an empty request line should be ignored according to RFC2616 section 4.1
		return m_rParser.activateStateRequestStart();
	}
	
	char const* pos = StringUtils::find(begin, end, ' ');
	pos = StringUtils::find(begin, pos, '\t');
	// pos points to the end of method
	BString method(line, begin, pos);
	
	begin = pos;
	for (; begin != end && (*begin == ' ' || *begin == '\t'); ++begin) {}
	// begin points to the start of uri
	
	pos = end;
	for (; pos != begin && *(pos-1) != ' ' && *(pos-1) != '\t'; --pos) {}
	// pos points to the beginning of the protocol
	
	HttpVersion http_version = HttpVersion::HTTP_0_9;
	if (http_version.parse(pos, end)) {
		end = pos;
		for (; end != begin && (*(end-1) == ' ' || *(end-1) == '\t'); --end) {}
		// end points to the end of uri
	} // otherwise it's an HTTP/0.9 request
	
	BString uri(line, begin, end);
	if (method == BString("CONNECT")) {
		// In case of CONNECT, the URI we get is not really a URI,
		// but just a host:port string. If we pass it to URI constructor
		// as is, it will be interpreted as scheme:path or maybe
		// as just path. To force it to be interpreted the right way,
		// we apply this hack.
		uri = BString("https://")+uri;
	}
	
	return m_rParser.activateStateHeaders(HttpRequestLine(method, URI(uri), http_version));
}
