/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPPARSERBASE_H_
#define HTTPPARSERBASE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpState.h"
#include "HttpStateError.h"
#include "HttpStateInactive.h"
#include "HttpStateSizedFlatBody.h"
#include "HttpStateUnsizedFlatBody.h"
#include "HttpStateChunkHeader.h"
#include "HttpStateChunkBody.h"
#include "HttpStateFooters.h"
#include "ErrorDescriptor.h"
#include "IntrusivePtr.h"
#include <string>

class SplittableBuffer;

class HttpParserBase
{
public:
	enum Status {
		DATA_PROCESSED,
		NEED_MORE_DATA,
		MESSAGE_ENDS,
		ERROR_STATE
	};
	
	HttpParserBase();
	
	Status processNewData(
		SplittableBuffer& input, bool eof,
		SplittableBuffer& body, bool& body_eof);
	
	HttpState* getState() { return m_pCurState; }
	
	HttpState const* getState() const { return m_pCurState; }
	
	bool isInactiveState() const { return m_pCurState == &m_stateInactive; }
	
	bool isErrorState() const { return m_pCurState == &m_stateError; }
	
	void switchToInactive();
	
	void switchToUnsizedFlatBody();
	
	IntrusivePtr<ErrorDescriptor>& errorDescriptor() {
		return m_stateError.errorDescriptor();
	}
	
	IntrusivePtr<ErrorDescriptor> const& errorDescriptor() const {
		return m_stateError.errorDescriptor();
	}
	
	std::string const& errorMessage() const {
		return m_stateError.errorMessage();
	}
	
	HttpState* activateStateInactive();
	
	HttpState* activateStateError(IntrusivePtr<ErrorDescriptor> const& err_desc);
	
	HttpState* activateStateError(std::string const& errmsg);
	
	HttpState* activateStateSizedFlatBody(uintmax_t body_length);
	
	HttpState* activateStateUnsizedFlatBody();
	
	HttpState* activateStateChunkHeader();
	
	HttpState* activateStateChunkBody(uintmax_t chunk_length);
	
	HttpState* activateStateFooters();
protected:
	~HttpParserBase();

	void setState(HttpState* state) { m_pCurState = state; }
private:
	HttpStateInactive        m_stateInactive;
	HttpStateError           m_stateError;
	HttpStateSizedFlatBody   m_stateSizedFlatBody;
	HttpStateUnsizedFlatBody m_stateUnsizedFlatBody;
	HttpStateChunkHeader     m_stateChunkHeader;
	HttpStateChunkBody       m_stateChunkBody;
	HttpStateFooters         m_stateFooters;
	HttpState*               m_pCurState;
};

inline HttpState*
HttpParserBase::activateStateInactive()
{
	m_stateInactive.activate();
	return &m_stateInactive;
}

inline HttpState*
HttpParserBase::activateStateError(IntrusivePtr<ErrorDescriptor> const& err_desc)
{
	m_stateError.activate(err_desc);
	return &m_stateError;
}

inline HttpState*
HttpParserBase::activateStateError(std::string const& errmsg)
{
	m_stateError.activate(errmsg);
	return &m_stateError;
}

inline HttpState*
HttpParserBase::activateStateSizedFlatBody(uintmax_t body_length)
{
	m_stateSizedFlatBody.activate(body_length);
	return &m_stateSizedFlatBody;
}

inline HttpState*
HttpParserBase::activateStateUnsizedFlatBody()
{
	m_stateUnsizedFlatBody.activate();
	return &m_stateUnsizedFlatBody;
}

inline HttpState*
HttpParserBase::activateStateChunkHeader()
{
	m_stateChunkHeader.activate();
	return &m_stateChunkHeader;
}

inline HttpState*
HttpParserBase::activateStateChunkBody(uintmax_t chunk_length)
{
	m_stateChunkBody.activate(chunk_length);
	return &m_stateChunkBody;
}

inline HttpState*
HttpParserBase::activateStateFooters()
{
	m_stateFooters.activate();
	return &m_stateFooters;
}

#endif
