/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AdvancedConfigWindow.h"
#include "AutoScrollingWindow.h"
#include "Application.h"
#include "OperationLog.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "UrlPatterns.h"
#include "UrlPatternsFile.h"
#include "SharedPtr.h"
#include "GlobalState.h"
#include "CompiledImages.h"
#include "EffectiveFileTimestamps.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_sys_stat.h>
#include <glibmm/convert.h>
#include <gtkmm/textview.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/button.h>
#include <gtkmm/alignment.h>
#include <gtkmm/label.h>
#include <gtkmm/stock.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/frame.h>
#include <gtkmm/notebook.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/messagedialog.h>
#include <pangomm/fontdescription.h>
#include <gdk/gdkkeysyms.h>
#include <memory>
#include <string>
#include <sstream>
#include <cassert>
#include <stddef.h>
#include <time.h>

using namespace std;

class AdvancedConfigWindow::Button : public Gtk::Button
{
public:
	Button() : Gtk::Button() {}
	
	Button(Gtk::StockID const& stock_id) : Gtk::Button(stock_id) {}
	
	Button(Glib::ustring const& label, bool mnemonic=false)
	: Gtk::Button(label, mnemonic) {}
	
	void setBold(bool bold = true);
private:
	Pango::FontDescription m_font;
};


class AdvancedConfigWindow::Page : public Gtk::ScrolledWindow
{
public:
	Page(AdvancedConfigWindow* window, bool readonly,
		Gtk::Notebook* notebook, int notebook_page);
	
	~Page();
	
	bool isReadOnly() const { return m_isReadOnly; }
	
	bool isPageSelected() const;
	
	void getCaretPosition(long* line, long* col);
	
	std::string getText() const;
	
	void setText(Glib::ustring const& text);
	
	void focusText() { m_pTextView->grab_focus(); }
	
	bool isTextModified() const { return m_ptrTextBuffer->get_modified(); }
	
	void setTextUnmodified() { m_ptrTextBuffer->set_modified(false); }
	
	bool isModified() const { return m_isModified; }
	
	time_t getFileLoadTimestamp() const { return m_fileLoadTimestamp; }
	
	void setFileLoadTimestamp(time_t val) { m_fileLoadTimestamp = val; }
	
	time_t getFileCurrentTimestamp() const { return m_fileCurrentTimestamp; }

	void setFileCurrentTimestamp(time_t val) { m_fileCurrentTimestamp = val; }
	
	void unselectTextOnce();
	
	void updateState();
	
	bool updateFileCurrentTimestamp(); // returns true if changed
	
	virtual Glib::ustring const& getTabName() const = 0;
	
	virtual time_t getFileEffectiveTimestamp() const { return 0; }
	
	virtual time_t readFileCurrentTimestamp() const { return 0; }
	
	virtual void loadFile() = 0;
	
	virtual void onReload() {}
	
	virtual void onSave() {}
private:
	AdvancedConfigWindow* const m_pWindow;
	bool const m_isReadOnly;
	Gtk::Notebook* const m_pNotebook;
	int const m_notebookPage;
	Glib::RefPtr<Gtk::TextBuffer> m_ptrTextBuffer;
	Gtk::TextView* m_pTextView;
	time_t m_fileLoadTimestamp;
	time_t m_fileCurrentTimestamp;
	bool m_isModified;
};


class AdvancedConfigWindow::ConfigPage : public AdvancedConfigWindow::Page
{
public:
	ConfigPage(
		AdvancedConfigWindow* window, bool readonly,
		Gtk::Notebook* notebook, size_t notebook_page);
	
	~ConfigPage();
	
	virtual Glib::ustring const& getTabName() const;
	
	virtual time_t getFileEffectiveTimestamp() const;
	
	virtual time_t readFileCurrentTimestamp() const;
	
	virtual void loadFile();
	
	virtual void onReload();
	
	virtual void onSave();
private:
	bool restoreOldConfig(Config const& config);
	
	Glib::ustring m_unmodifiedTabName;
	Glib::ustring m_modifiedTabName;
};


class AdvancedConfigWindow::ConfigDefaultPage : public AdvancedConfigWindow::Page
{
public:
	ConfigDefaultPage(
		AdvancedConfigWindow* window, bool readonly,
		Gtk::Notebook* notebook, size_t notebook_page);
	
	~ConfigDefaultPage();
	
	virtual Glib::ustring const& getTabName() const;
	
	virtual void loadFile();
private:
	Glib::ustring m_tabName;
};


class AdvancedConfigWindow::UrlsPage : public AdvancedConfigWindow::Page
{
public:
	UrlsPage(
		AdvancedConfigWindow* window, bool readonly,
		Gtk::Notebook* notebook, size_t notebook_page);
	
	~UrlsPage();
	
	virtual Glib::ustring const& getTabName() const;
	
	virtual void loadFile();
private:
	Glib::ustring m_tabName;
};


class AdvancedConfigWindow::UrlsLocalPage : public AdvancedConfigWindow::Page
{
public:
	UrlsLocalPage(
		AdvancedConfigWindow* window, bool readonly,
		Gtk::Notebook* notebook, size_t notebook_page);
	
	~UrlsLocalPage();
	
	virtual Glib::ustring const& getTabName() const;
	
	virtual time_t getFileEffectiveTimestamp() const;
	
	virtual time_t readFileCurrentTimestamp() const;
	
	virtual void loadFile();
	
	virtual void onReload();
	
	virtual void onSave();
private:
	Glib::ustring m_unmodifiedTabName;
	Glib::ustring m_modifiedTabName;
};


AdvancedConfigWindow* AdvancedConfigWindow::m_spInstance = 0;

AdvancedConfigWindow::AdvancedConfigWindow()
:	AbstractLogView(*OperationLog::instance()),
	m_isConstructionFinished(false),
	m_pNotebook(0),
	m_pReloadButton(0),
	m_pSaveButton(0),
	m_pLogView(0),
	m_pLineStatusbar(0),
	m_pColStatusbar(0),
	m_caretLine(-1),
	m_caretCol(-1)
{
	set_title("Advanced Configuration");
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	
	signal_focus_in_event().connect(
		sigc::mem_fun(*this, &AdvancedConfigWindow::onFocusChange)
	);
	signal_focus_out_event().connect(
		sigc::mem_fun(*this, &AdvancedConfigWindow::onFocusChange)
	);
	
	Gtk::VBox* vbox = manage(new Gtk::VBox);
	add(*vbox);
	m_pNotebook = manage(new Gtk::Notebook);
	vbox->pack_start(*m_pNotebook);
	
	Page* config_page = manage(new ConfigPage(this, false, m_pNotebook, 0));
	m_pNotebook->append_page(*config_page, config_page->getTabName());
	Page* config_default_page = manage(new ConfigDefaultPage(this, true, m_pNotebook, 1));
	m_pNotebook->append_page(*config_default_page, config_default_page->getTabName());
	Page* urls_page = manage(new UrlsPage(this, true, m_pNotebook, 2));
	m_pNotebook->append_page(*urls_page, urls_page->getTabName());
	Page* urls_local_page = manage(new UrlsLocalPage(this, false, m_pNotebook, 3));
	m_pNotebook->append_page(*urls_local_page, urls_local_page->getTabName());
	m_pNotebook->signal_switch_page().connect(
		sigc::hide(sigc::hide(
			sigc::mem_fun(*this, &AdvancedConfigWindow::onTabChanged)
		))
	);
	
	Gtk::Alignment* spacer1 = manage(new Gtk::Alignment);
	vbox->pack_start(*spacer1, Gtk::PACK_SHRINK);
	spacer1->set_size_request(5, 5);
	
	Gtk::HBox* hbox1 = manage(new Gtk::HBox(true));
	vbox->pack_start(*hbox1, Gtk::PACK_SHRINK);
	
	Gtk::HBox* hbox2 = manage(new Gtk::HBox);
	hbox1->pack_start(*hbox2);
	
	Gtk::Alignment* spacer2 = manage(new Gtk::Alignment);
	hbox2->pack_start(*spacer2, Gtk::PACK_SHRINK);
	spacer2->set_size_request(5, 5);
	
	Gtk::Alignment* align1 = manage(new Gtk::Alignment(0.0, 1.0, 0.0, 0.0));
	hbox2->pack_start(*align1);
	
	Gtk::Button* clear_log_btn = manage(new Gtk::Button("Clear Log"));
	align1->add(*clear_log_btn);
	clear_log_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &AdvancedConfigWindow::onClearLog)
	);
	
	Gtk::Alignment* align2 = manage(new Gtk::Alignment(0.5, 0.5, 0.0, 0.0));
	hbox1->pack_start(*align2);
	hbox1->pack_start(*manage(new Gtk::Alignment));
	
	Gtk::HButtonBox* bbox2 = manage(new Gtk::HButtonBox);
	align2->add(*bbox2);
	bbox2->set_spacing(10);
	
	m_pReloadButton = manage(new Button("Reload"));
	bbox2->pack_start(*m_pReloadButton);
	m_pReloadButton->set_sensitive(false);
	m_pReloadButton->signal_clicked().connect(
		sigc::mem_fun(*this, &AdvancedConfigWindow::onReload)
	);
	
	m_pSaveButton = manage(new Button("Save"));
	bbox2->pack_start(*m_pSaveButton);
	m_pSaveButton->set_sensitive(false);
	m_pSaveButton->signal_clicked().connect(
		sigc::mem_fun(*this, &AdvancedConfigWindow::onSave)
	);
	m_pSaveButton->add_accelerator(
		"clicked", get_accel_group(),
		GDK_s, Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE
	);
	
	Gtk::Frame* log_frame = manage(new Gtk::Frame(" Log "));
	vbox->pack_start(*log_frame);
	log_frame->set_border_width(1);
	
	AutoScrollingWindow* log_scroll_window = manage(new AutoScrollingWindow);
	log_frame->add(*log_scroll_window);
	log_scroll_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	log_scroll_window->set_shadow_type(Gtk::SHADOW_IN);
	log_scroll_window->set_size_request(570, 100);
	log_scroll_window->set_border_width(2);
	
	m_pLogView = manage(
		new Gtk::TextView(OperationLog::instance()->getTextBuffer()
	));
	log_scroll_window->add(*m_pLogView);
	m_pLogView->set_editable(false);
	m_pLogView->set_cursor_visible(false);
	m_pLogView->set_wrap_mode(Gtk::WRAP_WORD);
	
	Gtk::HBox* statusbar_hbox = manage(new Gtk::HBox);
	vbox->pack_end(*statusbar_hbox, Gtk::PACK_SHRINK);
	
	Gtk::Statusbar* empty_statusbar = manage(new Gtk::Statusbar);
	statusbar_hbox->pack_start(*empty_statusbar);
	empty_statusbar->set_has_resize_grip(false);
	
	m_pLineStatusbar = manage(new Gtk::Statusbar);
	statusbar_hbox->pack_start(*m_pLineStatusbar, Gtk::PACK_SHRINK);
	m_pLineStatusbar->set_has_resize_grip(false);
	m_pLineStatusbar->set_size_request(70);
	
	m_pColStatusbar = manage(new Gtk::Statusbar);
	statusbar_hbox->pack_start(*m_pColStatusbar, Gtk::PACK_SHRINK);
	m_pColStatusbar->set_size_request(75);
	
	config_page->loadFile();
	config_default_page->loadFile();
	urls_page->loadFile();
	urls_local_page->loadFile();
	
	updateLineColInfo(config_page);
	
	show_all_children();
	config_page->focusText();
	
	Application::instance()->destroySignal().connect(
		sigc::mem_fun(*this, &AdvancedConfigWindow::onAppDestroy)
	);
	
	m_isConstructionFinished = true;
}

AdvancedConfigWindow::~AdvancedConfigWindow()
{
	assert(m_spInstance);
	m_spInstance = 0;
	
	// prevent slots being called during destruction
	sigc::trackable::notify_callbacks();
}

void
AdvancedConfigWindow::showWindow()
{
	if (m_spInstance) {
		m_spInstance->present();
	} else {
		(m_spInstance = new AdvancedConfigWindow)->show();
	}
}

bool
AdvancedConfigWindow::on_delete_event(GdkEventAny*)
{
	if (prepareForWindowDestruction()) {
		delete this;
	}
	return true; // stop event propagation
}

bool
AdvancedConfigWindow::prepareForWindowDestruction()
{
	if (!haveModifiedPages()) {
		return true;
	}
	
	Gtk::MessageDialog dialog(
		*this,
		"Are you sure you want to close this window?\n"
		"Any <b>unsaved changes</b> will be lost.",
		true, /* use_markup */
		Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO,
		true /* modal */
	);
	int const response = dialog.run();
	return (response == Gtk::RESPONSE_YES);
}

bool
AdvancedConfigWindow::onFocusChange(GdkEventFocus*)
{
	bool const active = property_is_active();
	AbstractLogView::reportVisibility(active);
	
	if (active) {
		for (int i = m_pNotebook->get_n_pages() - 1; i >= 0; --i) {
			Page* page = getPage(i);
			if (page->updateFileCurrentTimestamp()) {
				page->updateState();
			}
		}
	}
	
	return false; // propagate event further
}

AdvancedConfigWindow::Page*
AdvancedConfigWindow::getPage(int idx)
{
	Gtk::Widget* widget = m_pNotebook->get_nth_page(idx);
	return dynamic_cast<Page*>(widget);
}

AdvancedConfigWindow::Page const*
AdvancedConfigWindow::getPage(int idx) const
{
	Gtk::Widget* widget = m_pNotebook->get_nth_page(idx);
	return dynamic_cast<Page const*>(widget);
}

AdvancedConfigWindow::Page*
AdvancedConfigWindow::getCurrentPage()
{
	return getPage(m_pNotebook->get_current_page());
}

bool
AdvancedConfigWindow::haveModifiedPages() const
{
	for (int i = m_pNotebook->get_n_pages() - 1; i >= 0; --i) {
		if (getPage(i)->isTextModified()) {
			return true;
		}
	}
	return false;
}

void
AdvancedConfigWindow::updateLineColInfo(Page* page)
{
	long line = 0, col = 0;
	page->getCaretPosition(&line, &col);
	if (line != m_caretLine) {
		m_caretLine = line;
		ostringstream strm;
		strm << m_caretLine;
		m_pLineStatusbar->pop();
		m_pLineStatusbar->push(Glib::ustring("Line: ")+strm.str());
	}
	if (col != m_caretCol) {
		m_caretCol = col;
		ostringstream strm;
		strm << m_caretCol;
		m_pColStatusbar->pop();
		m_pColStatusbar->push(Glib::ustring("Col: ")+strm.str());
	}
}

void
AdvancedConfigWindow::onTabChanged()
{
	if (!m_isConstructionFinished) {
		return;
	}
	Page* page = getCurrentPage();
	updateLineColInfo(page);
	page->updateState();
}

void
AdvancedConfigWindow::onCaretMove()
{
	updateLineColInfo(getCurrentPage());
}

void
AdvancedConfigWindow::onReload()
{
	Page* page = getCurrentPage();
	page->onReload();
	page->focusText();
}

void
AdvancedConfigWindow::onSave()
{
	Page* page = getCurrentPage();
	page->onSave();
	page->focusText();
}

void
AdvancedConfigWindow::onClearLog()
{
	OperationLog::instance()->clear();
}

void
AdvancedConfigWindow::onAppDestroy()
{
	delete this;
}


/*================== AdvancedConfigWindow::Button =======================*/

void
AdvancedConfigWindow::Button::setBold(bool bold)
{
	Pango::Weight weight = bold ? Pango::WEIGHT_BOLD : Pango::WEIGHT_NORMAL;
	if (m_font.get_weight() != weight) {
		m_font.set_weight(weight);
		get_child()->modify_font(m_font);
	}
}


/*=================== AdvancedConfigWindow::Page ========================*/

AdvancedConfigWindow::Page::Page(
	AdvancedConfigWindow* window, bool readonly,
	Gtk::Notebook* notebook, int notebook_page)
:	m_pWindow(window),
	m_isReadOnly(readonly),
	m_pNotebook(notebook),
	m_notebookPage(notebook_page),
	m_ptrTextBuffer(Gtk::TextBuffer::create()),
	m_pTextView(0),
	m_fileLoadTimestamp(0),
	m_fileCurrentTimestamp(0),
	m_isModified(false)
{
	set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	set_shadow_type(Gtk::SHADOW_IN);
	set_size_request(570, 200);
	
	m_pTextView = manage(new Gtk::TextView(m_ptrTextBuffer));
	add(*m_pTextView);
	m_pTextView->set_editable(!m_isReadOnly);
	m_ptrTextBuffer->signal_modified_changed().connect(
		sigc::mem_fun(*this, &Page::updateState)
	);
	m_ptrTextBuffer->signal_changed().connect(
		sigc::mem_fun(*this, &Page::updateState)
	);
	m_ptrTextBuffer->signal_mark_set().connect(
		sigc::hide(sigc::hide(
			sigc::mem_fun(*m_pWindow, &AdvancedConfigWindow::onCaretMove)
		))
	);
}

AdvancedConfigWindow::Page::~Page()
{
	// prevent slots being called during destruction
	sigc::trackable::notify_callbacks();
}

void
AdvancedConfigWindow::Page::getCaretPosition(long* line, long* col)
{
	Gtk::TextBuffer::iterator pos = m_ptrTextBuffer->get_insert()->get_iter();
	*line = pos.get_line() + 1;
	*col = pos.get_line_offset() + 1;
}

bool
AdvancedConfigWindow::Page::isPageSelected() const
{
	return m_pNotebook->get_current_page() == m_notebookPage;
}

std::string
AdvancedConfigWindow::Page::getText() const
{
	return Application::localeFromUtf8(m_ptrTextBuffer->get_text());
}

void
AdvancedConfigWindow::Page::setText(Glib::ustring const& text)
{
	m_ptrTextBuffer->set_text(text);
	m_ptrTextBuffer->set_modified(false);
	m_ptrTextBuffer->place_cursor(m_ptrTextBuffer->begin());
}

void
AdvancedConfigWindow::Page::updateState()
{
	bool const text_modified = isTextModified();
	bool const file_is_newer =
		(m_fileCurrentTimestamp != m_fileLoadTimestamp);
	bool const file_not_effective =
		(getFileEffectiveTimestamp() != m_fileLoadTimestamp);
	
	bool const was_modified = m_isModified;
	m_isModified = text_modified || file_not_effective;
	if (was_modified != m_isModified) {
		m_pNotebook->set_tab_label_text(*this, getTabName());
	}
	
	if (isPageSelected()) {
		Button* reload_btn = m_pWindow->getReloadButton();
		Button* save_btn = m_pWindow->getSaveButton();
		if (!m_isReadOnly) {
			reload_btn->set_sensitive(text_modified || file_is_newer);
			reload_btn->setBold(file_is_newer);
			save_btn->set_sensitive(m_isModified);
		}
		if (m_isReadOnly) {
			reload_btn->hide();
			save_btn->hide();
		} else {
			reload_btn->show();
			save_btn->show();
		}
	}
}

bool
AdvancedConfigWindow::Page::updateFileCurrentTimestamp()
{
	time_t const new_tstamp = readFileCurrentTimestamp();
	time_t const old_tstamp = getFileCurrentTimestamp();
	setFileCurrentTimestamp(new_tstamp);
	return (old_tstamp != new_tstamp);
}


/*================ AdvancedConfigWindow::ConfigPage ====================*/

AdvancedConfigWindow::ConfigPage::ConfigPage(
	AdvancedConfigWindow* window, bool readonly,
	Gtk::Notebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_unmodifiedTabName("config"),
	m_modifiedTabName("config *")
{
}

AdvancedConfigWindow::ConfigPage::~ConfigPage()
{
}

Glib::ustring const&
AdvancedConfigWindow::ConfigPage::getTabName() const
{
	return isModified() ? m_modifiedTabName : m_unmodifiedTabName;
}

time_t
AdvancedConfigWindow::ConfigPage::getFileEffectiveTimestamp() const
{
	return EffectiveFileTimestamps::config;
}

time_t
AdvancedConfigWindow::ConfigPage::readFileCurrentTimestamp() const
{
	Application& app = *Application::instance();
	ACE_stat st;
	if (ACE_OS::stat(app.getConfigFileName().c_str(), &st) != -1) {
		return st.st_mtime;
	}
	return 0;
}

void
AdvancedConfigWindow::ConfigPage::loadFile()
{
	Application& app = *Application::instance();
	string content;
	time_t mtime = 0;
	app.readFile(app.getConfigFileName(), content, &mtime);
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	setText(app.localeToUtf8(content));
}

void
AdvancedConfigWindow::ConfigPage::onReload()
{
	loadFile();
}

void
AdvancedConfigWindow::ConfigPage::onSave()
{
	Application& app = *Application::instance();
	Log* log = OperationLog::instance();
	log->appendRecord("Applying new config ... ");
	size_t num_records = log->getNumRecords();
	
	string const text = getText();
	Config new_config;
	ConfigFile new_config_file;
	
	if (!app.processConfig(text, new_config, new_config_file)) {
		return;
	}
	
	Config old_config(GlobalState::ReadAccessor()->config());
	
	if (!app.applyConfig(new_config)) {
		restoreOldConfig(old_config);
		return;
	}
	
	time_t mtime = 0;
	if (!app.writeFile(app.getConfigFileName(), text, &mtime)) {
		restoreOldConfig(old_config);
		return;
	}
	
	app.configFile().swap(new_config_file);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	EffectiveFileTimestamps::config = mtime;
	setTextUnmodified();
	updateState();
}

bool
AdvancedConfigWindow::ConfigPage::restoreOldConfig(Config const& config)
{
	Application& app = *Application::instance();
	Log* log = OperationLog::instance();
	log->appendRecord("Restoring the old config ... ");
	size_t num_records = log->getNumRecords();
	
	if (!app.applyConfig(config)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	return true;
}


/*============= AdvancedConfigWindow::ConfigDefaultPage =================*/

AdvancedConfigWindow::ConfigDefaultPage::ConfigDefaultPage(
	AdvancedConfigWindow* window, bool readonly,
	Gtk::Notebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_tabName("config.default")
{
}

AdvancedConfigWindow::ConfigDefaultPage::~ConfigDefaultPage()
{
}

Glib::ustring const&
AdvancedConfigWindow::ConfigDefaultPage::getTabName() const
{
	return m_tabName;
}

void
AdvancedConfigWindow::ConfigDefaultPage::loadFile()
{
	Application& app = *Application::instance();
	string content;
	app.readFile(app.getConfigDefaultFileName(), content);
	setText(app.localeToUtf8(content));
}


/*================= AdvancedConfigWindow::UrlsPage ===================*/

AdvancedConfigWindow::UrlsPage::UrlsPage(
	AdvancedConfigWindow* window, bool readonly,
	Gtk::Notebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_tabName("urls")
{
}

AdvancedConfigWindow::UrlsPage::~UrlsPage()
{
}

Glib::ustring const&
AdvancedConfigWindow::UrlsPage::getTabName() const
{
	return m_tabName;
}

void
AdvancedConfigWindow::UrlsPage::loadFile()
{
	Application& app = *Application::instance();
	string content;
	app.readFile(app.getStandardUrlPatternsFileName(), content);
	setText(app.localeToUtf8(content));
}


/*=============== AdvancedConfigWindow::UrlsLocalPage ===================*/

AdvancedConfigWindow::UrlsLocalPage::UrlsLocalPage(
	AdvancedConfigWindow* window, bool readonly,
	Gtk::Notebook* notebook, size_t notebook_page)
:	Page(window, readonly, notebook, notebook_page),
	m_unmodifiedTabName("urls.local"),
	m_modifiedTabName("urls.local *")
{
}

AdvancedConfigWindow::UrlsLocalPage::~UrlsLocalPage()
{
}

Glib::ustring const&
AdvancedConfigWindow::UrlsLocalPage::getTabName() const
{
	return isModified() ? m_modifiedTabName : m_unmodifiedTabName;
}

time_t
AdvancedConfigWindow::UrlsLocalPage::getFileEffectiveTimestamp() const
{
	return EffectiveFileTimestamps::urls_local;
}

time_t
AdvancedConfigWindow::UrlsLocalPage::readFileCurrentTimestamp() const
{
	Application& app = *Application::instance();
	ACE_stat st;
	if (ACE_OS::stat(app.getLocalUrlPatternsFileName().c_str(), &st) != -1) {
		return st.st_mtime;
	}
	return 0;
}

void
AdvancedConfigWindow::UrlsLocalPage::loadFile()
{
	Application& app = *Application::instance();
	string content;
	time_t mtime = 0;
	app.readFile(app.getLocalUrlPatternsFileName(), content, &mtime);
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	setText(app.localeToUtf8(content));
}

void
AdvancedConfigWindow::UrlsLocalPage::onReload()
{
	loadFile();
}

void
AdvancedConfigWindow::UrlsLocalPage::onSave()
{
	Application& app = *Application::instance();
	Log* log = OperationLog::instance();
	log->appendRecord("Applying new local url patterns ... ");
	size_t num_records = log->getNumRecords();
	
	string const text = getText();
	UrlPatterns new_patterns;
	UrlPatternsFile new_patterns_file;
	
	if (!app.processUrlPatterns(text, "urls.local",
		                    new_patterns, new_patterns_file)) {
		return;
	}
	
	time_t mtime = 0;
	if (!app.writeFile(app.getLocalUrlPatternsFileName(), text, &mtime)) {
		return;
	}
	
	app.applyLocalUrlPatterns(new_patterns);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	setFileLoadTimestamp(mtime);
	setFileCurrentTimestamp(mtime);
	EffectiveFileTimestamps::urls_local = mtime;
	setTextUnmodified();
	updateState();
}
