/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>
#include <fstream>
#include "lexgen.h"
#include "NfaChar.h"
#include "NfaCharClass.h"
#include "NfaNegatedCharClass.h"
#include "NfaConcatenation.h"
#include "NfaClosure.h"
#include "NfaPositiveClosure.h"
#include "NfaString.h"
#include "NfaInsensitiveString.h"
#include "NfaUnion.h"
#include "NfaEpsilon.h"

#include "ConfLexerDefinitions.h"

int main(int argc, char** argv)
{
	if (argc < 8) {
		std::cerr << "Usage: makelexer <OutputClass> <out.h> <out.cpp> "
			"<Definitions> <definitions.h> <Subclass> <subclass.h>" << std::endl;
		return 1;
	}
	const char* out_class = argv[1];
	const char* out_header = argv[2];
	const char* out_impl = argv[3];
	const char* def_class = argv[4];
	const char* def_header = argv[5];
	const char* subclass = argv[6];
	const char* subclass_header = argv[7];
	
	std::ofstream header_strm(out_header);
	if (header_strm.fail()) {
		std::cerr << "Failed opening " << out_header << " for writing" << std::endl;
	}
	
	std::ofstream impl_strm(out_impl);
	if (impl_strm.fail()) {
		std::cerr << "Failed opening " << out_impl << " for writing" << std::endl;
	}
	
	LexGen lexgen(out_class, subclass);
	lexgen.enableTrackingTokenPosition();
	
	NfaCharClass space(" \t");
	NfaCharClass comm_begin(";#");
	NfaChar LF('\n');
	NfaString CRLF("\r\n");
	NfaUnion newline(CRLF, LF);
	NfaCharClass alpha('a', 'z');
	alpha.addChars('A', 'Z');
	NfaCharClass key_chars = alpha;
	key_chars.addChars('0', '9');
	key_chars.addChars("-_");
	NfaConcatenation heredoc_ident(alpha, NfaClosure(key_chars));
	NfaConcatenation heredoc_begin(NfaString("<<"), heredoc_ident);
	NfaNegatedCharClass any;
	NfaEpsilon epsilon;
	
	typedef ConfLexerDefinitions Defs;

	lexgen.addRule(Defs::INITIAL, NfaPositiveClosure(space), "/* skip spaces */");
	lexgen.addRule(Defs::INITIAL, newline,
		"obj->processBlankLine(); obj->newLine(tok_end);"
	);
	lexgen.addRule(Defs::INITIAL, comm_begin,
		"BEGIN(COMMENT); obj->setCommentSymbol(*tok_begin);"
	);
	lexgen.addRule(Defs::INITIAL, alpha, "BEGIN(KEY); CONSUME_NOTHING();");
	lexgen.addRule(Defs::INITIAL, NfaChar('['),
		"if (obj->isAtLineStart(tok_begin))\n\t"
		"BEGIN(SECTION);\n\t"
		"else { BEGIN(RECOVERY); obj->processBrokenSection(); }"
	);
	lexgen.addRule(Defs::INITIAL, any,
		"BEGIN(RECOVERY); CONSUME_NOTHING(); obj->processUnexpectedCharacter(tok_begin);");
	
	
	lexgen.addRule(Defs::COMMENT, NfaConcatenation(NfaClosure(any), CRLF),
		"BEGIN(INITIAL); obj->processComment(tok_begin, tok_end-2); obj->newLine(tok_end);"
	).setLazy();
	lexgen.addRule(Defs::COMMENT, NfaConcatenation(NfaClosure(any), LF),
		"BEGIN(INITIAL); obj->processComment(tok_begin, tok_end-1); obj->newLine(tok_end);"
	).setLazy();
	lexgen.addRule(Defs::COMMENT, NfaClosure(any),
		"BEGIN(INITIAL); obj->processComment(tok_begin, tok_end);"
	);
	
	
	lexgen.addRule(Defs::SECTION, NfaConcatenation(NfaClosure(NfaNegatedCharClass("\n")), NfaChar(']')),
		"BEGIN(AFTER_SECTION_OR_VALUE); obj->processSection(tok_begin, tok_end-1);"
	);
	lexgen.addRule(Defs::SECTION, epsilon, "BEGIN(RECOVERY); obj->processBrokenSection();");
	lexgen.addRule(Defs::KEY, NfaConcatenation(alpha, NfaClosure(key_chars)),
		"BEGIN(EQUAL); obj->processKey(tok_begin, tok_end);"
	);
	
	
	lexgen.addRule(Defs::EQUAL, NfaPositiveClosure(space), "/* ignore spaces */");
	lexgen.addRule(Defs::EQUAL, NfaConcatenation(NfaChar('='), NfaClosure(space)),
		"BEGIN(VALUE);"
	);
	lexgen.addRule(Defs::EQUAL, any,
		"BEGIN(RECOVERY); CONSUME_NOTHING(); obj->processUnexpectedCharacter(tok_begin);"
	);
	
	
	lexgen.addRule(Defs::VALUE, NfaChar('"'), "BEGIN(VALUE_Q_NE);");
	lexgen.addRule(Defs::VALUE, heredoc_begin, newline,
		"BEGIN(VALUE_HEREDOC);\n\t"
		"obj->saveHeredocIdentifier(tok_begin+2, tok_end);\n\t"
		"obj->newLine(match_end);\n\t"
		"CONSUME_TRAILER();"
	);
	lexgen.addRule(Defs::VALUE, epsilon, "BEGIN(VALUE_NQ);");
	
	
	lexgen.addRule(Defs::VALUE_NQ, NfaConcatenation(NfaClosure(any), CRLF),
		"BEGIN(AFTER_SECTION_OR_VALUE); LESS(2);\n\t"
		"obj->valueAppend(tok_begin, tok_end);\n\t"
		"obj->processValue();"
	).setLazy();
	lexgen.addRule(Defs::VALUE_NQ, NfaConcatenation(NfaClosure(any), LF),
		"BEGIN(AFTER_SECTION_OR_VALUE); LESS(1);\n\t"
		"obj->valueAppend(tok_begin, tok_end);\n\t"
		"obj->processValue();"
	).setLazy();
	lexgen.addRule(Defs::VALUE_NQ, NfaClosure(any),
		"BEGIN(INITIAL);\r\n"
		"obj->valueAppend(tok_begin, tok_end); obj->processValue();"
	);
	
	
	lexgen.addRule(Defs::VALUE_Q_NE, NfaPositiveClosure(NfaNegatedCharClass("\"\n\\")),
		"obj->valueAppend(tok_begin, tok_end);"
	);
	lexgen.addRule(Defs::VALUE_Q_NE, NfaChar('\\'), "BEGIN(VALUE_Q_E);");
	lexgen.addRule(Defs::VALUE_Q_NE, NfaChar('"'),
		"BEGIN(AFTER_SECTION_OR_VALUE); obj->processValue();"
	);
	lexgen.addRule(Defs::VALUE_Q_NE, any,
		"BEGIN(RECOVERY); CONSUME_NOTHING(); obj->processUnclosedQuote();"
	);
	
	
	lexgen.addRule(Defs::VALUE_Q_E, any, "BEGIN(VALUE_Q_NE); obj->valueAppendEscaped(*tok_begin);");
	
	
	lexgen.addRule(Defs::VALUE_HEREDOC, NfaConcatenation(NfaClosure(any), CRLF),
		"LESS(2);\n\t"
		"if (obj->isHeredocIdentifier(match_begin, tok_end))\n\t"
		"{ BEGIN(AFTER_SECTION_OR_VALUE); obj->processValue(); }\n\t"
		"else { BEGIN(VALUE_HEREDOC_NEWLINE); obj->valueAppend(tok_begin, tok_end); }"
	).setLazy();
	lexgen.addRule(Defs::VALUE_HEREDOC, NfaConcatenation(NfaClosure(any), LF),
		"LESS(1);\n\t"
		"if (obj->isHeredocIdentifier(match_begin, tok_end))\n\t"
		"{ BEGIN(AFTER_SECTION_OR_VALUE); obj->processValue(); }\n\t"
		"else { BEGIN(VALUE_HEREDOC_NEWLINE); obj->valueAppend(tok_begin, tok_end); }"
	).setLazy();
	lexgen.addRule(Defs::VALUE_HEREDOC, NfaClosure(any), /* EOF instead of newline */
		"if (obj->isHeredocIdentifier(match_begin, tok_end))\n\t"
		"{ BEGIN(AFTER_SECTION_OR_VALUE); obj->processValue(); }\n\t"
		"else { obj->valueAppend(tok_begin, tok_end); }"
	);
	
	
	lexgen.addRule(Defs::VALUE_HEREDOC_NEWLINE, newline,
		"BEGIN(VALUE_HEREDOC); MORE();\n\t"
		"obj->newLine(tok_end);"
	);
	
	
	lexgen.addRule(Defs::AFTER_SECTION_OR_VALUE, newline, "BEGIN(INITIAL); obj->newLine(tok_end);");
	lexgen.addRule(Defs::AFTER_SECTION_OR_VALUE, NfaPositiveClosure(space), "/* skip spaces */");
	lexgen.addRule(Defs::AFTER_SECTION_OR_VALUE, any,
		"BEGIN(RECOVERY); CONSUME_NOTHING(); obj->processUnexpectedCharacter(tok_begin);"
	);
	
	
	lexgen.addRule(Defs::RECOVERY, NfaConcatenation(NfaClosure(any), newline),
		"BEGIN(INITIAL); obj->newLine(tok_end);"
	).setLazy();
	
	
	lexgen.writeLexer(header_strm, impl_strm, def_class, def_header, subclass_header);
	return 0;
}
