/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ConfIO.h"
#include "ConfError.h"
#include "ConfLexerDefinitions.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include "StringUtils.h"
#include <memory>
#include <istream>
#include <ostream>
#include <sstream>
#include <cstring>
#include <cstdlib>
#include <cassert>
#include <stddef.h>

using namespace std;

ConfIO::ConfIO()
:	ConfLexerCore(this),
	m_commentSymbol(';'),
	m_lineNumber(1),
	m_lineStartPos(0)
{
}

ConfIO::~ConfIO()
{
}

bool
ConfIO::read(std::istream& strm)
{
	SplittableBuffer data;
	auto_ptr<DataChunk> chunk;
	while (!isAborted()) {
		chunk = DataChunk::create(4000);
		strm.read(chunk->getDataAddr(), chunk->getDataSize());
		if (strm.gcount() == 0) {
			break;
		}
		data.append(DataChunk::resize(chunk, strm.gcount()));
		consume(data, false);
	}
	if (!isAborted()) {
		consume(data, true);
		switch (scond()) {
			case ConfLexerDefinitions::INITIAL:
			case ConfLexerDefinitions::COMMENT:
			case ConfLexerDefinitions::AFTER_SECTION_OR_VALUE:
			case ConfLexerDefinitions::RECOVERY:
			break;
			default:
			processError(ConfError("Unexpected end of file"));
		}
	}
	bool is_aborted = isAborted();
	reset();
	m_lineNumber = 1;
	m_lineStartPos = 0;
	return !is_aborted;
}

bool
ConfIO::read(std::string const& text)
{
	istringstream strm(text);
	return read(strm);
}

void
ConfIO::write(std::ostream& strm,
	std::list<Element> const& elements, int flags)
{
	list<Element>::const_iterator it = elements.begin();
	list<Element>::const_iterator const end = elements.end();
	for (; it != end; ++it) {
		it->toStream(strm, flags);
		strm << '\n';
	}
}

ConfIO::ElementIterator
ConfIO::findNextSection(
	ElementIterator const& begin, ElementIterator const& end)
{
	ElementIterator it = begin;
	for (; it != end; ++it) {
		if (it->getType() == Element::SECTION) {
			break;
		}
	}
	return it;
}

ConfIO::ElementIterator
ConfIO::getSectionItemsEnd(
	ElementIterator const& section_element,
	ElementIterator const& end)
{
	assert(section_element->getType() == Element::SECTION);
	
	ElementIterator item = section_element;
	ElementIterator it = section_element;
	while (++it != end) {
		if (it->getType() == Element::KEY_VALUE) {
			item = it;
		} else if (it->getType() == Element::SECTION) {
			break;
		}
	}
	return ++item;
}

/*
Distributes comments (and blank lines) between the two items following
each other. An item can be either a key=value pair or a [section] element.
The comments are contained in the [begin, end) interval.
Example:
[section1]
key=val
#comment1  <-- begin
           <-- return value; forms two intervals: [begin, rval) and [rval, end)
#comment2
[section2] <-- end
*/
ConfIO::ElementIterator
ConfIO::distributeComments(
	ElementIterator const& begin, ElementIterator const& end)
{
	struct Gap
	{
		ElementIterator begin;
		ElementIterator end;
		int size;
		
		Gap(ElementIterator const& pos)
		: begin(pos), end(pos), size(0) {}
		
		void reset(ElementIterator const& pos) {
			begin = pos;
			end = pos;
			size = 0;
		}
	};
	
	Gap cur_gap(end);
	Gap max_gap(end);
	ElementIterator it = begin;
	while (it != end) {
		for (; it != end && it->getType() != Element::BLANK_LINE; ++it) {
			// skip anything that's not BLANK_LINE
		}
		
		cur_gap.reset(it);
		for (; it != end && it->getType() == Element::BLANK_LINE; ++it) {
			++cur_gap.end;
			++cur_gap.size;
		}
		if (cur_gap.size > max_gap.size) {
			max_gap = cur_gap;
		}
	}
	
	if (max_gap.size > 0) {
		return max_gap.begin;
	} else {
		/*
		In a situation like this:
		key=val
		#comment1
		#comment2
		[section]
		
		We consider both comments to belong to [section].
		*/
		return begin;
	}
}

ConfIO::ElementIterator
ConfIO::getParamAreaBegin(
	ElementIterator const& begin, ElementIterator const& param)
{
	list<Element>::reverse_iterator prev(param);
	list<Element>::reverse_iterator const rend(begin);
	for (; prev != rend; ++prev) {
		if (prev->getType() == Element::SECTION ||
		    prev->getType() == Element::KEY_VALUE) {
			break;
		}
	}
	
	return distributeComments(prev.base(), param); 
}

ConfIO::ElementIterator
ConfIO::getParamAreaEnd(
	ElementIterator const& param, ElementIterator const& end)
{
	ElementIterator after_param(param);
	++after_param;
	ElementIterator next(after_param);
	for (; next != end; ++next) {
		if (next->getType() == Element::SECTION ||
		    next->getType() == Element::KEY_VALUE) {
			break;
		}
	}
	
	return distributeComments(after_param, next);
}

void
ConfIO::newLine(Iterator const& pos)
{
	++m_lineNumber;
	m_lineStartPos = tokenPosition(pos);
}

void
ConfIO::processBlankLine()
{
	if (!processElement(Element(), m_lineNumber)) {
		abort();
	}
}

void
ConfIO::processBrokenSection()
{
	ConfError err("Invalid section declaration", m_lineNumber);
	processError(err);
	abort();
}

void
ConfIO::processUnexpectedCharacter(Iterator const& pos)
{
	ConfError err(
		"Unexpected character", m_lineNumber,
		tokenPosition(pos) - m_lineStartPos + 1
	);
	if (!processError(err)) {
		abort();
	}
}

void
ConfIO::processComment(Iterator const& begin, Iterator const& end)
{
	Element el(m_commentSymbol, tokenAsString(begin, end));
	if (!processElement(el, m_lineNumber)) {
		abort();
	}
}

void
ConfIO::processSection(Iterator const& begin, Iterator const& end)
{
	Element el(tokenAsString(begin, end));
	if (!processElement(el, m_lineNumber)) {
		abort();
	}
}

void
ConfIO::processKey(Iterator const& begin, Iterator const& end)
{
	m_key = tokenAsString(begin, end);
	m_value.resize(0);
}

void
ConfIO::processValue()
{
	Element el(m_key, m_value);
	if (!processElement(el, m_lineNumber)) {
		abort();
	}
}

void
ConfIO::processUnclosedQuote()
{
	ConfError err("Unclosed quote", m_lineNumber);
	if (!processError(err)) {
		abort();
	}
}

void
ConfIO::saveHeredocIdentifier(Iterator const& begin, Iterator const& end)
{
	m_heredocIdentifier = SplittableBuffer::toBString(begin, end);
}

bool
ConfIO::isHeredocIdentifier(Iterator const& begin, Iterator const& end) const
{
	return StringUtils::equal(
		begin, end,
		m_heredocIdentifier.begin(),
		m_heredocIdentifier.end()
	);
}

bool
ConfIO::isAtLineStart(Iterator const& pos)
{
	return tokenPosition(pos) == m_lineStartPos;
}

void
ConfIO::valueAppend(Iterator const& begin, Iterator const& end)
{
	m_value += tokenAsString(begin, end);
}

void
ConfIO::valueAppendEscaped(char ch)
{
	m_value += unescapeChar(ch);
}

void
ConfIO::writeValue(std::ostream& strm, std::string const& value, int flags)
{
	bool need_quote = false;
	bool need_heredoc = value.length() > 60;
	char const* pos = value.c_str();
	char const* const end = pos + value.length();
	for (; pos != end && !need_heredoc; ++pos) {
		unsigned char ch = *pos;
		if (ch == ' ' || ch == '\n' || ch == '\r' || ch == '"') {
			need_quote = true;
		}
		if (ch < 32) {
			need_heredoc = true;
		}
	}
	
	if (need_heredoc && !(flags & DISABLE_HEREDOC)) {
		writeHeredocValue(strm, value);
	} else if (need_quote || need_heredoc) {
		writeQuotedValue(strm, value);
	} else {
		strm << value;
	}
}

void
ConfIO::writeHeredocValue(std::ostream& strm, std::string const& value)
{
	string id = generateHeredocIdentifier(value);
	strm << "<<" << id << '\n' << value << '\n' << id;
}

void
ConfIO::writeQuotedValue(std::ostream& strm, std::string const& value)
{
	strm << '\"';
	char const* pos = value.c_str();
	char const* const end = pos + value.length();
	for (; pos != end; ++pos) {
		char ch = *pos;
		char escaped = escapeChar(ch);
		if (ch == '"' || ch == '\\' || escaped != ch) {
			strm << '\\' << escaped;
		} else {
			strm << ch;
		}
	}
	strm << '\"';
}

std::string
ConfIO::generateHeredocIdentifier(std::string const& value)
{
	std::string id("END");
	char const* pos = value.c_str();
	char const* const end = pos + value.length();
	while (true) {
		while (StringUtils::startsWith(pos, end, id)) {
			mutateHeredocIdentifier(id);
		}
		pos = StringUtils::find(pos, end, '\n');
		if (pos != end) {
			++pos;
		} else {
			break;
		}
	}
	return id;
}

void
ConfIO::mutateHeredocIdentifier(std::string& id)
{
	static char const digits[] = {'0','1','2','3','4','5','6','7','8','9'};
	id += digits[rand() % 10];
}

char
ConfIO::escapeChar(char ch)
{
	static char const in[]  = {'\a','\b','\f','\n','\r','\t','\v'};
	static char const out[] = {'a', 'b', 'f', 'n', 'r', 't', 'v'}; 
	char const* found = static_cast<char const*>(memchr(in, ch, sizeof(in)));
	if (found) {
		return out[found - in];
	}
	return ch;
}

char
ConfIO::unescapeChar(char ch)
{
	static char const in[]  = {'a', 'b', 'f', 'n', 'r', 't', 'v'}; 
	static char const out[] = {'\a','\b','\f','\n','\r','\t','\v'};
	char const* found = static_cast<char const*>(memchr(in, ch, sizeof(in)));
	if (found) {
		return out[found - in];
	}
	return ch;
}


/*========================== ConfIO::Element =========================*/

ConfIO::Element::Element()
:	m_type(BLANK_LINE)
{
}

ConfIO::Element::Element(std::string const& section)
:	m_type(SECTION),
	m_value(section)
{
}

ConfIO::Element::Element(char comment_symbol, std::string const& comment)
:	m_type(COMMENT),
	m_commentSymbol(comment_symbol),
	m_value(comment)
{
}

ConfIO::Element::Element(std::string const& key, std::string const& value)
:	m_type(KEY_VALUE),
	m_key(key),
	m_value(value)
{
}

ConfIO::Element::~Element()
{
}

void
ConfIO::Element::toStream(std::ostream& strm, int flags) const
{
	switch (m_type) {
		case BLANK_LINE: {
			break;
		}
		case Element::COMMENT: {
			strm << m_commentSymbol << m_value;
			break;
		}
		case Element::SECTION: {
			strm << '[' << m_value << ']';
			break;
		}
		case Element::KEY_VALUE: {
			strm << m_key << '=';
			writeValue(strm, m_value, flags);
			break;
		}
	}
}

bool
ConfIO::Element::operator==(Element const& other) const
{
	return (m_type == other.m_type && m_commentSymbol == other.m_commentSymbol
		&& m_key == other.m_key && m_value == other.m_value);
}
