/* help.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "commands/cmds.h"
#include "commands/help.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_aliases, 0, "aliases", 0, \
      N_("Display command aliases")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit."))

t_uchar arch_cmd_help_help[] = N_("provide help with arch\n"
                                "\n"
                                "This command prints a list of the available commands.\n"
                                "\n"
                                "To see just a list of the options to a particular command,\n"
                                "use:\n"
                                "\n"
                                "    baz $cmd -h\n"
                                "\n"
                                "(where $cmd is the name of the command).   For additional\n"
                                "explanation about a given command, use:\n"
                                "\n"
                                "    baz $cmd -H\n"
                                "\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_help (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int aliases = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_help_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
	case opt_aliases:
	  aliases = 1;
	  break;
	  
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;
        }
    }

  if (argc != 1)
    goto usage_error;

  {
    int x;
    int max_length = 0;

    for (x = 0; arch_commands[x].name; ++x)
      if (!arch_commands[x].deprecated
          && arch_commands[x].fn)
        {
          int len;

          len = str_length (arch_commands[x].name);
          if (len > max_length)
            max_length = len;
        }

    max_length += 2;

    safe_printfmt (1, _("                        baz sub-commands\n"));
    safe_printfmt (1,   "                        ----------------");

    for (x = 0; arch_commands[x].name; ++x)
      if (! arch_commands[x].deprecated)
        {
          if (!arch_commands[x].fn)
            {
              if (!arch_commands[x].name[0])
                safe_printfmt (1, "\n");
              else
                safe_printfmt (1, "\n\n* %s\n\n", (t_uchar *)gettext(arch_commands[x].name));
            }
          else if (arch_commands[x].alias_of)
            {
              int name_len;
              t_uchar * help_nl;
              t_uchar * long_help;

	      if (!aliases)
		  continue;

              name_len = (int)str_length (arch_commands[x].name);
              long_help = (t_uchar *)gettext(arch_commands[x].long_help);
              help_nl = str_chr_index (long_help, '\n');
              if (!help_nl)
                help_nl = arch_commands[x].long_help + str_length (arch_commands[x].long_help);
              safe_printfmt (1, _("%*s%s : (alias for %s)\n"), (max_length - name_len), "", arch_commands[x].name, arch_commands[x].alias_of);
            }
          else
            {
              int name_len;
              t_uchar * help_nl;
              t_uchar * long_help;

              name_len = (int)str_length (arch_commands[x].name);
              long_help = (t_uchar *)gettext(arch_commands[x].long_help);
              help_nl = str_chr_index (long_help, '\n');
              if (!help_nl)
                help_nl = arch_commands[x].long_help + str_length (arch_commands[x].long_help);
              safe_printfmt (1, _("%*s%s : %.*s\n"), (max_length - name_len), "", arch_commands[x].name, (int)(help_nl - arch_commands[x].long_help), arch_commands[x].long_help);
            }
        }
    
    safe_printfmt (1, "\n");
  }

  safe_printfmt (1, _("\nUse baz command -h for help on `command', or baz command -H for detailed help.\n"));

  return 0;
}




/* tag: Tom Lord Wed Jun 11 21:04:35 2003 (cmd-help.c)
 */
