# Copyright (C) 1999, 2000 Jay Beale
# Copyright (C) 2001-2003 Hewlett Packard Corporation
# Licensed under the GNU General Public License

package Bastille::API;

## TO DO:
#
#
#   1) Look for more places to insert error handling... 
#
#   2) Document this module more
#
#

## TO DO LATER:
#
#   - Use peterw's suggestion about storing diffs of original and Bastille'd
#     files...



##########################################################################
#
# This module forms the basis for the v1.1 API.
# 
 ##########################################################################

# 
# This module forms the initial basis for the Bastille Engine, implemented
# presently via a Perl API for Perl modules.
#
# This is still under construction -- it is very useable, but not very well
# documented, yet.
#

##########################################################################
#
#                          API Function Listing                               
#
##########################################################################
# The routines which should be called by Bastille modules are listed here,
# though they are better documented throughout this file.
#
# Distro Specific Stuff:
#
#  &GetDistro     - figures out what distro we're running, if it knows it...
#  &ConfigureForDistro - sets global variables based on the distro
#  &GetGlobal - returns hash values defined in ConfigureForDistro
#
#
# TUI Integration Stuff:
#
#  &getGlobalConfig - returns value of hash set up by ReadConfig
#
# Logging Specific Stuff:
#
#  &ActionLog    -- logs a message to the bastille-action-log file
#  &ErrorLog     -- logs a message to the bastille-error-log file
#  &DebugLog     -- logs a message to the bastille-debug-log file
#
#
# Input functions for the old input method...
#
#  &GetYN      * -- accepts a Y/N answer from console, ignoring hashed comments
#  &GetString  * -- accepts a line from console, ignoring commented lines
#
# File open/close/backup functions
#
#  &B_open     * -- opens a file handle and logs the action/error (OLD WAY!)
#  &B_open_plus  -- opens a pair of file handles for the old and new version
#                   of a file; respects logonly flag.  (NEW WAY)
#  &B_close    * -- closes a file handle and logs the action/error (OLD WAY!)
#  &B_close_plus -- closes a pair of file handles opened by B_open_plus,
#                   backing up one file and renaming the new file to the
#                   old one's name, logging actions/errors.  Respects the 
#                   logonly flag -- needs B_backup file.  Finally, sets
#                   new file's mode,uid,gid to old file's...  (NEW WAY)
#  &B_backup_file - backs up a file that is being changed/deleted into the
#                   $GLOBAL_BDIR{"backup"} directory.
#
# Content (Line-based) file modification functions
#
#  &B_blank_file  - blanks a file if it isn't ours, so that we can rewrite it
#  &B_append_line - opens a file, appends a line to it, and closes file
#  &B_insert_line - opens a file, inserts a line after a specified line, closes
#  &B_prepend_line- opens a file, prepends a line to it, and closes file
#  &B_replace_line- replaces one or more lines in a file using regep matching
#  &B_hash_comment_line
#                 - hash - comments one or more lines in a file
#  &B_hash_uncomment_line
#                 - hash - uncomments one or more lines in a file
#  &B_delete_line - deletes one or more lines in a file using regep matching
#
# Content (multiple-line-based) file mod functions (for PeterW's ipchains.pm)
#
#  &B_chunk_append
#
#  &B_chunk_replace
#
#
# Non-content file modification functions
#
#  &B_delete_file - deletes the named file, backing up a copy
#  &B_create_file - creates the named file, if it doesn't exist
#
#  &B_print       	- print a line to the named filehandle, logging the action
#  &B_chmod       	- chmod a file, logging the action
#  &B_chmod_if_exists  	- chmod a file, logging the action only if the file exists
#  &B_chown       	- change only the owner on a file, logging the action
#  &B_chgrp       	- change the group owner on a file, logging the action
#  &B_remove_suid       - remove suid bit from a given file.
#  &B_symlink     	- create a symlink to a file, recording the revert rm
#
# More stuff
#
#  &B_chkconfig_off - removed all S links from rcX.d directory
#  &B_chkconfig_on  - make all S links from rcX.d directory
#  &B_createdir     - make a directory, if it doesn't exist, record revert rmdir
#  &B_cp            - copy a file, respecting LOGONLY and revert func.
#  &B_mknod         - wrap mknod with revert and logonly and prefix functionality
#
#  &B_read_sums     - reads sum.csv file and parses input into the GLOBAL_SUM hash
#  &B_write_sums    - writes sum.csv file from GLOBAL_SUM hash
#  &B_check_sum($)  - take a file name and compares the stored cksum with the current
#                     cksum of said file
#  &B_set_sum($)    - takes a file name and gets that files current cksum then sets
#                     that sum in the GLOBAL_SUM hash
#  &B_revert_log - create entry in shell script, executed later by bastille -r
#  &showDisclaimer  - Print the disclaimer and wait for 5 minutes for acceptance
###########################################################################
# Note:  GLOBAL_VERBOSE
# 
# All logging functions now check GLOBAL_VERBOSE and, if set, will print
# all the information sent to log files to STDOUT/STDERR as well.
#

#
# Note:  GLOBAL_LOGONLY
# 
# All Bastille API functions now check for the existence of a GLOBAL_LOGONLY 
# variable.  When said variable is set, no function actually modifies the 
# system.
#
# Note:  GLOBAL_DEBUG
#
# The new DebugLog function now checks GLOBAL_DEBUG and, if set, it will
# print all the information to a new debug-log file. If GLOBAL_VERBOSE is
# set it might log to STDOUT/STDERR as well (not yet implemented, pending
# discussion). Developers should populate appropiate places with &DebugLog
# in order to be able to tell users to use this options and send the logs
# for inspection and debugging.
#
#


# Libraries for the Backup_file routine: Cwd and File::Path
use Cwd;
use Bastille::HP_API;
use Bastille::OSX_API;    
use File::Path;
use File::Basename;

# Export the API functions listed below for use by the modules.

use Exporter;
@ISA = qw ( Exporter );
@EXPORT = qw( setOptions PrepareToRun GetDistro ConfigureForDistro StartLogging ActionLog ErrorLog DebugLog
	      GetYN GetString SanitizeEnv
              B_open B_close B_print B_symlink StopLogging 
	      B_Display B_open_plus B_close_plus B_blank_file B_append_line
	      B_insert_line
	      B_prepend_line B_replace_line B_replace_pattern
	      B_chunk_replace B_chunk_append
	      B_hash_comment_line B_hash_uncomment_line 
	      B_delete_line
	      B_create_file B_read_sums B_check_sum  B_set_sum
	      B_create_dir B_remove_suid
	      B_delete_file B_chmod B_chmod_if_exists B_chown B_chgrp 
	      B_chkconfig_off B_chkconfig_on B_cp B_place B_mknod
	      B_ch_rc B_set_value B_System B_Schedule B_TODO B_install_jail 
              B_chperm showDisclaimer 
              B_load_ipf_rules B_deactivate_inetd_service get_supported_OS_list
	      B_set_rc B_get_rc
              $GLOBAL_OS
	       $GLOBAL_LOGONLY $GLOBAL_VERBOSE $GLOBAL_DEBUG $errorFlag
	      %GLOBAL_BIN %GLOBAL_DIR %GLOBAL_FILE
	      %GLOBAL_BDIR %GLOBAL_BFILE
	      %GLOBAL_CONFIG %GLOBAL_SUM
	      %GLOBAL_SERVICE %GLOBAL_SERVTYPE %GLOBAL_PROCESS

              getGlobal getGlobalConfig getGlobalSwlist

	      );

&SanitizeEnv;

# Set up some common error messages.  These are independent of 
# operating system

# These will allow us to line up the warnings and error messages
my $err ="ERROR:  ";
my $spc ="        ";

#OS independent Error Messages Follow
$GLOBAL_ERROR{"usage"}="$err Invalid argument list:\n".
    "$spc Usage: bastille [ -b | -c | -r | -x [ --os version ] ]\n".
    "$spc -b : use a saved config file to apply changes\n".
    "$spc      directly to system\n".
    "$spc -c : use the Curses (non-X11) TUI\n".
    "$spc -r : revert all Bastille changes to-date\n".
    "$spc -x : use the Perl/Tk (X11) GUI\n" .
    "$spc --os version : ask all questions for the given operating system\n" . 
    "$spc                version.  e.g. --os RH6.0\n";

# These options don't work universally, so it's best not to
# document them here (yet).  Hopefully, we'll get them
# straightened out soon.
#"$spc --log : log-only option\n".
#"$spc -v : verbose mode\n".
#"$spc --debug : debug mode\n";


##############################################################################
#
#                         Bastille v1.1 directory structure
#
##############################################################################
#
#  /root/Bastille  -- parent directory for Bastille Linux
#
#  /root/Bastille/revert -- directory holding all revert scripts and backups...
#  /root/Bastille/revert/backup -- all the original files that Bastille modifies,
#                                with permissions intact
#
#  /root/Bastille/logs -- directory holding all logs
#
##############################################################################

##############################################################################
#
#                         Bastille v1.2 directory structure
#
##############################################################################
#
#  /usr/sbin/          -- location of Bastille binaries
#  /usr/lib/Bastille   -- location of Bastille modules
#  /usr/share/Bastille -- location of Bastille data files
#  /etc/Bastille       -- location of Bastille config files
#
#  /var/log/Bastille      -- location of Bastille log files
#  /var/log/Bastille/revert -- directory holding all Bastille-created revert scripts
#  /var/log/Bastille/revert/backup -- directory holding the original files that
#                                   Bastille modifies, with permissions intact
#
##############################################################################






##############################################################################
##############################################################################
##################  Actual functions start here... ###########################
##############################################################################
##############################################################################

###########################################################################
# setOptions takes three arguments, $GLOBAL_DEBUG, $GLOBAL_LOGONLY, and
# $GLOBAL_VERBOSE.
###########################################################################
sub setOptions($$$) {
    ($GLOBAL_DEBUG,$GLOBAL_LOGONLY,$GLOBAL_VERBOSE) = @_;
}
###########################################################################
# 
# SanitizeEnv load a proper environment so Bastille cannot be tricked
# and Perl modules work correctly.
# 
###########################################################################
sub SanitizeEnv {
	 delete @ENV{'IFS','CDPATH','ENV','BASH_ENV'};
	 $ENV{CDPATH}=".";
	 $ENV{BASH_ENV}= "";
	 # Bin is needed here or else  /usr/lib/perl5/5.005/Cwd.pm 
	 # will not find `pwd`
	 # Detected while testing with -w, jfs
	 $ENV{PATH} = "/bin:/usr/bin";
	 # Giorgi, is /usr/local/bin needed? (jfs)
}

###########################################################################
# 
# PrepareToRun sets up Bastille to run.  It checks the ARGV array for
# special options and runs ConfigureForDistro to set necessary file
# locations and other global variables.
#
###########################################################################

sub PrepareToRun {
    
    # Make sure we're root!
    if ( $> != 0 ) {
	&ErrorLog("ERROR:   Bastille BackEnd must run as root!\n");
        exit(1);
    }

   
    # Make any directories that don't exist...
    foreach my $dir ( &getGlobal('BDIR', "revert") , &getGlobal('BDIR', "backup"), &getGlobal('BDIR', "log"),&getGlobal('BDIR', "config") ) {
	mkpath ($dir,0,0700);
    }

    if(&GetDistro =~ "^HP-UX") {
	&B_check_system;
    }

    &ActionLog("\n########################################################\n" .
	       "#  Begin Bastille Run                                  #\n" .
	       "########################################################\n\n");

    #read sum file if it exists.
    &B_read_sums;

    # check the integrety of the files listed
    for my $file (sort keys %GLOBAL_SUM) {
	&B_check_sum($file);
    }
    # write out the newly flagged sums
    &B_write_sums;


}

###########################################################################
#
# GetDistro checks to see if the target is a known distribution and reports
# said distribution.
#
# This is used throughout the script, but also by ConfigureForDistro.
# 
#
###########################################################################

sub GetDistro() {

    my ($release,$distro);

    # Only read files for the distro once.
    # if the --os option was used then 
    if (defined $GLOBAL_OS) {
	$distro = $GLOBAL_OS;
    } else {

	if ( -e "/etc/mandrake-release" ) {
	    open(MANDRAKE_RELEASE,"/etc/mandrake-release");
	    $release=<MANDRAKE_RELEASE>;
	    unless ( ($release =~ /^Mandrake Linux release (\d+\.\d+\w*)/) or ($release =~ /^Linux Mandrake release (\d+\.\d+\w*)/) ) {
		print STDERR "$err Couldn't determine Mandrake version! Setting to 6.0!\n";
		$distro="MN6.0";
	    }
	else {
	    $distro="MN$1";
	}
	    close(MANDRAKE_RELEASE);
	}
	elsif ( -e "/etc/immunix-release" ) {
	    open(IMMUNIX_RELEASE,"/etc/immunix-release");
	    $release=<IMMUNIX_RELEASE>;
	    unless ($release =~ /^Immunix Linux release (\d+\.\d+\w*)/) {
		print STDERR "$err Couldn't determine Immunix version! Setting to 6.2!\n";
		$distro="RH6.2";
	    }
	    else {
		$distro="RH$1";
	    }
	    close(*IMMUNIX_RELEASE);
	}
	elsif ( -e "/etc/redhat-release" ) {
	    open(*REDHAT_RELEASE,"/etc/redhat-release");
	    $release=<REDHAT_RELEASE>;
	    unless ($release =~ /^Red Hat Linux release (\d+\.?\d*\w*)/) {
		print STDERR "$err Couldn't determine Red Hat version! Setting to 8.0!\n";
		$distro="RH8.0";
	    }
	    else {
		$distro="RH$1";
	    }      
	    close(REDHAT_RELEASE);   
	}
	elsif ( -e "/etc/debian_version" ) {
	    $stable="3.0"; #Change this when Debian stable changes
	    open(*DEBIAN_RELEASE,"/etc/debian_version");
	    $release=<DEBIAN_RELEASE>;
	    unless ($release =~ /^(\d+\.\d+\w*)/) {
		print STDERR "$err System is not running a stable Debian GNU/Linux version. Setting to $stable.\n";
		$distro="DB$stable";
	    }
	    else {
		$distro="DB$1";
	    }      
	    close(DEBIAN_RELEASE);
	}
	elsif ( -e "/etc/SuSE-release" ) {
	    open(*SUSE_RELEASE,"/etc/SuSE-release");
	    $release=<SUSE_RELEASE>;
	    unless ($release =~ /^SuSE Linux (\d+\.\d+\w*)/) {
		print STDERR "$err Couldn't determine SuSE version! Setting to 7.2!\n";
		$distro="SE7.2";
	    } 
	    else {      
		$distro="SE$1";
	    }
	    close(SUSE_RELEASE);
	}   
	elsif ( -e "/etc/turbolinux-release") {
	    open(*TURBOLINUX_RELEASE,"/etc/turbolinux-release");
	    $release=<TURBOLINUX_RELEASE>;
	    unless ($release =~ /^Turbolinux Workstation (\d+\.\d+\w*)/) {
		print STDERR "$err Couldn't determine TurboLinux version! Setting to 7.0!\n";
		$distro="TB7.0";
	    }
	    else {
		$distro="TB$1";
	    }
	    close(TURBOLINUX_RELEASE);
	}
	else {
	    # We're either on Mac OS X, HP-UX or an unsupported O/S.
            if ( -x '/usr/bin/uname') {
		# uname is in /usr/bin on Mac OS X
		$release=`/usr/bin/uname -sr`;
	    }
	    elsif ( -x '/bin/uname') {
		# uname is in /bin on HP-UX
		$release=`/bin/uname -sr`;
	    }
	    else {
	 	print STDERR "$err Could not determine operating system version!\n";
		$distro="unknown";
            }

	    # Figure out what kind of system we're on.
	    if ($release) {
		if ($release =~ /^Darwin\s+(\d+)\.(\d+)/) {
		    if ($1 == 6 ) {
			$distro = "OSX10.2.$2";
		    }
		    else {
		        $distro = "unknown";
		    }
		}
	        elsif ( $release =~ /(^HP-UX)\s*B\.(\d+\.\d+)/ ) {
		   $distro="$1$2";
		}
		else {
		   print STDERR "$err Could not determine operating system version!\n";
	           $distro="unknown";
		}
	    }
	}
	
	$GLOBAL_OS=$distro;
    }
    
    return $distro;
}

###################################################################################
#   &getActualDistro;                                                             #
#                                                                                 #
#    This subroutine returns the actual os version in which is running on.  This  #
#    os version is indepentant of the --os switch feed to bastille.               #
#                                                                                 #
###################################################################################
sub getActualDistro {
    # set local variable to $GLOBAL_OS
    my $os = $GLOBAL_OS;
    # undef GLOBAL_OS so that the GetDistro routine will return
    # the actualDistro, it might otherwise return the distro set
    # by the --os switch.
    $GLOBAL_OS = undef;
    my $actualDistro = &GetDistro;
    # reset the GLOBAL_OS variable
    $GLOBAL_OS = $os;
    
    return $actualDistro;

}
# These are helper routines which used to be included inside GetDistro
sub is_OS_supported ($) {
   my $os=$_[0];

   my $supported=0;
   foreach my $supported_os ( @{&get_supported_OS_list} ) {
      if ( $supported_os eq $os ) {
        $supported=1;
      }
   }

   return $supported;
}

sub get_supported_OS_list () {
  my @list = ( "DB2.2", "DB3.0", 
           "RH6.0","RH6.1","RH6.2","RH7.0","RH7.1","RH7.2","RH7.3","RH8.0","RH9",
           "MN6.0","MN6.1","MN7.0","MN7.1","MN7.2","MN8.0","MN8.1","MN8.2",
           "HP-UX11.00","HP-UX11.11", "HP-UX11.22", "HP-UX11.23",
           "SE7.2","SE7.3", "SE8.0","TB7.0", 
	   "OSX10.2.0","OSX10.2.1","OSX10.2.2","OSX10.2.3","OSX10.2.4");
  return \@list;
}


###########################################################################
#
# &ConfigureForDistro configures the API for a given distribution.  This
# includes setting global variables that tell the Bastille API about
# given binaries and directories.
#
# WARNING: If a distro is not covered here, Bastille may not be 100% 
#          compatible with it, though 1.1 is written to be much smarter
#          about unknown distros...
#
###########################################################################

sub ConfigureForDistro {

    my $retval=1;

    # checking to see if the os version given is in fact supported
    my $distro = &GetDistro;

    # checking to see if the actual os version is in fact supported
    my $actualDistro = &getActualDistro;
    if ((! &is_OS_supported($distro)) or (! &is_OS_supported($actualDistro))  ) {
	# if either is not supported then print out a list of supported versions
	if (! &is_OS_supported($distro)) {
	    print STDERR "$err '$distro' is not a supported operating system.\n";
	}
	else {
	    print STDERR "$err Bastille is unable to operate correctly on this\n";
	    print STDERR "$spc operating system.\n";
	}
	my @supported_OS = @{&get_supported_OS_list};
	print STDERR "$spc Valid operating system versions are as follows:\n";
	my $os_number = 1;
	print STDERR "$spc ";
	foreach my $os (@supported_OS) {
	    print STDERR "'$os' ";
	    if ($os_number == 5){
		print STDERR "\n$spc ";
		$os_number = 1;
	    }
	    else {
		$os_number++;
	    }
	}

	print "\n" . $GLOBAL_ERROR{"usage"};
	exit(1);
    }


    # Directories, as explained in Bastille directory structure...

    if ( ($actualDistro =~ "^RH" ) or ($actualDistro =~ "^MN") or ($actualDistro =~ "^DB") or ($actualDistro =~ "^SE") or ($actualDistro =~ "^TB")){
	
	$GLOBAL_BDIR{"home"}= "/root/Bastille";

	$GLOBAL_BDIR{"log"} = "/var/log/Bastille";
	$GLOBAL_BFILE{"action-log"} = &getGlobal('BDIR', "log") . "/action-log";
        # This one is referenced explicitly in ErrorLog to avoid infinite
        # recursion (out of memory error), but also defined here so the
        # file location can be referenced
	$GLOBAL_BFILE{"error-log"} = &getGlobal('BDIR', "log") . "/error-log";
	$GLOBAL_BFILE{"debug-log"} = &getGlobal('BDIR', "log") . "/debug-log";

	$GLOBAL_BDIR{"share"}= "/usr/share/Bastille";
	$GLOBAL_BDIR{"revert"}= &getGlobal('BDIR', "log") . "revert";
	$GLOBAL_BFILE{"TOREVERT"}= $GLOBAL_BDIR{"revert"} . "/TOREVERT.txt";
	$GLOBAL_BDIR{"oldconfig"}= &getGlobal('BDIR', "log") . "oldconfig";
	$GLOBAL_BFILE{"created-files"}= &getGlobal('BDIR', "revert") . "/revert-created-files";
	$GLOBAL_BFILE{"created-dirs"}= &getGlobal('BDIR', "revert") . "/revert-created-dirs";
	$GLOBAL_BFILE{"created-symlinks"} = &getGlobal('BDIR', "revert") . "/revert-created-symlinks";
	$GLOBAL_BFILE{"removed-symlinks"} = &getGlobal('BDIR', "revert") . "/revert-removed-symlinks";
	$GLOBAL_BFILE{"executed-commands"}= &getGlobal('BDIR', "revert") . "/revert-executed-commands";
	$GLOBAL_BFILE{"sum.csv"}= &getGlobal('BDIR', "revert") . "/sum.csv";
	# Required files for new revert process...
	$GLOBAL_BFILE{"revert-actions"}= &getGlobal('BDIR', "revert") . "/revert-actions";
        $GLOBAL_BFILE{"revert-directory-perms.sh"}=&getGlobal('BDIR', "revert") . "/revert-directory-perms.sh";
        $GLOBAL_BFILE{"BastilleBackEnd"} = "/usr/sbin/BastilleBackEnd";

	$GLOBAL_BFILE{"complete.xbm"}="/usr/share/Bastille/complete.xbm";
	$GLOBAL_BFILE{"incomplete.xbm"}="/usr/share/Bastille/incomplete.xbm";

	$GLOBAL_BFILE{"credits"}="/usr/share/Bastille/Credits";
	$GLOBAL_BDIR{"backup"}= &getGlobal('BDIR', "revert") . "/backup";
    
	$GLOBAL_BDIR{"config"} = "/etc/Bastille";
	$GLOBAL_BFILE{"config"} = &getGlobal('BDIR', "config") . "/config";

	$GLOBAL_BFILE{"TODO"}= $GLOBAL_BDIR{"log"} . "/TODO";
	$GLOBAL_BFILE{'lockfile'} = "/var/lock/subsys/bastille-lock";
	if ( $distro =~ "^DB" )  {
	        $GLOBAL_BFILE{'lockfile'} = "/var/lock/bastille/bastille-lock";
	}
	$GLOBAL_BFILE{'last.config'} = "/var/log/Bastille/last.config";
	$GLOBAL_BFILE{"nodisclaimer"}= &getGlobal('BDIR', "share") . "/.nodisclaimer";
    }
    elsif ($actualDistro =~ "^HP-UX" ) {
	&HP_ConfigureForDistro;
    }
    elsif ($actualDistro =~ "OSX" ) {
	&OSX_ConfigureForDistro;
    }
    else { 
	return 0;
    }
    if ( ($distro =~ "^RH" ) or ($distro =~ "^MN") or ($distro =~ "^DB") or ($distro =~ "^SE") or ($distro =~ "^TB")){

	#
	# Set necessary binaries
	#

	if (($distro =~ "^RH" ) or ($distro =~ "^MN") or ($distro =~ "^SE") or ($distro =~ "^TB") or ($distro =~ "^DB") ) {
	    $GLOBAL_BIN{"accton"} = "/usr/sbin/accton";
	    $GLOBAL_FILE{"accton"} = "/usr/sbin/accton";
	}
	if ($distro =~ "^RH6.2" ) {
	    $GLOBAL_BIN{"accton"} = "/sbin/accton";
	    $GLOBAL_FILE{"accton"} = "/sbin/accton";
	}

	$GLOBAL_BIN{"bash"} = "/bin/bash";
        $GLOBAL_BIN{"chattr"}="/usr/bin/chattr";
	$GLOBAL_BIN{"chgrp"}="/bin/chgrp";
	$GLOBAL_BIN{"chmod"}="/bin/chmod";
        $GLOBAL_BIN{"chown"}="/bin/chown";
        $GLOBAL_BIN{"cksum"}="/usr/bin/cksum";
        $GLOBAL_BIN{"cp"}="/bin/cp";
        $GLOBAL_BIN{"crontab"}="/usr/bin/crontab";
        $GLOBAL_BIN{"diff"}="/usr/bin/diff";
        $GLOBAL_BIN{"groupadd"}="/usr/sbin/groupadd";
        $GLOBAL_BIN{"killall"}="/usr/bin/killall";
	$GLOBAL_BIN{"lilo"}="/sbin/lilo";
        $GLOBAL_BIN{"lpq"}="/usr/bin/lpq";
        $GLOBAL_BIN{"lprm"}="/usr/bin/lprm";
        $GLOBAL_BIN{"lpr"}="/usr/bin/lpr";
	$GLOBAL_BIN{"lpstat"}="/usr/bin/lpstat";
	$GLOBAL_BIN{"lpd"}="/usr/sbin/lpd";
	$GLOBAL_BIN{"cupsd"}="/usr/sbin/cupsd";
        $GLOBAL_BIN{"mknod"}="/bin/mknod";
        $GLOBAL_BIN{"mount"}="/bin/mount";
	$GLOBAL_BIN{"smbmnt"}='/usr/bin/smbmnt';
        $GLOBAL_BIN{"mv"}="/bin/mv";
	$GLOBAL_BIN{"rpm"}="/bin/rpm";
        $GLOBAL_BIN{"touch"}="/bin/touch";
	$GLOBAL_BIN{"more"}="/bin/more";
	$GLOBAL_BIN{"echo"}="/bin/echo";
         if ( $distro =~ /^DB/ ) {
		$GLOBAL_BIN{"dpkg"}="/usr/bin/dpkg";
		 $GLOBAL_BIN{"apt-get"}="/usr/sbin/apt-get";
	}
        $GLOBAL_BIN{"sulogin"}="/sbin/sulogin";
        $GLOBAL_BIN{"umount"}="/bin/umount";
        $GLOBAL_BIN{"useradd"}="/usr/sbin/useradd";
	# Added the next few for 1.2.0.pre12
	$GLOBAL_BIN{"ping"}="/bin/ping";
	if ( $distro =~ /^RH(\d+)/) {
		if ($1 >= 7.0) {
			$GLOBAL_BIN{"ping6"}="/usr/sbin/ping6";
		}
	}
	if ( $distro =~ /^DB(\d+)/) {
		$GLOBAL_BIN{"ping6"}="/bin/ping6";
	}
	$GLOBAL_BIN{"dump"}="/sbin/dump";
	$GLOBAL_BIN{"restore"}="/sbin/restore";
	$GLOBAL_BIN{"cardctl"}="/sbin/cardctl";
	$GLOBAL_BIN{"at"}="/usr/bin/at";
	$GLOBAL_BIN{"dos"}="/usr/bin/dos";
	$GLOBAL_BIN{"inndstart"}="/usr/bin/inndstart";
	$GLOBAL_BIN{"startinnfeed"}="/usr/bin/startinnfeed";
	$GLOBAL_BIN{"rsh"}="/usr/bin/rsh";
	$GLOBAL_BIN{"rcp"}="/usr/bin/rcp";
	$GLOBAL_BIN{"rlogin"}="/usr/bin/rlogin";
	$GLOBAL_BIN{"usernetctl"}="/usr/sbin/usernetctl";
	$GLOBAL_BIN{"traceroute"}="/usr/sbin/traceroute";
	$GLOBAL_BIN{"traceroute"}="/usr/sbin/traceroute";
	if ( $distro =~ /^RH(\d+)/) {
		if ($1 >= 7.0) {
			$GLOBAL_BIN{"traceroute6"}="/usr/sbin/traceroute6";
		}
	}
	if ( $distro =~ /^DB(\d+)/) {
		$GLOBAL_BIN{"traceroute6"}="/usr/bin/traceroute6";
	}
	$GLOBAL_BIN{"rlogind"}="/usr/sbin/in.rlogind";
	$GLOBAL_BIN{"rshd"}="/usr/sbin/in.rshd";
	$GLOBAL_BIN{"rexecd"}="/usr/sbin/in.rexecd";
	$GLOBAL_BIN{"rm"}="/bin/rm";
	$GLOBAL_BIN{"rmdir"}="/bin/rmdir";
	$GLOBAL_BIN{"ln"}="/bin/ln";
	$GLOBAL_BIN{"md5sum"}="/usr/bin/md5sum";
	$GLOBAL_BIN{"Xwrapper"}="/usr/X11R6/bin/Xwrapper";
	$GLOBAL_BIN{"XFree86"}="/usr/X11R6/bin/XFree86";

	#
	# Set necessary directories
	#

	$GLOBAL_DIR{"home"}="/home";
	$GLOBAL_DIR{"initd"}="/etc/rc.d/init.d";
        if ( ($distro =~ /^DB/) or ($distro =~ /^SE/)) {
	         $GLOBAL_DIR{"initd"} = "/etc/init.d";
	}
        $GLOBAL_DIR{"log"}="/var/log";
        $GLOBAL_DIR{"pamd"}="/etc/pam.d";
        $GLOBAL_DIR{"rcd"}="/etc/rc.d";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_DIR{"rcd"} = "/etc";
	}
	$GLOBAL_DIR{"sbin"}="/sbin";


	#
	# Set necessary config/misc files
	#

        $GLOBAL_DIR{"floppy"}="/mnt/floppy";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_DIR{"floppy"}="/floppy";
	}
        $GLOBAL_FILE{"group"}="/etc/group";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_FILE{"httpd.conf"}="/etc/apache/httpd.conf";
	}
	elsif ($distro =~ /^SE/ ) {
		$GLOBAL_FILE{"httpd.conf"}="/etc/httpd/httpd.conf";
	}
	else {
		$GLOBAL_FILE{"httpd.conf"}="/etc/httpd/conf/httpd.conf";
	}
	$GLOBAL_FILE{"Questions"} = &getGlobal('BDIR', "share") . "/Questions.txt";

	if ( $distro =~ /^DB2/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/apache/access.conf";
	} elsif ( $distro =~ /^DB3/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/apache/httpd.conf";
	} elsif ( $distro =~ /^RH6.[01]/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/httpd/conf/access.conf";
	} elsif ($distro =~ /^SE/ ) {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/httpd/httpd.conf";
	} else {
		$GLOBAL_FILE{"httpd_access.conf"}="/etc/httpd/conf/httpd.conf";
	}

        $GLOBAL_FILE{"inittab"}="/etc/inittab";
        $GLOBAL_FILE{"lilo.conf"}="/etc/lilo.conf";
	$GLOBAL_FILE{"grub.conf"}="/etc/grub.conf";
        $GLOBAL_FILE{"limits.conf"}="/etc/security/limits.conf";
        $GLOBAL_FILE{"mtab"}="/etc/mtab";
        $GLOBAL_FILE{"pam_access.conf"}="/etc/security/access.conf";
	$GLOBAL_FILE{"hosts.allow"}="/etc/hosts.allow";
	$GLOBAL_FILE{"inetd.conf"}="/etc/inetd.conf";
	$GLOBAL_FILE{"xinetd.conf"}="/etc/xinetd.conf";
	$GLOBAL_DIR{"xinetd.d"}="/etc/xinetd.d";

	# Added for 1.2.0.pre12 to implement more requires_ tags...
	$GLOBAL_FILE{"rsh"}="/usr/bin/rsh";
	$GLOBAL_FILE{"gcc"}="/usr/bin/gcc";
	$GLOBAL_FILE{"chkconfig_apmd"}=&getGlobal('DIR', "rcd")."/rc3.d/S26apmd";
	$GLOBAL_FILE{"chkconfig_nfs"}=&getGlobal('DIR', "rcd")."/rc3.d/S60nfs";
	$GLOBAL_FILE{"chkconfig_pcmcia"}=&getGlobal('DIR', "rcd")."/rc3.d/S45pcmcia";
	$GLOBAL_FILE{"chkconfig_dhcpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S65dhcpd";
	$GLOBAL_FILE{"chkconfig_gpm"}=&getGlobal('DIR', "rcd")."/rc3.d/S85gpm";
	$GLOBAL_FILE{"chkconfig_innd"}=&getGlobal('DIR', "rcd")."/rc3.d/S95innd";
	$GLOBAL_FILE{"chkconfig_gated"}=&getGlobal('DIR', "rcd")."/rc3.d/S32gated";
	$GLOBAL_FILE{"chkconfig_routed"}=&getGlobal('DIR', "rcd")."/rc3.d/S55gated";
	$GLOBAL_FILE{"chkconfig_ypbind"}=&getGlobal('DIR', "rcd")."/rc3.d/S17ypbind";
	$GLOBAL_FILE{"chkconfig_snmpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S50snmpd";
	$GLOBAL_FILE{"ypserv"}=&getGlobal('DIR', "initd")."ypserv";
	$GLOBAL_FILE{"rc.config"}="/etc/rc.config"; 

	if ($distro =~ /^SE/ ) {
	    $GLOBAL_FILE{"chkconfig_apmd"}=&getGlobal('FILE', "rc.config");
	    $GLOBAL_FILE{"chkconfig_nfs"}=&getGlobal('FILE', "rc.config");
	    $GLOBAL_FILE{"chkconfig_pcmcia"}=&getGlobal('FILE', "rc.config");
	    $GLOBAL_FILE{"chkconfig_dhcpd"}=&getGlobal('FILE', "rc.config");
	    $GLOBAL_FILE{"chkconfig_gpm"}=&getGlobal('FILE', "rc.config");
	    $GLOBAL_FILE{"chkconfig_ypbind"}=&getGlobal('FILE', "rc.config");
	    $GLOBAL_FILE{"ypserv"}=&getGlobal('FILE', "rc.config");
        }
#TODO: This values need to be checked more thoroughly for Debian (jfs)
	if ( $distro =~ /^DB/ ) {
#        $GLOBAL_FILE{"chkconfig_apmd"}=&getGlobal('DIR', "rcd")."/rc3.d/S26apmd";
#        $GLOBAL_FILE{"chkconfig_nfs"}=&getGlobal('DIR', "rcd")."/rc3.d/S60nfs";
#        $GLOBAL_FILE{"chkconfig_pcmcia"}=&getGlobal('DIR', "rcd")."/rc3.d/S45pcmcia";
#        $GLOBAL_FILE{"chkconfig_dhcpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S65dhcpd";
		$GLOBAL_FILE{"chkconfig_gpm"}=&getGlobal('DIR', "rcd")."/rc3.d/S20gpm";
#        $GLOBAL_FILE{"chkconfig_innd"}=&getGlobal('DIR', "rcd")."/rc3.d/S95innd";
#        $GLOBAL_FILE{"chkconfig_gated"}=&getGlobal('DIR', "rcd")."/rc3.d/S32gated";
#        $GLOBAL_FILE{"chkconfig_routed"}=&getGlobal('DIR', "rcd")."/rc3.d/S55gated";
#        $GLOBAL_FILE{"chkconfig_ypbind"}=&getGlobal('DIR', "rcd")."/rc3.d/S17ypbind";
#        $GLOBAL_FILE{"chkconfig_snmpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S50snmpd";
       }


	$GLOBAL_FILE{"sendmail.cf"}="/etc/sendmail.cf"; 
	$GLOBAL_FILE{"sysconfig_sendmail"}="/etc/sysconfig/sendmail";
	$GLOBAL_FILE{"named"}="/usr/sbin/named";
        $GLOBAL_BIN{'named-xfer'}='/usr/sbin/named-xfer';
# Note: Shouldn't all these be placed in the same zone so as to not
# increase the chances of not fixing them for any given distro? (jfs)
	$GLOBAL_FILE{"chkconfig_named"}=&getGlobal('DIR', "rcd")."/rc3.d/S55named";
	$GLOBAL_FILE{"chkconfig_httpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S85httpd";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_FILE{"chkconfig_named"}=&getGlobal('DIR', "rcd")."/rc3.d/S15named";
		$GLOBAL_FILE{"chkconfig_httpd"}=&getGlobal('DIR', "rcd")."/rc3.d/S91httpd";
	}
	$GLOBAL_FILE{"httpd"}="/usr/sbin/httpd";
	if ( $distro =~ /^DB/ ) {
		$GLOBAL_FILE{"httpd"}="/usr/sbin/apache";
	}
	$GLOBAL_FILE{"lpr"}="/usr/bin/lpr";
	$GLOBAL_FILE{"ftpaccess"}="/etc/ftpaccess";
	$GLOBAL_FILE{"tcpd"}="/usr/sbin/tcpd";
	if ($distro eq 'RH7.2') {
	    $GLOBAL_FILE{"banners_makefile"}="/usr/share/doc/tcp_wrappers-7.6/Banners.Makefile";
	}
	elsif ($distro eq 'TB7.0') {
	    $GLOBAL_FILE{"banners_makefile"}="/usr/share/doc/packages/tcp_wrappers-7.6/Banners.Makefile";}
	else {
	    $GLOBAL_FILE{"banners_makefile"}="/usr/share/doc/tcp_wrappers-7.5/Banners.Makefile";
	}
	$GLOBAL_FILE{"profile"}="/etc/profile";
	$GLOBAL_FILE{"csh.login"}="/etc/csh.login";
	$GLOBAL_FILE{"rootprofile"}="/root/.bash_profile";
	$GLOBAL_FILE{"zprofile"}="/etc/zprofile";
	$GLOBAL_FILE{"motd"}="/etc/motd";
	$GLOBAL_FILE{"issue"}="/etc/issue";
	$GLOBAL_FILE{"cron.allow"}='/etc/cron.allow';
    }
    elsif ($distro =~ "^HP-UX" ) {
	&HP_ConfigureForDistro;
    }
    else {
	$retval=0;
    }

    # OS dependent error messages (after configuring file locations)
    my $nodisclaim_file = &getGlobal('BFILE', "nodisclaimer");
    $GLOBAL_ERROR{"disclaimer"}="$err Unable to touch $nodisclaim_file:" .
	    "$spc You must use Bastille\'s -n flag (for example:\n" .
	    "$spc bastille -i -n) or \'touch $nodisclaim_file \'\n"; 

  return $retval;
}


##############################################################################
# &ActionLog ($Log_items) prints $log_items to the action-log.  If said log
# doesn't exist, it is created.
##############################################################################

sub ActionLog {
	my @Actions = @_; # List of parameters passed to ActionLog
	my $datestamp = "{" . localtime() . "}";

	# Just in case
	if ( defined $GLOBAL_BFILE{"action-log"} ) {
		unless ( -e &getGlobal('BFILE', "action-log") ) {
		        unless ( -e &getGlobal('BDIR', "log") ) {
	                   mkpath (&getGlobal('BDIR',"log"),0,0700);
		        }
			open ACTIONLOG,">" . &getGlobal('BFILE', "action-log");
			close ACTIONLOG;
		}

		open ACTIONLOG,">>" . &getGlobal('BFILE', "action-log") or print STDERR "Couldn't open action-log file in ".&getGlobal('BFILE', "action-log").": $!\n";
		foreach my $thing ( @Actions ) {
			print ACTIONLOG "$datestamp $thing";
		}

		close ACTIONLOG;
	}
	if ( $GLOBAL_VERBOSE ) {
		foreach my $thing ( @Actions ) {
			print STDERR "$thing";
		}
	}

}

##############################################################################
# &DebugLog ($Log_items) prints $log_items to the debug-log.  If said log
# doesn't exist, it is created.
##############################################################################
sub DebugLog {
       my @Debugs = @_; # List of parameters passed to DebugLog
       my $datestamp = "{" . localtime() . "}";

       # Check if the debug log file exists and the GLOBAL_DEBUG
       # option is set (this avoids having a lot of
       # &DebugLog() if $GLOBAL_DEBUG;
       # statements)
       # Note: probably this is less efficient because there's an
       # unnecesary call to a subroutine.
       if ( $GLOBAL_DEBUG && defined $GLOBAL_BFILE{"debug-log"} ) {
               unless ( -e &getGlobal('BFILE', "debug-log") ) {
                       open DEBUGLOG,">" . &getGlobal('BFILE', "debug-log");
                      close DEBUGLOG;
               }

               open DEBUGLOG,">>" . &getGlobal('BFILE', "debug-log") or print STDERR "Couldn't open debug-log file in ".&getGlobal('BFILE', "debug-log").": $!\n";
               foreach my $thing ( @Debugs ) {
                       print DEBUGLOG "$datestamp $thing";
               }

               close DEBUGLOG;
       }
# Do we want this? (print to STDERR the debug output as well as in the debug file?
# (Pending discussion - jfs)
#      if ( $GLOBAL_VERBOSE && $GLOBAL_DEBUG) {
#              foreach my $thing ( @Debugs ) {
#                      print STDERR "DEBUG: $thing";
#              }
#      }

}


##############################################################################
# &ErrorLog ($log_items) prints $log_items to the error-log.  If said log
# doesn't exist, it is created.
#
# The idea for this log was Mike Rash's (mbr).
##############################################################################

sub ErrorLog {
    my @Errors = @_; # List of errors passed to ErrorLog
    # Write something to our ongoing "errorlog" file
    
    my $datestamp = "{" . localtime() . "}";

    if(! defined $errorFlag) {
	$errorFlag = 1;
    }

    my $errorlog="/var/log/Bastille/error-log";

    if(&getActualDistro =~ "^HP-UX"){
       $errorlog = "/var/opt/sec_mgmt/bastille/log/error-log";
    } 

    unless ( -e $errorlog ) {
	open ERRORLOG,">$errorlog";
	close ERRORLOG;
    }

    unless (open ERRORLOG,">>$errorlog") {
       print STDERR "ERROR:  Couldn't open error-log: $!\n";
    }
    
    foreach $thing ( @Errors ) {
	print ERRORLOG "$datestamp $thing";
	print STDERR $thing;  
	# IMHO if the user is not warned of stuff  he might not be aware of
	# the logs (jfs)
    }
    
    close ERRORLOG;	


}


###########################################################################
# GetYN and GetString  are the two original v1.0.0-v1.0.3 input functions #
# used to read in (and log) input.                                        #
# Both functions ignore hash-commented lines.                             #
#                                                                         #
# Under the new architecture, these functions should fall out of use      #
# entirely , as we move to a TUI that writes AppConfig files.             #
#                                                                         #
###########################################################################

# &GetYN grabs a Y or an N from stdin.  It 
sub GetYN {
    # enhanced by Don Wilder and Peter W.
    my ( $prefix ) = @_;        # optional prefix
    my ($line, $ok, $warn);
    
    print "Press <Shift><Page Up>/<Page Down> to see previously scrolled text.\n";
    $ok = 0;
    while ( $ok == 0 ) {
       print "$warn$prefix(Y or N): ";
       $line=<STDIN>;
       print INPUTLOG $line;   # log answer
       if ( ($line =~ /^#/) or ($line !~ /^[YN]/i ) ) {
          # If we're in this loop, we've either read a comment, or the
	  # input did not begin with Y or N.  Ask again.
	  $warn = "Please respond with Y or N. ";
       } 
       else {
          $ok = 1;
       }
    }
    return(uc(substr($line,0,1)));

}

sub GetString {

   my $line;
		      
   #  Take the first line that isn't a comment, that is, the first line
   #  not beginning with:      
   #                   whitespace #
   while ( ($line=<STDIN>) =~ /^(\s*)#/) { 
      print INPUTLOG $line;
   } 

   print INPUTLOG $line;
   # Now, strip out the comment at the end...

   #
   # Note the strange structure (via the while statement).  This is to
   # prevent weird comments, like:
 
   #                
   #           #  this is a comment # with pound's in the # middle...
   # or 
   #           ##  this is a better example: 
   #      
   #           #  dial  #,#,1,2,3
   # 
    
   while ( $line =~ /^(.*[^\\])#/ ) {
      $pre_pound=$1;
      $pre_pound =~ s/(.*[^\s])\s+/$1/;
      $line = $pre_pound;
   }
   
   # Finally, change all escaped #'s to #'s:   
   $line =~ s/\\#/#/g;

   # Return the found line
   $line;
			      
}


###########################################################################
###########################################################################
#                                                                         #
# The B_<perl_function> file utilities are replacements for their Perl    #
# counterparts.  These replacements log their actions and their errors,   #
# but are very similar to said counterparts.                              #
#                                                                         #
###########################################################################
###########################################################################


###########################################################################
# B_open was the v1.0 open command.  It is still used in places in the
# code, though it should fall out of use quickly in v1.1, as it has been
# replaced by B_open_plus, which implements a new, smarter, backup scheme.
#
# B_open opens the given filehandle, associated with the given filename
# and logs appropriately.
#
###########################################################################

sub B_open {
   my $retval=1;
   my ($handle,$filename)=@_;

   unless ($GLOBAL_LOGONLY) {
       $retval = open $handle,$filename;
   }

   ($handle) = "$_[0]" =~ /[^:]+::[^:]+::([^:]+)/;
   &ActionLog("open $handle,\"$filename\";\n");
   unless ($retval) {
      &ActionLog("#open $handle , $filename failed...\n");
      &ErrorLog("#open $handle, $filename failed...\n");
   }
   
   return $retval;
}   

###########################################################################
# B_open_plus is the v1.1 open command.
# 
# &B_open_plus($handle_file,$handle_original,$file) opens the file $file
# for reading and opens the file ${file}.bastille for writing.  It is the
# counterpart to B_close_plus, which will move the original file to
# $GLOBAL_BDIR{"backup"} and will place the new file ${file}.bastille in its
# place.
#
# &B_open_plus makes the appropriate log entries in the action and error
# logs.
###########################################################################

sub B_open_plus {

    my ($handle_file,$handle_original,$file)=@_;
    my $retval=1;
    my $return_file=1;
    my $return_old=1;
  
    my $original_file = $file;

    # Open the original file and open a copy for writing.
    unless ($GLOBAL_LOGONLY) {
	# if the temporary filename already exists then the open operation will fail.
	if (-e "${file}.bastille") {
	    &ErrorLog("ERROR:   Unable to open $file as the\n" . 
		      "         swap file ${file}.bastille\n" .
		      "         already exists.  Rename the swap file to allow Bastille\n" .
		      "         to make desired file modifications.\n");
	    $return_old=0;
	    $return_file=0;
	}
	else {
	    $return_old = open $handle_original,"$file";
	    $return_file = open $handle_file,("> $file.bastille");
	}
    }
    
    # Error handling/logging here...
    #&ActionLog("# Modifying file $original_file via temporary file $original_file.bastille\n");
    unless ($return_file) {
	$retval=0;
	&ErrorLog("ERROR:   open $original_file.bastille failed...\n");
    }
    unless ($return_old) {
	$retval=0;
	&ErrorLog("ERROR:   open $original_file failed.\n");
    }

    return $retval;
        
}

###########################################################################
# B_close was the v1.0 close command.  It is still used in places in the
# code, though it should fall out of use quickly in v1.1, as it has been
# replaced by B_close_plus, which implements a new, smarter, backup scheme.
#
# B_close closes the given filehandle, associated with the given filename
# and logs appropriately.
###########################################################################


sub B_close {
   my $retval=1;

   unless ($GLOBAL_LOGONLY) {
       $retval = close $_[0];
   }

   &ActionLog( "close $_[0];\n");
   unless ($retval) {
      &ActionLog("#ERROR: close $_[0] failed...\n");
      &ErrorLog( "#ERROR: close $_[0] failed...\n");
   }

   return $retval;
}


###########################################################################
# B_close_plus is the v1.1 close command.
# 
# &B_close_plus($handle_file,$handle_original,$file) closes the files
# $file and ${file}.bastille, backs up $file to $GLOBAL_BDIR{"backup"} and
# renames ${file}.bastille to $file.  This backup is made using the
# internal API function &B_backup_file.  Further, it sets the new file's
# permissions and uid/gid to the same as the old file.
#
# B_close_plus is the counterpart to B_open_plus, which opened $file and 
# $file.bastille with the file handles $handle_original and $handle_file, 
# respectively.
#
# &B_close_plus makes the appropriate log entries in the action and error
# logs.
###########################################################################

sub B_close_plus {
    my ($handle_file,$handle_original,$file)=@_;
    my ($mode,$uid,$gid);
    my @junk;

    my $original_file;

    my $retval=1;
    my $return_file=1;
    my $return_old=1;

    # Append the global prefix, but save the original for B_backup_file b/c
    # it appends the prefix on its own...

    $original_file=$file;

    #
    # Close the files and prepare for the rename
    #

    unless ($GLOBAL_LOGONLY) {
	$return_file = close $handle_file;
	$return_old = close $handle_original;
    }

    # Error handling/logging here...
    #&ActionLog("#Closing $original_file and backing up to " . &getGlobal('BDIR', "backup"));
    #&ActionLog("/$original_file\n");

    unless ($return_file) {
	$retval=0;
	&ErrorLog("#close $original_file failed...\n");
    }
    unless ($return_old) {
	$retval=0;
	&ErrorLog("#close $original_file.bastille failed.\n");
    }

    #
    # If we've had no errors, backup the old file and put the new one
    # in its place, with the Right permissions.
    #

    unless ( ($retval == 0) or $GLOBAL_LOGONLY) {

	# Read the permissions/owners on the old file
	
	@junk=stat ($file);
	$mode=$junk[2];
	$uid=$junk[4];
	$gid=$junk[5];

	# Set the permissions/owners on the new file

	chmod $mode, "$file.bastille" or &ErrorLog("B_close_plus: Not able to retain permissions on $original_file!!!\n");
	chown $uid, $gid, "$file.bastille" or &ErrorLog("B_close_plus: Not able to retain owners on $original_file!!!\n");

	# Backup the old file and put a new one in place.
	
	&B_backup_file($original_file);
	rename "$file.bastille", $file or &ErrorLog("B_close_plus: not able to move $original_file.bastille to $original_file\n");

    }

    return $retval;
}

###########################################################################
# &B_backup_file ($file) makes a backup copy of the file $file in 
# &getGlobal('BDIR', "backup").  Note that this routine is intended for internal
# use only -- only Bastille API functions should call B_backup_file.
#
###########################################################################

sub B_backup_file {

    my $file=$_[0];
    my $complain = 1;
    my $original_file = $file;

    my $backup_dir = &getGlobal('BDIR', "backup");
    my $backup_file = $backup_dir . $original_file;

    # We don't use this -- this is mostly here for commenting.
    #my $relative_file;

    my $retval=1;

    # First, separate the file into the directory and the relative filename

    my $directory ="";
    if ($file =~ /^(.*)\/([^\/]+)$/) {
	#$relative_file=$2;
	$directory = $1;
    } else {
        $directory=cwd;
    }

    # Now, if the directory does not exist, create it.
    # Later:
    #   Try to set the same permissions on the patch directory that the
    #   original had...?

    unless ( -d ($backup_dir . $directory) ) {
	mkpath(( $backup_dir . $directory),0,0700);

    }

    # Now we backup the file.  If there is already a backup file there,
    # we will leave it alone, since it exists from a previous run and
    # should be the _original_ (possibly user-modified) distro's version 
    # of the file.

    if ( -e $file ) {
	# We add the file to the GLOBAL_SUMS hash if it is not already present
	&B_set_sum($file);

	unless ( -e $backup_file ) {
	    $command=&getGlobal("BIN","cp");
            `$command -p $file $backup_file`;
	    &B_revert_log (&getGlobal("BIN","mv"). " $backup_file $file");
	}

	# Later, following PeterW's suggestion, place a diff of the file
	# in a second backup directory

    } else {
	# The file we were trying to backup doesn't exist.

	$retval=0;
	# This is a non-fatal error, not worth complaining about
	$complain = 0;
	#&ErrorLog ("# Failed trying to backup file $file -- it doesn't exist!\n");
    }

    # Check to make sure that the file does exist in the backup location.
    
    unless ( -e $backup_file ) {
	$retval=0;
	if ( $complain == 1 ) { 
	    &ErrorLog("# Failed trying to backup $file -- the copy was not created.\n"); 
	}
    }

    return $retval;
}


###########################################################################
# &B_read_sums reads in the sum.csv file which contains information
#   about Bastille modified files. The file structure is as follows:
#
#     filename,filesize,cksum,change_flag
# 
#   It reads the information into the GLOBAL_SUM hash i.e.
#      $GLOBAL_SUM{$file}{sum} = $cksum
#      $GLOBAL_SUM{$file}{filesize} = $size
#      $GLOBAL_SUM{$file}{flag} = $flag
#   For the first run of Bastille on a given system this subroutine
#   is a no-op.
###########################################################################

sub B_read_sums {

    my $sumFile = &getGlobal('BFILE',"sum.csv");

    if ( -e $sumFile ) {

	open( SUM, "< $sumFile") or &ErrorLog("Unable to open $sumFile for read.\n$!\n");

	while( my $line = <SUM> ) {
	    chomp $line;
	    my ($file,$filesize,$sum,$flag) = split /,/, $line;
	    if(-e $file) {
		$GLOBAL_SUM{"$file"}{filesize} = $filesize;
		$GLOBAL_SUM{"$file"}{sum} = $sum;
		$GLOBAL_SUM{"$file"}{flag} = $flag;
	    }
	}

	close(SUM);
    }
}


###########################################################################
# &B_write_sums writes out the sum.csv file which contains information
#   about Bastille modified files. The file structure is as follows:
#
#     filename,filesize,cksum,change_flag
# 
#   It writes the information from the GLOBAL_SUM hash i.e.
#
#      $file,$GLOBAL_SUM{$file}{sum},$GLOBAL_SUM{$file}{filesize},$GLOBAL_SUM{$file}{flag}
#
#   This subroutine requires access to the GLOBAL_SUM hash.
###########################################################################

sub B_write_sums {

    my $sumFile = &getGlobal('BFILE',"sum.csv");

    if ( defined %GLOBAL_SUM ) {

	open( SUM, "> $sumFile") or &ErrorLog("Unable to open $sumFile for write.\n$!\n");

	for my $file (sort keys %GLOBAL_SUM) {
	    if( -e $file) {
		print SUM "$file,$GLOBAL_SUM{\"$file\"}{filesize},$GLOBAL_SUM{\"$file\"}{sum},$GLOBAL_SUM{\"$file\"}{flag}\n";
	    }
	}

	close(SUM);
    }

}


###########################################################################
# &B_check_sum($file) compares the stored cksum and filesize of the given
#   file compared to the current cksum and filesize respectivly.
#   It uses the GLOBAL_SUM hash to determine if the file has been modified
#   since the last run of Bastille.  If it has been modified the file flag
#   is set.
#
#     $GLOBAL_SUM{$file}{flag} = 1;
#
#   This subroutine also keeps the state of the sum check by setting the
#   checked flag which tells the subroutine that on this run this file
#   has already been checked.
#
#     $GLOBAL_SUM{$file}{checked} = 1;
#
#   This subroutine requires access to the GLOBAL_SUM hash.
###########################################################################
	
sub B_check_sum($) {
    my $file = $_[0];
    my $cksum = &getGlobal('BIN',"cksum");
    if(-e $file) {
	my ($sum,$size,$ckfile) = split /\s+/, `$cksum $file`;
        my $commandRetVal = ($? >> 8);  # find the command's return value

	if($commandRetVal != 0) {
	    &ErrorLog("$cksum reported the following error:\n$!\n");
	}
	else {
	    
	    if( exists $GLOBAL_SUM{$file} ) {
		# if the flag is currently set to 0
		if(! $GLOBAL_SUM{$file}{flag}) {
		    # if the file size or file sum differ from those recorded.
		    if( $GLOBAL_SUM{$file}{filesize} != $size ||
			$GLOBAL_SUM{$file}{sum} != $sum ) {
			# setting flag to 1
			$GLOBAL_SUM{$file}{flag} = 1;
		    }
		}
	    }
	    else {  
		&ErrorLog("ERROR:   The file $file does not\nexist in the sums database\n");
	    }
	}
    }
}

###########################################################################
# &B_set_sum($file) sets the current cksum and filesize of the given
#   file into the GLOBAL_SUM hash.
#
#     $GLOBAL_SUM{$file}{filesize} = $size;
#     $GLOBAL_SUM{$file}{sum} = $cksum;
#
#   This subroutine requires access to the GLOBAL_SUM hash.
###########################################################################
	
sub B_set_sum($) {

    my $file = $_[0];
    my $cksum = &getGlobal('BIN',"cksum");
    if( -e $file) {
    
	my ($sum,$size,$ckfile) = split /\s+/, `$cksum $file`;
        my $commandRetVal = ($? >> 8);  # find the command's return value

	if($commandRetVal != 0) {
	    
	    &ErrorLog("$cksum reported the following error:\n$!\n");
	    
	}
	else {
	    
	    # new file size and sum are added to the hash
	    $GLOBAL_SUM{$file}{filesize} = $size;
	    $GLOBAL_SUM{$file}{sum} = $sum;
	    # Ensure that each new sum added to the hash has a flag defined
	    if( ! exists $GLOBAL_SUM{$file}{flag} ) {
		$GLOBAL_SUM{$file}{flag} = 0;
	    }
	    &B_write_sums;

	}
    }
}


###########################################################################
# &B_blank_file ($filename,$pattern) blanks the file $filename, unless the
# pattern $pattern is present in the file.  This lets us completely redo
# a file, if it isn't the one we put in place on a previous run...
#
# B_blank_file respects $GLOBAL_LOGONLY and uses B_open_plus and B_close_plus
# so that it makes backups and only modifies files when we're not in "-v"
# mode...
#
# If the file does not exist, the function does nothing, and gives an error 
# to the Error Log
#
###########################################################################

sub B_blank_file($$) {
    
    my ($filename,$pattern) = @_;
    my $retval;

    # If this variable is true, we won't blank the file...

    my $found_pattern=0;

    if ($retval=&B_open_plus (*BLANK_NEW,*BLANK_OLD,$filename) ) {

	my @lines;
	
	while (my $line = <BLANK_OLD>) {

	    push @lines,$line;
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	}

	# Only copy the old file if the new one didn't match.
	if ($found_pattern) {
	    while ($line = shift @lines ) {
		&B_print(*BLANK_NEW,$line);
	    }
	}
	else {
	    &ActionLog("Blanked file $filename\n");
	}
	&B_close_plus(*BLANK_NEW,*BLANK_OLD,$filename);
    }
    else {
	&ErrorLog("Couldn't blank file $filename since we couldn't open it or its replacement\n");
    }

    return $retval;

}


###########################################################################
# &B_append_line ($filename,$pattern,$line_to_append)  modifies $filename,
# appending $line_to_append unless one or more lines in the file matches
# $pattern.  This is an enhancement to the append_line_if_no_such_line_exists
# idea.
#
# Additionally, if $pattern is set equal to "", the line is always appended.  
#
# B_append_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here's examples of where you might use this:
#
# You'd like to add a   root   line to /etc/ftpusers if none exists.
# You'd like to add a   Options Indexes  line to Apache's config. file,
# after you delete all Options lines from said config file.
#
###########################################################################

sub B_append_line {
   
    my ($filename,$pattern,$line_to_append) = @_;

    my $found_pattern=0;
    my $retval=1;

    if ( &B_open_plus (*APPEND_NEW,*APPEND_OLD,$filename) ) {
	while (my $line=<APPEND_OLD>) {
	    &B_print(*APPEND_NEW,$line);
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	}
	# Changed != 0 to $pattern so that "" works instead of 0 and perl
	# does not give the annoying
	# Argument "XX" isn't numeric in ne at ...
	if ( $pattern eq "" or ! $found_pattern ) {
	    &B_print(*APPEND_NEW,$line_to_append);
	    &ActionLog("Appended the following line to $filename:\n");
	    &ActionLog("$line_to_append");
	}
	&B_close_plus (*APPEND_NEW,*APPEND_OLD,$filename);
    }
    else {
	$retval=0;
	&ErrorLog("# Couldn't append line to $filename, since open failed.");
    }

    return $retval;

}


###########################################################################
# &B_insert_line ($filename,$pattern,$line_to_insert,$line_to_follow)  
# modifies $filename, inserting $line_to_insert unless one or more lines
# in the file matches $pattern.  The $line_to_insert will be placed
# immediately after $line_to_follow, if it exists.  If said line does not
# exist, the line will not be inserted and this routine will return 0.
#
# B_insert_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here's examples of where you might use this:
#
# You'd like to insert a line in Apache's configuration file, in a 
# particular section.
#
###########################################################################

sub B_insert_line {
  
    my ($filename,$pattern,$line_to_insert,$line_to_follow) = @_;

    my @lines;
    my $found_pattern=0;
    my $found_line_to_follow=0;

    my $retval=1;

    if ( &B_open_plus (*INSERT_NEW,*INSERT_OLD,$filename) ) {

	# Read through the file looking for a match both on the $pattern
	# and the line we are supposed to be inserting after...

	my $ctr=1;
	while (my $line=<INSERT_OLD>) {
	    push (@lines,$line);
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	    if ( ($found_line_to_follow < 1) and ($line =~ $line_to_follow)) {
		$found_line_to_follow=$ctr;
	    }
	    $ctr++;
	}

	# Log an error if we never found the line we were to insert after
	unless ($found_line_to_follow ) {
	    $retval=0;
	    &ErrorLog("Never found the line that we were supposed to insert after in $filename\n");
	}

	# Now print the file back out, inserting our line if we should...

	$ctr=1;
	while (my $line = shift @lines) {
	    &B_print(*INSERT_NEW,$line);
	    if ( ($ctr == $found_line_to_follow) and ($found_pattern == 0) ) {
		&B_print(*INSERT_NEW,$line_to_insert);
		&ActionLog("Inserted the following line in $filename:\n");
		&ActionLog("$line_to_insert");
	    }
	    $ctr++;
	}

	&B_close_plus (*INSERT_NEW,*INSERT_OLD,$filename);
	
    }
    else {
	$retval=0;
	&ErrorLog("# Couldn't prepend line to $filename, since open failed.");
    }

    return $retval;

}


###########################################################################
# &B_prepend_line ($filename,$pattern,$line_to_prepend)  modifies $filename,
# prepending $line_to_prepend unless one or more lines in the file matches
# $pattern.  This is an enhancement to the prepend_line_if_no_such_line_exists
# idea.  
#
# B_prepend_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here's examples of where you might use this:
#
# You'd like to insert the line "auth   required   pam_deny.so" to the top
# of the PAM stack file /etc/pam.d/rsh to totally deactivate rsh.
#
###########################################################################

sub B_prepend_line {
   
    my ($filename,$pattern,$line_to_prepend) = @_;

    my @lines;
    my $found_pattern=0;
    my $retval=1;

    if ( &B_open_plus (*PREPEND_NEW,*PREPEND_OLD,$filename) ) {
	while (my $line=<PREPEND_OLD>) {
	    push (@lines,$line);
	    if ($line =~ $pattern) {
		$found_pattern=1;
	    }
	}
	unless ($found_pattern) {
	    &B_print(*PREPEND_NEW,$line_to_prepend);
	}
	while (my $line = shift @lines) {
	    &B_print(*PREPEND_NEW,$line);
	}

	&B_close_plus (*PREPEND_NEW,*PREPEND_OLD,$filename);
	
	# Log the action
	&ActionLog("Prepended the following line to $filename:\n");
	&ActionLog("$line_to_prepend");
    }
    else {
	$retval=0;
	&ErrorLog("# Couldn't prepend line to $filename, since open failed.\n");
    }

    return $retval;

}


###########################################################################
# &B_replace_line ($filename,$pattern,$line_to_switch_in) modifies $filename,
# replacing any lines matching $pattern with $line_to_switch_in.
#
# It returns the number of lines it replaced (or would have replaced, if
# LOGONLY mode wasn't on...)
#
# B_replace_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here an example of where you might use this:
#
# You'd like to replace any Options lines in Apache's config file with:
#            Options Indexes FollowSymLinks
#
###########################################################################

sub B_replace_line {
   
    my ($filename,$pattern,$line_to_switch_in) = @_;
    my $retval=0;

    if ( &B_open_plus (*REPLACE_NEW,*REPLACE_OLD,$filename) ) {
	while (my $line=<REPLACE_OLD>) {
	    unless ($line =~ $pattern) {    
		&B_print(*REPLACE_NEW,$line);
	    }
	    else {
		# Don't replace the line if it's already there.
		unless ($line eq $line_to_switch_in) {
		    &B_print(*REPLACE_NEW,$line_to_switch_in);
		
		    $retval++;
		    &ActionLog("# File modification in $filename -- replaced line\n");
		    &ActionLog($line);
		    &ActionLog("with:\n");
		    &ActionLog("$line_to_switch_in");
		}
                # But if it is there, make sure it stays there! (by Paul Allen)
		else {
		    &B_print(*REPLACE_NEW,$line);
                }    
	    }
	}
	&B_close_plus (*REPLACE_NEW,*REPLACE_OLD,$filename);
    }
    else {
	$retval=0;
	&ErrorLog("Couldn't replace line(s) in $filename because open failed.\n");
    }

    return $retval;
}

################################################################################################
# &B_replace_pattern ($filename,$pattern,$pattern_to_remove,$text_to_switch_in)
# modifies $filename, acting on only lines that match $pattern, replacing a 
# string that matches $pattern_to_remove with $text_to_switch_in.
#
# Ex:
#  B_replace_pattern('/etc/httpd.conf','^\s*Options.*\bIncludes\b','Includes','IncludesNoExec')
#
#   replaces all "Includes" with "IncludesNoExec" on Apache Options lines.
#
# It returns the number of lines it altered (or would have replaced, if
# LOGONLY mode wasn't on...)
#
# B_replace_pattern uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
#################################################################################################

sub B_replace_pattern {
   
    my ($filename,$pattern,$pattern_to_remove,$text_to_switch_in) = @_;
    my $retval=0;

    if ( &B_open_plus (*REPLACE_NEW,*REPLACE_OLD,$filename) ) {
	while (my $line=<REPLACE_OLD>) {
	    unless ($line =~ $pattern) {    
		&B_print(*REPLACE_NEW,$line);
	    }
	    else {
		print "JJB: We found a matching line: $line\n";
		my $orig_line =$line;
		$line =~ s/$pattern_to_remove/$text_to_switch_in/;

		&B_print(*REPLACE_NEW,$line);

		$retval++;
		&ActionLog("# File modification in $filename -- replaced line\n");
		&ActionLog("$orig_line");
		&ActionLog("via pattern with:\n");
		&ActionLog("$line");
	    }
	}
	&B_close_plus (*REPLACE_NEW,*REPLACE_OLD,$filename);
    }
    else {
	$retval=0;
	&ErrorLog("Couldn't pattern-replace line(s) in $filename because open failed.\n");
    }

    return $retval;
}

###########################################################################
# &B_hash_comment_line ($filename,$pattern) modifies $filename, replacing 
# any lines matching $pattern with a "hash-commented" version, like this:
#
#
#        finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
# becomes:
#        #finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
#
# Also:
#       tftp        dgram  udp wait   root /usr/lbin/tftpd    tftpd\
#        /opt/ignite\
#        /var/opt/ignite
# becomes:
#       #tftp        dgram  udp wait   root /usr/lbin/tftpd    tftpd\
#       # /opt/ignite\
#       # /var/opt/ignite
#
#
# B_hash_comment_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
###########################################################################

sub B_hash_comment_line {
   
    my ($filename,$pattern) = @_;
    my $retval=1;

    if ( &B_open_plus (*HASH_NEW,*HASH_OLD,$filename) ) {
	while (my $line=<HASH_OLD>) {
	    unless ( ($line =~ $pattern) and ($line !~ /^\s*\#/) ) {    
		&B_print(*HASH_NEW,$line);
	    }
	    else {
		&B_print(*HASH_NEW,"#$line");
		&ActionLog("# File modification in $filename -- hash commented line\n");
		&ActionLog($line);
		&ActionLog("like this:\n");
		&ActionLog("#$line");
		# while the line has a trailing \ then we should also comment out the line below
		while($line =~ m/\\\n$/) {
		    if($line=<HASH_OLD>) {
			&B_print(*HASH_NEW,"#$line");
			&ActionLog("# File modification in $filename -- hash commented line\n");
			&ActionLog($line);
			&ActionLog("like this:\n");
			&ActionLog("#$line");
		    }
		    else {
			$line = "";
		    }
		}
			
	    }
	}
	&B_close_plus (*HASH_NEW,*HASH_OLD,$filename);
    }
    else {
	$retval=0;
	&ErrorLog("Couldn't hash-comment line(s) in $filename because open failed.\n");
    }

    return $retval;
}


###########################################################################
# &B_hash_uncomment_line ($filename,$pattern) modifies $filename, 
# removing any commentting from lines that match $pattern.
#
#        #finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
# becomes:
#        finger  stream  tcp     nowait  nobody  /usr/sbin/tcpd  in.fingerd
#
#
# B_hash_uncomment_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
###########################################################################

sub B_hash_uncomment_line {
   
    my ($filename,$pattern) = @_;
    my $retval=1;

    if ( &B_open_plus (*HASH_NEW,*HASH_OLD,$filename) ) {
	while (my $line=<HASH_OLD>) {
	    unless ( ($line =~ $pattern) and ($line =~ /^\s*\#/) ) {    
		&B_print(*HASH_NEW,$line);
	    }
	    else {
		$line =~ /^\s*\#+(.*)$/;
		$line = "$1\n";

		&B_print(*HASH_NEW,"$line");
		&ActionLog("File modification in $filename -- hash uncommented line\n");
		&ActionLog($line);
	    }
	}
	&B_close_plus (*HASH_NEW,*HASH_OLD,$filename);
    }
    else {
	$retval=0;
	&ErrorLog("Couldn't hash-uncomment line(s) in $filename because open failed.\n");
    }

    return $retval;
}



###########################################################################
# &B_delete_line ($filename,$pattern) modifies $filename, deleting any 
# lines matching $pattern.  It uses B_replace_line to do this.
#
# B_replace_line uses B_open_plus and B_close_plus, so that the file
# modified is backed up...
#
# Here an example of where you might use this:
#
# You'd like to remove any timeout=  lines in /etc/lilo.conf, so that your
# delay=1 modification will work.

#
###########################################################################


sub B_delete_line {

    my ($filename,$pattern)=@_;
    my $retval=&B_replace_line($filename,$pattern,"");

    return $retval;
}


###########################################################################
# &B_chunk_replace ($file,$pattern,$replacement) reads $file replacing the
# first occurence of $pattern with $replacement.
# 
###########################################################################

sub B_chunk_replace {

    my ($file,$pattern,$replacement) = @_;

    my @lines;
    my $big_long_line;
    my $retval=1;

    &B_open (*OLDFILE,$file);

    # Read all lines into one scalar.
    @lines = <OLDFILE>;
    &B_close (*OLDFILE);
    foreach my $line ( @lines ) {
	$big_long_line .= $line;
    }

    # Substitution routines get weird unless last line is terminated with \n
    chomp $big_long_line;
    $big_long_line .= "\n";

    # Exit if we don't find a match
    unless ($big_long_line =~ $pattern) {
	return 0;
    }
    
    $big_long_line =~ s/$pattern/$replacement/s;

    $retval=&B_open_plus (*NEWFILE,*OLDFILE,$file);
    if ($retval) {
	&B_print (*NEWFILE,$big_long_line);
	&B_close_plus (*NEWFILE,*OLDFILE,$file);
    }

    return $retval;
}


###########################################################################
# &B_chunk_append ($file,$pattern,$payload) reads $file adding the $payload
# after the first occurence of $pattern.
# 
###########################################################################

sub B_chunk_append {

    my ($file,$pattern,$payload) = @_;

    my @lines;
    my $big_long_line;
    my $retval=1;
    my $position;

    my ($first_half,$second_half);

    &B_open(*OLDFILE,$file);

    # Read all lines into one scalar.
    @lines = <OLDFILE>;
    &B_close(*OLDFILE);
    foreach my $line ( @lines ) {
	$big_long_line .= $line;
    }

    # Substitution routines get weird unless last line is terminated with \n
    chomp $big_long_line;
    $big_long_line .= "\n";

    # Exit if we don't find a match
    unless ($big_long_line =~ $pattern) {
	return 0;
    }

    # We need to figure out which position in the file to insert the text
    
    $position = pos $big_long_line;

    # Insert the payload, unless it's already there.

    $first_half = substr ($big_long_line, 0, ($position-1) );
    $second_half = substr ($big_long_line, $position);

    my $length_to_check = length ($payload);
    unless ($legnth_to_check < 10) {
	$legnth_to_check =10;
    }
    unless ( substr($second_half,0,length ($payload)) eq $payload ) { 
	$big_long_line = $first_half . $payload . $second_half;
    }

    # Write out the result

    $retval=&B_open_plus (*NEWFILE,*OLDFILE,$file);
    if ($retval) {
	&B_print (*NEWFILE,$big_long_line);
	&B_close_plus (*NEWFILE,*OLDFILE,$file);
    }

    return $retval;
}



###########################################################################
# &B_delete_file ($file)  deletes the file $file and makes a backup to
# the backup directory.
#
# Here an example of where you might use this:
#
# You'd like to add a   Options Indexes  line to Apache's config. file,
# after you delete all Options lines from said config file.
#
##########################################################################


sub B_delete_file {

    #
    # This API routine deletes the named file, backing it up first to the
    # backup directory.
    # 

    my $filename=shift @_;
    my $retval=1;

    # We have to append the prefix ourselves since we don't use B_open_plus

    my $original_filename=$filename;

    &ActionLog("# Deleting (and backing-up) file $original_filename\n");
    &ActionLog("rm $original_filename\n");

    unless ($filename) {
	&ErrorLog("# B_delete_file called with no arguments!\n");
    }
    
    unless ($GLOBAL_LOGONLY) {
	if ( B_backup_file($original_filename) ) {
	    unless ( unlink $filename ) {
		&ErrorLog("# Couldn't unlink file $original_filename");
		$retval=0;
	    }
	}
	else {
	    $retval=0;
	    &ErrorLog("# B_delete_file did not delete $original_filename since it could not back it up\n");
	}
    }

    $retval;

}


###########################################################################
# &B_create_file ($file) creates the file $file, if it doesn't already
# exist.
# It will set a default mode of 0700 and a default uid/gid or 0/0.
#
# &B_create_file, to support Bastille's revert functionality, writes an
# rm $file command to the end of the file &getGlobal('BFILE', "created-files").
#
##########################################################################


sub B_create_file {

    my $file = $_[0];
    my $retval=1;

    # We have to create the file ourselves since we don't use B_open_plus

    my $original_file = $file;

    unless ( -e $file ) {

	unless ($GLOBAL_LOGONLY) {

	    # find the directory in which the file is to reside.
	    my $dirName = dirname($file);
	    # if the directory does not exist then
	    if(! -d $dirName) {
		# create it.
		mkpath ($dirName,0,0700);
	    }

	    $retval=open CREATE_FILE,">$file";
	    
	    if ($retval) {
		close CREATE_FILE;
		chmod 0700,$file;
		# Make the revert functionality
		&B_revert_log( &getGlobal('BIN','rm') . " $original_file \n");
	    } else {
		&ErrorLog("# Couldn't create file $original_file even though " . 
			  "it didn't already exist!\n");
	    }   
	}
	&ActionLog("# Created file $original_file\n");
    } else {
	&ActionLog("# Didn't create file $original_file since it already existed.\n");
	$retval=0;
    }

    $retval;
}

	    
###########################################################################
# &B_create_dir ($dir) creates the directory $dir, if it doesn't already
# exist.
# It will set a default mode of 0700 and a default uid/gid or 0/0.
#
# &B_create_dir, to support Bastille's revert functionality, writes an
# rmdir $file command to the end of the file &getGlobal('BFILE', "created-files").
#
##########################################################################


sub B_create_dir {

    my $dir = $_[0];
    my $retval=1;

    # We have to append the prefix ourselves since we don't use B_open_plus

    my $original_dir=$dir;

    unless ( -d $dir ) {
	unless ($GLOBAL_LOGONLY) {
	    $retval=mkdir $dir,0700;
	    
	    if ($retval) {
		# Make the revert functionality
		&B_revert_log (&getGlobal('BIN','rmdir') . " $original_dir\n");
	    }
	    else {
		&ErrorLog("# Couldn't create dir $original_dir even though it didn't already exist!");
	    }
	    
	}
	&ActionLog("# Created directory $original_dir\n");
    }
    else {
	&ActionLog("# Didn't create directory $original_dir since it already existed.\n");
	$retval=0;
    }

    $retval;
}
		

###########################################################################
# &B_print ($handle,@list) prints the items of @list to the filehandle
# $handle.  It logs the action and respects the $GLOBAL_LOGONLY variable.
#
###########################################################################

sub B_print {
   my $handle=shift @_;

   my $result=1;

   unless ($GLOBAL_LOGONLY) {
       $result=print $handle @_;
   }

   ($handle) = "$handle" =~ /[^:]+::[^:]+::([^:]+)/;

   $result;
}


###########################################################################
# &B_remove_suid ($file) removes the suid bit from $file if it
# is set and the file exist. If you would like to remove the suid bit
# from /bin/ping then you need to use:
# 
#                 &B_remove_suid("/bin/ping");
#
# &B_remove_suid respects GLOBAL_LOGONLY.
# &B_remove_suid uses &B_chmod to make the permission changes
# &B_remove_suid allows for globbing.  tyler_e
#
###########################################################################

sub B_remove_suid($) {
   my $file_expr = $_[0];

   &ActionLog("Removing SUID bit from \"$file_expr\".");
   unless ($GLOBAL_LOGONLY) {
     &SanitizeEnv;
     my @files = glob($file_expr);

     foreach my $file (@files) {
# check file existance
	 if(-e $file){
# stat current file to get raw permissions
	    my $old_perm_raw = (stat $file)[2];
	    # test to see if suidbit is set
	    my $suid_bit = (($old_perm_raw/2048) % 2);
	    if($suid_bit == 1){
		# new permission without the suid bit
		my $new_perm = ((($old_perm_raw/512) % 8 ) - 4) . 
		    (($old_perm_raw/64) % 8 ) . 
			(($old_perm_raw/8) % 8 ) . 
			    (($old_perm_raw) % 8 );
		if(&B_chmod(oct($new_perm), $file)){
		    &ActionLog("Removed SUID bit from \"$file\".");
		}
		else {
		    &ErrorLog("Could not remove SUID bit from \"$file\".");
		}
	    } # No action if SUID bit is not set
	}# No action if file does not exist
      }# Repeat for each file in the file glob
    } # unless Global_log
}
    

###########################################################################
# &B_chmod_if_exists ($mode, $file) sets the mode of $file to $mode *if*
# $file exists.  $mode must be stored in octal, so if you want to give 
# mode 700 to /etc/aliases, you need to use:
#
#                 &B_chmod_if_exists ( 0700 , "/etc/aliases");
#
# where the 0700 denotes "octal 7-0-0".
#
# &B_chmod_if_exists respects GLOBAL_LOGONLY and uses 
# &B_revert_log to reset the permissions of the file.
#
# B_chmod_if_exists allow for globbing now, as of 1.2.0.  JJB
#
##########################################################################


sub B_chmod_if_exists {
   my ($new_perm,$file_expr)=@_;
   # If $file_expr has a glob character, pass it on (B_chmod won't complain
   # about nonexistent files if given a glob pattern)
   if ( $file_expr =~ /[\*\[\{]/ ) {   # } just to match open brace for vi
       &ActionLog("Running chmod $new_perm $file_expr"); 
       return(&B_chmod($new_perm,$file_expr));
   }
   # otherwise, test for file existence
   if ( -e $file_expr ) { 
       &ActionLog("Running chmod $new_perm $file_expr"); 
       return(&B_chmod($new_perm,$file_expr)); 
   }
}


###########################################################################
# &B_chmod ($mode, $file) sets the mode of $file to $mode.  $mode must
# be stored in octal, so if you want to give mode 700 to /etc/aliases,
# you need to use:
#
#                 &B_chmod ( 0700 , "/etc/aliases");
#
# where the 0700 denotes "octal 7-0-0".
#
# &B_chmod respects GLOBAL_LOGONLY and uses 
# &B_revert_log used to insert a shell command that will return
#         the permissions to the pre-Bastille state.
#
# B_chmod allow for globbing now, as of 1.2.0.  JJB
#
##########################################################################


sub B_chmod {
   my ($new_perm,$file_expr)=@_;
   my $old_perm;
   my $old_perm_raw;
   my $new_perm_formatted;

   my $retval=1;

   my $file;

   my @files = glob ($file_expr);

   foreach $file (@files) {

       # Prepend global prefix, but save the original filename for B_backup_file
       my $original_file=$file;
       
       # Store the old permissions so that we can log them.
       $old_perm_raw=(stat $file)[2];   
       $old_perm= (($old_perm_raw/512) % 8) . 
	   (($old_perm_raw/64) % 8) .
	       (($old_perm_raw/8) % 8) . 
		   ($old_perm_raw % 8);
       
       # formating for simple long octal output of the permissions in string form
       $new_perm_formatted=sprintf "%5lo",$new_perm;
   
       &ActionLog ("# change permissions on $original_file from $old_perm to $new_perm_formatted\n");
       
       &ActionLog( "chmod $new_perm_formatted,\"$original_file\";\n");
       
       # Change the permissions on the file
       
       if ( -e $file ) {
	   unless ($GLOBAL_LOGONLY) {
	       $retval=chmod $new_perm,$file;
	       if($retval){
		   # if the distrobution is HP-UX then the modifications should
		   # also be made to the IPD (installed product database)
		   if(&GetDistro =~ "^HP-UX"){
		       &B_swmodify($file);
		   }
		   # making changes revert-able
		   &B_revert_log(&getGlobal('BIN', "chmod") . " $old_perm $file\n");
	       }
	   }
	   unless ($retval) {
	       &ActionLog("#ERROR: couldn't change permissions on $original_file from $old_perm to $new_perm_formatted\n");
	       &ErrorLog("#ERROR: couldn't change permissions on $original_file from $old_perm to $new_perm_formatted\n");
	       
	       $retval=0;
	   }				       
       }
       else {
	   &ActionLog( "#ERROR: chmod: File $original_file doesn't exist!\n");
	   &ErrorLog( "#ERROR: chmod: File $original_file doesn't exist!\n");	 
	   $retval=0;
       }
   }

   $retval;

}


###########################################################################
# &B_chown ($uid, $file) sets the owner of $file to $uid, like this:
#
#                 &B_chown ( 0 , "/etc/aliases");
#
# &B_chown respects $GLOBAL_LOGONLY  and uses 
# &B_revert_log to insert a shell command that will return
#         the file/directory owner to the pre-Bastille state.
#
# Unlike Perl, we've broken the chown function into B_chown/B_chgrp to
# make error checking simpler.
#
# As of 1.2.0, this now supports file globbing. JJB
#
##########################################################################


sub B_chown {
   my ($newown,$file_expr)=@_;
   my $oldown;
   my $oldgown;

   my $retval=1;
   
   my $file;
   &SanitizeEnv;
   my @files = glob($file_expr);

   foreach $file (@files) {

       # Prepend prefix, but save original filename 
       my $original_file=$file;
       
       $oldown=(stat $file)[4];
       $oldgown=(stat $file)[5];
       
       &ActionLog ("# change ownership on $original_file from $oldown to $newown\n");
       &ActionLog ("chown $newown,$oldgown,\"$original_file\";\n");
       if ( -e $file ) {
	   unless ($GLOBAL_LOGONLY) {
	       # changing the files owner using perl chown function
	       $retval = chown $newown,$oldgown,$file;
	       if($retval){
		   # if the distrobution is HP-UX then the modifications should
		   # also be made to the IPD (installed product database)
		   if(&GetDistro =~ "^HP-UX"){
		       &B_swmodify($file);
		   }
		   # making ownership change revert-able
		   &B_revert_log(&getGlobal('BIN', "chown") . " $oldown $file\n");
	       }	       
	   }
	   unless ($retval) {
	       &ActionLog( "#ERROR: couldn't change ownership to $newown on file $original_file\n");
	       &ErrorLog("#ERROR: couldn't change ownership to $newown on file $original_file\n");
	   }
       }
       else {
	   &ActionLog("#ERROR: chown: File $original_file doesn't exist!\n");
	   &ErrorLog("#ERROR: chown: File $original_file doesn't exist!\n");
	   $retval=0;
       }
   }
   
   $retval;
}




###########################################################################
# &B_chgrp ($gid, $file) sets the group owner of $file to $gid, like this:
#
#                 &B_chgrp ( 0 , "/etc/aliases");
#
# &B_chgrp respects $GLOBAL_LOGONLY  and uses 
# &B_revert_log to insert a shell command that will return
#         the file/directory group to the pre-Bastille state.
#
# Unlike Perl, we've broken the chown function into B_chown/B_chgrp to
# make error checking simpler.
#
# As of 1.2.0, this now supports file globbing.  JJB
#
##########################################################################


sub B_chgrp {
   my ($newgown,$file_expr)=@_;
   my $oldown;
   my $oldgown;

   my $retval=1;

   my $file;
   &SanitizeEnv;
   my @files = glob($file_expr);
   
   foreach $file (@files) {
   
       # Prepend global prefix, but save original filename for &B_backup_file
       my $original_file=$file;
       
       $oldown=(stat $file)[4];
       $oldgown=(stat $file)[5];
       
       &ActionLog( "# change group ownership on $original_file from $oldgown to $newgown\n");
       &ActionLog( "chown $oldown,$newgown,\"$original_file\";\n");
       if ( -e $file ) {
	   unless ($GLOBAL_LOGONLY) {
	       # changing the group for the file/directory
	       $retval = chown $oldown,$newgown,$file;
	       if($retval){
		   # if the distrobution is HP-UX then the modifications should
		   # also be made to the IPD (installed product database)
		   if(&GetDistro =~ "^HP-UX"){
		       &B_swmodify($file);
		   }
		   &B_revert_log(&getGlobal('BIN', "chgrp") . " $oldgown $file\n");
	       }	       
	   }
	   unless ($retval) {
	       &ActionLog("#ERROR: couldn't change ownership to $newgown on file $original_file\n");
	       &ErrorLog("#ERROR: couldn't change ownership to $newgown on file $original_file\n");	    
	   }
       }
       else {
	   &ActionLog( "#ERROR: chgrp: File $original_file doesn't exist!\n");
	   &ErrorLog("#ERROR: chgrp: File $original_file doesn't exist!\n");
	   $retval=0;
       }
   }

   $retval;
}


###########################################################################
# &B_symlink ($original_file,$new_symlink) creates a symbolic link from
# $original_file to $new_symlink.
#
# &B_symlink respects $GLOBAL_LOGONLY.  It supports
# the revert functionality that you've come to know and love by adding every
# symbolic link it creates to &getGlobal('BFILE', "created-symlinks"), currently set to:
#
#         /root/Bastille/revert/revert-created-symlinks
#
# The revert script, if it works like I think it should, will run this file,
# which should be a script or rm's...
#
##########################################################################

sub B_symlink {
    my ($source_file,$new_symlink)=@_;
    my $retval=1;
    my $original_source = $source_file;
    my $original_symlink = $new_symlink;
    
    unless ($GLOBAL_LOGONLY) {
	$retval=symlink $source_file,$new_symlink;
	if ($retval) {
	    &B_revert_log (&getGlobal('BIN',"rm") .  " $original_symlink\n");
	}
    }

    &ActionLog( "# created a symbolic link called $original_symlink from $original_source\n");
    &ActionLog( "symlink \"$original_source\",\"$original_symlink\";\n");
    unless ($retval) { 
        &ActionLog("#ERROR: couldn't symlink $original_symlink -> $original_source\n");
	&ErrorLog("#ERROR: couldn't symlink $original_symlink -> $original_source\n");
    }

    $retval;

}

###########################################################################
# B_Display was the display function used by IPCHAINS in v1.0.0-v1.0.3.
# Its use should be fully deprecated in 1.1.
#
###########################################################################

sub B_Display {
	# routine to display a long string, automatically wrapping
	# each line to no more than 80 characters
	my ( $text) = @_;
	my ($chunk1, $chunk2, $chunk3);
	while ( length($text) >= 80 ) {
		# pull the first 80 chars off the string
		$text =~ s/^(.{80})//s;
		$chunk1 = $1;
		if ( $chunk1 =~ /^(.*?\n)(.*?)$/s ) {
			$chunk2 = $1; $chunk3 = $2;
			# explicit line endings should be respected
			print $chunk2;
			$text = $chunk3 . $text;
		} elsif ( $chunk1 =~ /\s/ ) {
			# there's a space, grab everything up to the _last_ space
			$chunk1 =~ /^(.*)\s([^\s]*?)$/s;
			$chunk2 = $1; $chunk3 = $2;
			# print the stuff up to the last space
			print "$chunk2\n";
			# put the remaining data back at the front of $text
			$text = $chunk3 . $text;
		} else {
			# no space, just print
			print "$chunk1\n";
		}
	}
	if ( $text =~ /\S/ ) {
		# some non-whitespace data, print it
		print "$text\n";
	}
}


###########################################################################
# &B_chkconfig_on ($daemon_name) creates the symbolic links that are
# named in the "# chkconfig: ___ _ _ " portion of the init.d files.  We
# need this utility, in place of the distro's chkconfig, because of both
# our need to add revert functionality and our need to harden distros that
# are not mounted on /.
#
# It uses the following global variables to find the links and the init
# scripts, respectively:
#
#   &getGlobal('DIR', "rcd")    -- directory where the rc_.d subdirs can be found
#   &getGlobal('DIR', "initd")  -- directory the rc_.d directories link to
#
# Here an example of where you might use this:
#
# You'd like to tell the system to run the firewall at boot:
#       B_chkconfig_on("bastille-firewall")
#
###########################################################################

# PW: Blech. Copied B_chkconfig_off() and changed a few things,
#		then changed a few more things....

sub B_chkconfig_on {

    my $startup_script=$_[0];
    my $retval=1;

    my $chkconfig_line;
    my ($runlevelinfo,@runlevels);
    my ($start_order,$stop_order,$filetolink);

    &ActionLog("# chkconfig_on enabling $startup_script\n");
    
    # In Debian system there is no chkconfig script, runlevels are checked
    # one by one (jfs)
    if (&GetDistro =~/^DB.*/) {
	    $filetolink = &getGlobal('DIR', "initd") . "/$startup_script";
	    if (-x $filetolink)
	    {
		    foreach my $level ("0","1","2","3","4","5","6" ) {
			    my $link = '';
			    $link = &getGlobal('DIR', "rcd") . "/rc" . "$level" . ".d/K50" . "$startup_script";
			    $retval=symlink($filetolink,$link);
		    }
	    }
	    return $retval;
    }
    # Run through the init script looking for the chkconfig line...
    $retval = open CHKCONFIG,&getGlobal('DIR', "initd") . "/$startup_script";
    unless ($retval) {
	&ActionLog("# Didn't chkconfig_on $startup_script because we couldn't open " . &getGlobal('DIR', "initd") . "/$startup_script\n");
    }
    else {

      READ_LOOP:
	while (my $line=<CHKCONFIG>) {

	    # We're looking for lines like this one:
	    #      # chkconfig: 2345 10 90
	    # OR this
	    #      # chkconfig: - 10 90
	    
	    if ($line =~ /^#\s*chkconfig:\s*([-\d]+)\s*(\d+)\s*(\d+)/ ) {
		$runlevelinfo = $1;
		$start_order = $2;
		$stop_order = $3;
		# handle a runlevels arg of '-'
		if ( $runlevelinfo eq '-' ) {
		    &ActionLog("# chkconfig_on saw '-' for runlevels for \"$startup_script\", is defaulting to levels 3,4,5\n");
		    $runlevelinfo = '345';
		}
		@runlevels = split(//,$runlevelinfo);
		# make sure the orders have 2 digits
		$start_order =~ s/^(\d)$/0$1/;
		$stop_order =~ s/^(\d)$/0$1/;
		last READ_LOOP;
	    }
	}
	close CHKCONFIG;

	# Do we have what we need?
	if ( (scalar(@runlevels) < 1) || (! $start_order =~ /^\d{2}$/) || (! $stop_order =~ /^\d{2}$/) ) {
		# problem
		&ErrorLog("# B_chkconfig_on $startup_script failed -- no valid runlevel/start/stop info found\n");
		return(-1);
	}

	# Now, run through creating symlinks...
	&ActionLog("# chkconfig_on will use runlevels ".join(",",@runlevels)." for \"$startup_script\" with S order $start_order and K order $stop_order\n");
	
	$retval=0;
	# BUG: we really ought to readdir() on &getGlobal('DIR', "rcd") to get all levels
	foreach my $level ( "0","1","2","3","4","5","6" ) {
		my $link = '';
		# we make K links in runlevels not specified in the chkconfig line
	    	$link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/K$stop_order" . $startup_script;
		my $klink = $link;
		# now we see if this is a specified runlevel; if so, make an S link
		foreach my $markedlevel ( @runlevels ) {
			if ( $level == $markedlevel) {
	    			$link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/S$start_order" . $startup_script;
			}
		}
	    	my $target = &getGlobal('DIR', "initd") ."/" . $startup_script;
	    	my $local_return;

		if ( (-e "$klink") && ($klink ne $link) ) {
		    # there's a K link, but this level needs an S link
		    unless ($GLOBAL_LOGONLY) {
			$local_return = unlink("$klink");
			if ( ! local_return ) {
			    # unlinking old, bad $klink failed
			    &ErrorLog("# Unlinking $klink failed\n");
			} else {
			    &ActionLog("# Removed link $klink\n");
			    # If we removed the link, add a link command to the revert file
			    &B_revert_log (&getGlobal('BIN','ln') . " -s $target $klink\n");
			} # close what to do if unlink works	
		    }	# if not GLOBAL_LOGONLY
		}	# if $klink exists and ne $link
	   	
		# OK, we've disposed of any old K links, make what we need 
	    	if ( (! ( -e "$link" )) && ($link ne '') ) {
		    # link doesn't exist and the start/stop number is OK; make it
		    unless ($GLOBAL_LOGONLY) {
			# create the link
			$local_return = &B_symlink($target,$link);
			if ($local_return) {
			    $retval++;
			    &ActionLog("# Created link $link\n");
			} else {
			    &ErrorLog("Couldn't create $link when trying to chkconfig on $startup_script\n");
			}
		    }
		    
		} # link doesn't exist
	    } # foreach level
	
    }

    if ($retval < @runlevels) {
	$retval=0;
    }
    
    $retval;

}


###########################################################################
# &B_chkconfig_off ($daemon_name) deletes the symbolic links that are
# named in the "# chkconfig: ___ _ _ " portion of the init.d files.  We
# need this utility, in place of the distro's chkconfig, because of both
# our need to add revert functionality and our need to harden distros that
# are not mounted on /.
#
# chkconfig allows for a REVERT of its work by writing to an executable
# file &getGlobal('BFILE', "removed-symlinks").
#
# It uses the following global variables to find the links and the init
# scripts, respectively:
#
#   &getGlobal('DIR', "rcd")    -- directory where the rc_.d subdirs can be found
#   &getGlobal('DIR', "initd")  -- directory the rc_.d directories link to
#
# Here an example of where you might use this:
#
# You'd like to tell stop running sendmail in daemon mode on boot:
#       B_chkconfig_off("sendmail")
#
###########################################################################



sub B_chkconfig_off {

    my $startup_script=$_[0];
    my $retval=1;

    my $chkconfig_line;
    my @runlevels;
    my ($start_order,$stop_order,$filetolink);

    if (&GetDistro =~/^DB.*/) {
	    $filetolink = &getGlobal('DIR', "initd") . "/$startup_script";
	    if (-x $filetolink)
	    {
		    # Three ways to do this in Debian:
		    # 1.- have the initd script set to 600 mode
		    # 2.- Remove the links in rcd (re-installing the package
		    # will break it)
		    # 3.- Use update-rc.d --remove (same as 2.)
		    # (jfs) 
		    &B_chmod(0600,$filetolink);
		    $retval=6;
		    
		    # The second option
		    #foreach my $level ("0","1","2","3","4","5","6" ) {
		    #my $link = '';
		    #$link = &getGlobal('DIR', "rcd") . "/rc" . "$level" . ".d/K50" . "$startup_script"; 
		    #unlink($link);
		    #}
	    }
    }
    else {

	    # Run through the init script looking for the chkconfig line...


	    $retval = open CHKCONFIG,&getGlobal('DIR', "initd") . "/$startup_script";
	    unless ($retval) {
		    &ActionLog("Didn't chkconfig_off $startup_script because we couldn't open " . &getGlobal('DIR', "initd") . "/$startup_script\n");
	    }
	    else {

		    READ_LOOP:
		    while (my $line=<CHKCONFIG>) {

			    # We're looking for lines like this one:
			    #      # chkconfig: 2345 10 90

			    if ($line =~ /^#\s*chkconfig:\s*([-\d]+)\s*(\d+)\s*(\d+)/ ) {
				    @runlevels=split //,$1;
				    $start_order=$2;
				    $stop_order=$3;


				    # Change single digit runlevels to double digit -- otherwise,
				    # the alphabetic ordering chkconfig depends on fails.
				    if ($start_order =~ /^\d$/ ) {
					    $start_order = "0" . $start_order;
					    &ActionLog("# chkconfig_off converted start order to $start_order\n");
				    }
				    if ($stop_order =~ /^\d$/ ) {
					    $stop_order = "0" . $stop_order;
					    &ActionLog("# chkconfig_off converted stop order to $stop_order\n");
				    }

				    last READ_LOOP;
			    }
		    }
		    close CHKCONFIG;

		    # If we never found a chkconfig line, can we just run through all 5 
		    # rcX.d dirs from 1 to 5...?

		    # unless ( $start_order and $stop_order ) {
		    #	 @runlevels=("1","2","3","4","5");
		    #	 $start_order = "*"; $stop_order="*";
		    # }

		    # Now, run through removing symlinks...



		    $retval=0;

		    # Handle the special case that the runlevel specified is solely "-"
		    if ($runlevels[0] =~ /-/) {
			    @runlevels = ( "0","1","2","3","4","5","6" );
		    }

		    foreach $level ( @runlevels ) {
			    my $link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/S$start_order" . $startup_script;
			    my $new_link = &getGlobal('DIR', "rcd") . "/rc" . $level . ".d/K$stop_order" . $startup_script;
			    my $target = &getGlobal('DIR', "initd") ."/" . $startup_script;
			    my $local_return;


			    # Replace the S__ link in this level with a K__ link.
			    if ( -e $link ) {
				    unless ($GLOBAL_LOGONLY) {
					    $local_return=unlink $link;
					    if ($local_return) {
						    $local_return=symlink $target,$new_link;
						    unless ($local_return) {
							    &ErrorLog("# Linking $target to $new_link failed.\n");
						    }
					    }
					    else {  # unlinking failed
						    &ErrorLog("# Unlinking $link failed\n");
					    }

				    }
				    if ($local_return) {
					    $retval++;
					    &ActionLog("# Removed link $link\n");

					    #
					    # If we removed the link, add a link command to the revert file
					    # Write out the revert information for recreating the S__
					    # symlink and deleting the K__ symlink.
					    &B_revert_log(&getGlobal('BIN',"ln") . " -s $target $link\n");
					    &B_revert_log(&getGlobal('BIN',"rm") . " -f $new_link\n");
				    }
				    else {
					    &ErrorLog("# B_chkconfig_off $startup_script failed\n");
				    }

			    }
		    } # foreach

	    } # else-unless

    } # else-DB
    if ($retval < @runlevels) {
	    $retval=0;
    }

    $retval;

}


############################################################################
# &B_cp is the Bastille cp command, which is based on Perl's File::cp.
# &B_cp($source,$target).  It is somewhat strange, to make the Backup and
# revert functions easier to implement, in that:
#
#
#     It can ONLY copy from one file to another! Both $source and
#     $target must be files, not directories!!!!
#
# It respects $GLOBAL_LOGONLY. 
# If $target is an already-existing file, it is backed up.
#
# revert either appends another "rm $target" to &getGlobal('BFILE', "revert-actions")  or
# backs up the file that _was_ there into the &getGlobal('BDIR', "backup").
#
############################################################################

sub B_cp {

    my ($source,$target)=@_;
    my $retval=0;

    my $had_to_backup_target=0;

    use File::Copy;

    my $original_source=$source;
    my $original_target=$target;

    unless ($GLOBAL_LOGONLY) {
	unless ( -e $target and -f $target ) {
	    &B_backup_file($original_target);
	    &ActionLog("About to copy $original_source to $original_target -- had to backup target\n");
	    $had_to_backup_target=1;
	}

	$retval=copy($source,$target);
	if ($retval) {
	    &ActionLog("cp $original_source $original_target\n");
	    
	    #
	    # We want to add a line to the &getGlobal('BFILE', "created-files") so that the
	    # file we just put at $original_target gets deleted.
	    #
	    &B_revert_log(&getGlobal('BIN',"rm") . " $original_target\n");
	} else {
	    &ErrorLog("Failed to copy $original_source to $original_target\n");
	}
    }
    $retval;
}



############################################################################
# &B_place puts a file in place, using Perl's File::cp.  This file is taken
# from &getGlobal('BDIR', "home") and is used to place a file that came with
# Bastille.  
#
# This should be DEPRECATED in favor of &B_cp, since the only reason it exists
# is because of GLOBAL_PREFIX, which has been broken for quite some time.
# Otherwise, the two routines are identical.
#
# It respects $GLOBAL_LOGONLY.
# If $target is an already-existing file, it is backed up.
#
# revert either appends another "rm $target" to &getGlobal('BFILE', "revert-actions")  or
# backs up the file that _was_ there into the &getGlobal('BDIR', "backup"),
# appending a "mv" to revert-actions to put it back.
#
# UPDATE:  
#         20010218 - changed the location to /usr/share/Bastille because of
#                    the file location update that MandrakeSoft suggests.
#
############################################################################

sub B_place {

    my ($source,$target)=@_;
    my $retval=0;

    my $had_to_backup_target=0;

    use File::Copy;

    my $original_source=$source;
    $source  = &getGlobal('BDIR', "share") . $source;
  #  $source = "/usr/share/Bastille" . $source; #removed hard-coded path
    my $original_target=$target;

    unless ($GLOBAL_LOGONLY) {
	if ( -e $target and -f $target ) {
	    &B_backup_file($original_target);
	    &ActionLog("About to copy $original_source to $original_target -- had to backup target\n");
	    $had_to_backup_target=1;
	}
	$retval=copy($source,$target);
	if ($retval) {
	    &ActionLog("placed file $original_source  as  $original_target\n");   
	    #
	    # We want to add a line to the &getGlobal('BFILE', "created-files") so that the
	    # file we just put at $original_target gets deleted.
	    &B_revert_log(&getGlobal('BIN',"rm") . " $original_target\n");
	} else {
	    &ErrorLog("Failed to place $original_source as $original_target\n");
	}
    }

    $retval;
}





#############################################################################
#############################################################################
#############################################################################

###########################################################################
# &B_mknod ($file) creates the node $file, if it doesn't already
# exist.  It uses the prefix and suffix, like this:
#
#            mknod $prefix $file $suffix
#
# This is just a wrapper to the mknod program, which tries to introduce
# revert functionality, by writing    rm $file     to the end of the 
# file &getGlobal('BFILE', "created-files").
#
##########################################################################


sub B_mknod {

    my ($prefix,$file,$suffix) = @_;
    my $retval=1;

    # We have to create the filename ourselves since we don't use B_open_plus

    my $original_file = $file;

    unless ( -e $file ) {
	unless ($GLOBAL_LOGONLY) {

	    my $command = &getGlobal("BIN","mknod") . " $prefix $file $suffix";

	    if ( system($command) == 0) {
		# Since system will return 0 on success, invert the error code
		$retval=1;
	    }
	    else {
		$retval=0;
	    }

	    if ($retval) {

		# Make the revert functionality
		&B_revert_log(&getGlobal('BIN',"rm") . " $original_file\n");
	    } else {
		&ErrorLog("# Couldn't mknod $prefix $original_file $suffix even though it didn't already exist!\n");
	    }
	    
	}
	&ActionLog("# mknod $prefix $original_file $suffix\n");
    }
    else {
	&ActionLog("# Didn't mknod $prefix $original_file $suffix since $original_file already existed.\n");
	$retval=0;
    }
    
    $retval;
}

###########################################################################
# &B_revert_log("reverse_command") prepends a command to a shell script.  This 
# shell script is intended to be run by bastille -r to reverse the changes that 
# Bastille made, returning the files which Bastille changed to their original 
# state.
###########################################################################

sub B_revert_log($) {
   
    my $revert_command = $_[0];
    my $revert_actions = &getGlobal('BFILE', "revert-actions");
    my @lines;
    

    if (! (-e $revert_actions)) {
	if (open REVERT_ACTIONS,">" . $revert_actions){ # create revert file
	    close REVERT_ACTIONS; # chown to root, rwx------
	    chmod 0700,$revert_actions;
	    chown 0,0,$revert_actions;
	}
	else { 
	    &ErrorLog ("ERROR:   Can not create revert-actions file.\n" .
		       "         Unable to add the following command to the revert\n" . 
		       "         actions script:\n"  .
		       "           $revert_command\n");
	    exit(1);
	}
	
    }
	
    &B_open_plus (*REVERT_NEW, *REVERT_OLD, $revert_actions); 

    while (my $line=<REVERT_OLD>) { #copy file into @lines
	push (@lines,$line);
    }
    print REVERT_NEW $revert_command .  "\n";  #make the revert command first in the new file
    while (my $line = shift @lines) { #write the rest of the lines of the file
	print REVERT_NEW $line;
    }
    close REVERT_OLD;
    close REVERT_NEW;
    if (rename "${revert_actions}.bastille", $revert_actions) { #replace the old file with the new file we
	chmod 0700,$revert_actions;                # just made / mirrors B_close_plus logic
	chown 0,0,$revert_actions;
    } else {
	&ErrorLog("B_revert_log: not able to move ${revert_actions}.bastille to ${revert_actions}!!! $!) !!!\n");
    }
}


###########################################################################
# &getGlobalConfig($$)
#
# returns the requested GLOBAL_CONFIG hash value, ignoring the error 
# if the value does not exist (because every module uses this to find
# out if the question was answered "Y")
###########################################################################
sub getGlobalConfig ($$) {
  my $module = $_[0];
  my $key = $_[1];
  if (exists $GLOBAL_CONFIG{$module}{$key}) {
    my $answer=$GLOBAL_CONFIG{$module}{$key};
    &ActionLog("Answer to question $module.$key is \"$answer\".\n");
    return $answer;
  } else {
    &ActionLog("Answer to question $module.$key is undefined, do not\n" . 
	       "implement question\n");
    return undef;
  }
}

###########################################################################
# &getGlobal($$)
#
# returns the requested GLOBAL_* hash value, and logs an error 
# if the variable does not exist.
###########################################################################
sub getGlobal ($$) {
  my $type = uc($_[0]);
  my $key = $_[1];

  # define a mapping from the first argument to the proper hash
  my %map = ("BIN"   => \%GLOBAL_BIN,
             "FILE"  => \%GLOBAL_FILE,
             "BFILE" => \%GLOBAL_BFILE,
             "DIR"   => \%GLOBAL_DIR,
             "BDIR"  => \%GLOBAL_BDIR,
	     "ERROR" => \%GLOBAL_ERROR,
	     "SERVICE" => \%GLOBAL_SERVICE,
	     "SERVTYPE" => \%GLOBAL_SERVTYPE,
	     "PROCESS" => \%GLOBAL_PROCESS,
            );

  # check to see if the desired key is in the desired hash
  if (exists $map{$type}->{$key}) {
    # get the value from the right hash with the key
    return $map{$type}->{$key};
  } else {
    # i.e. Bastille tried to use $GLOBAL_BIN{'cp'} but it does not exist.
    &ErrorLog("Bastille tried to use \$GLOBAL_${type}\{\'$key\'} but it does not exist.\n");
    return undef;
  }
}


###########################################################################
# &showDisclaimer:
# Print the disclaimer and wait for 2 minutes for acceptance
# Do NOT do so if any of the following conditions hold
# 1. the -n option was used
# 2. the file ~/.spc_disclaimer exists
###########################################################################

sub showDisclaimer($) {

# Get passwd information on the effective user
    my $nodisclaim = $_[0];
    my $nodisclaim_file = &getGlobal('BFILE', "nodisclaimer");
    my $response;
    my $WAIT_TIME = 300; # we'll wait for 5 minutes
    my $developersAnd;
    my $developersOr;
    if ($GLOBAL_OS =~ "^HP-UX") {
	$developersAnd ="HP AND ITS";
	$developersOr ="HP OR ITS";
    }else{
	$developersAnd ="JAY BEALE, THE BASTILLE DEVELOPERS, AND THEIR";
	$developersOr ="JAY BEALE, THE BASTILLE DEVELOPERS, OR THEIR";
    }	
    my $DISCLAIMER =
	"\n" .
        "Copyright (C) 1999-2002 Jay Beale\n" .
        "Copyright (C) 1999-2001 Peter Watkins\n" .
        "Copyright (C) 2000 Paul L. Allen\n" .
        "Copyright (C) 2001-2003 Hewlett Packard Company\n" .
        "Bastille is free software; you are welcome to redistribute it under\n" .
        "certain conditions.  See the \'COPYING\' file in your distribution for terms.\n\n" .
	"DISCLAIMER.  Use of Bastille can help optimize system security, but does not\n" .
	"guarantee system security. Information about security obtained through use of\n" .
	"Bastille is provided on an AS-IS basis only and is subject to change without\n" .
	"notice. Customer acknowledges they are responsible for their system\'s security.\n" .
	"TO THE EXTENT ALLOWED BY LOCAL LAW, Bastille (\"SOFTWARE\") IS PROVIDED TO YOU \n" .
	"\"AS IS\" WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, WHETHER ORAL OR WRITTEN,\n" .
	"EXPRESS OR IMPLIED.  $developersAnd SUPPLIERS\n" .
	"DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE \n" .
	"IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.\n" . 
	"Some countries, states and provinces do not allow exclusions of implied\n" .
	"warranties or conditions, so the above exclusion may not apply to you. You may\n" .
	"have other rights that vary from country to country, state to state, or province\n" .
	"to province.  EXCEPT TO THE EXTENT PROHIBITED BY LOCAL LAW, IN NO EVENT WILL\n" .
	"$developersOr SUBSIDIARIES, AFFILIATES OR\n" .
	"SUPPLIERS BE LIABLE FOR DIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL OR OTHER\n" .
	"DAMAGES (INCLUDING LOST PROFIT, LOST DATA, OR DOWNTIME COSTS), ARISING OUT OF\n" .
	"THE USE, INABILITY TO USE, OR THE RESULTS OF USE OF THE SOFTWARE, WHETHER BASED\n" .
	"IN WARRANTY, CONTRACT, TORT OR OTHER LEGAL THEORY, AND WHETHER OR NOT ADVISED\n" .
	"OF THE POSSIBILITY OF SUCH DAMAGES. Your use of the Software is entirely at your\n" .
	"own risk. Should the Software prove defective, you assume the entire cost of all\n" .
	"service, repair or correction. Some countries, states and provinces do not allow\n" .
	"the exclusion or limitation of liability for incidental or consequential \n" .
	"damages, so the above limitation may not apply to you.\n";
    
# once we prompt the user, we'll wait for $WAIT_TIME seconds for a response
# if no response is received, we exit nicely -- we catch the SIGALRM
    $SIG{'ALRM'} = sub {
	&ErrorLog(
		  "\n\nERROR:   Waited for $WAIT_TIME seconds. No response received.\n" .
		  "             Quitting.\n" );
	exit 1;
    };

# If the user has specified not to show the disclaimer, or
# the .spc_disclaimer file already exists, then return
    if( ( $nodisclaim ) || -e $nodisclaim_file ) { return 1; }

# otherwise, show the disclaimer
    my $echo = &getGlobal('BIN', "echo");
    my $more = &getGlobal('BIN', "more");
    system "$echo \"$DISCLAIMER\" | $more";
    alarm $WAIT_TIME; # start alarm
    print("You must accept the terms of this disclaimer to use\n" .
	  "Bastille.  Type \"accept\" (without quotes) within 5\n" .
	  "minutes to accept the terms of the above disclaimer\n" .  "> " );

    chop( $response = <STDIN> ); # Script blocks on STDIN here
    alarm 0; # turn off alarm immediately after getting line
 
# there is a response
    if( lc( $response ) =~ "accept" ) {
	my $touch = &getGlobal('BIN', "touch");
	my $retVal = system("$touch $nodisclaim_file");
	if( $retVal != 0 ) {
	    &ErrorLog ( &getGlobal('ERROR','disclaimer'));
	} # if
	else {
	    print("This disclaimer will not appear again on this machine.\n" .
		  "To suppress the disclaimer on other machines, use Bastille\'s\n" .
		  "-n flag (example: bastille -n).\n");
	} # else
    } # outer if
    else { # something besides "accept" was typed
	print("You must accept the terms of the disclaimer before using\n" .
	      "Bastille.  Exiting.\n" );
	exit 0;
    } # else
} # showDisclaimer

1;
