# Copyright (C) 1999, 2000 Jay Beale
# Licensed under the GNU General Public License

package Bastille::BootSecurity;
use lib "/usr/lib";

use Bastille::API;


# Expected tough bug area:  ProtectLILO!!!! (Especially floppy work)


#######################################################################
##                Protecting LILO and single user mode               ##
#######################################################################

&ProtectLILO;
&ProtectGRUB;
&SecureInittab;
&DisableAutologin;

sub ProtectLILO {

    ## ProtectLILO can password the lilo prompt so that specifying special
    ## options to the kernel at the LILO prompt requires a password.  It can
    ## also set the delay to zero so that the user/attacker has no time to
    ## type anything at the LILO prompt.
    ##
    ## Both of these protect lots of different attacks which boot the kernel
    ## under "special parameters."  One such very common practice is to boot
    ## Red Hat Linux in single user mode _without_a_password_ by typing
    ## linux single at the LILO prompt.

    my $set_lilo_delay_zero=0;
    my $lilo_password="";
    my @lilo_config_files;


    #
    # Determine what changes to make...
    # 

    # Should we set the lilo delay to zero?
    if (&getGlobalConfig("BootSecurity","lilodelay") eq "Y") {
	$set_lilo_delay_zero=1;
    }

    # Should we password protect the lilo prompt?
    if ((&getGlobalConfig("BootSecurity","protectlilo") eq "Y") and(&getGlobalConfig("BootSecurity","protectlilo_password"))) { 
	$lilo_password=&getGlobalConfig("BootSecurity","protectlilo_password");
    }



    #
    # Which media (hard disk/floppy) should we modify lilo on?
    #

    # If they want to modify the hard disk's lilo file, make sure it exists.
    if (&getGlobalConfig("BootSecurity","lilosub_drive") eq "Y") {
	if ( -e ${GLOBAL_PREFIX} . &getGlobal('FILE', "lilo.conf")  ) {
	    push @lilo_config_files,&getGlobal('FILE', "lilo.conf");
	}
	else {
	    &ErrorLog("Couldn't modify hard drive's lilo.conf -- couldn't");
	    &ErrorLog("find " . &getGlobal('FILE', "lilo.conf") . "\n");
	}
    }

    # If they want to modify the floppy disk's lilo file, mount the floppy,
    # make sure we can write to it...

    if ((&getGlobalConfig("BootSecurity","lilosub_floppy") eq "Y") and (&getGlobalConfig("BootSecurity","lilosub_writefloppy") =~ /fd(\d)/ )) {
	my $dev_number=$1;
	my $floppy_dev="/dev/fd$dev_number";
	my $floppy_file = &getGlobal('DIR', "floppy") . &getGlobal('FILE', "lilo.conf");

	# Check to see that the drive isn't already mounted...
	if ( open MTAB,&getGlobal('FILE', "mtab") ) {
	    while (my $line = <MTAB>) {
		if ($line =~ /\/dev\/fd$dev_number/ ) {
		    my $command=&getGlobal('BIN',"umount");
                    `$command $floppy_dev`;
		}
	    }
	    close MTAB;
	}

	# Now mount the drive and add it to the list of drives to lilo
	my $command=&getGlobal('BIN',"mount");
        my $floppydir=&getGlobal('DIR', "floppy");
        `$command -t ext2 $floppy_dev $floppydir`;
	if ( -e $floppy_file ) {
	    push @lilo_config_files,$floppy_file;
	}
    }

    #
    # Make the modifications to the file now...
    #

    if ( ($set_lilo_delay_zero or $lilo_password) and (@lilo_config_files)){
	
	&ActionLog("# sub ProtectLILO\n");
	
        my $file;
	
	foreach $file (@lilo_config_files) {

	    #
	    # Make the actual insertions
	    #

	    # Set the lilo delay to zero?
	    if ( $set_lilo_delay_zero ) {

		# lilo.conf man page is inaccurate on "delay" option -- if you 
		# set it to 0, it waits indefinitely; 
		# instead set to 1 (tenth of a second)

		&B_hash_comment_line($file,'^\s*timeout=');
		&B_hash_comment_line($file,'^\s*prompt');
		&B_prepend_line($file,'^\s*delay',"delay=1\n");
	    }

	    # Password protect the lilo prompt
	    if ( $lilo_password ) {

		my $line="restricted\n" . "password=" . $lilo_password . "\n";
		&B_prepend_line($file,'^\s*restricted',$line);
	    }

	    #
	    # Put good permissions on the file, especially since the lilo
	    # password needs to be hidden from view by users.
	    #
                  
	    &B_chmod(0600,"$file");
	    &B_chown(0,"$file");

	    unless ($GLOBAL_LOGONLY) {

		# Now, re-run lilo to make the configuration take effect, but
		# only if we're actually operating on our own drive/partition.
		# Basically, if $GLOBAL_PREFIX is set, than we're not operating
		# on our native boot device...

		unless ($GLOBAL_PREFIX) {
		    if (&getGlobal('BIN',"lilo")) {
			if ($file =~ /^&getGlobal('DIR', "floppy")/) {
                            my $command=&getGlobal('BIN',"lilo");
                            my $arg=&getGlobal('DIR', "floppy");
                            `$command -r $arg`;
			}
			else {
			    my $command=&getGlobal('BIN',"lilo");
                            `$command`;
			}
			&ActionLog("# Re-running lilo");
		    }
		    else {
			&ErrorLog("Couldn't re-run lilo because we couldn't find lilo!.\nPlease re-run lilo by typing:\n\tlilo   (for hard drive booting)\n\tlilo -r " . &getGlobal('DIR', "floppy") . "   (for floppy drive booting)\n");
		    }
		}
		else {
		    &ErrorLog("Didn't re-run lilo because the -r option was used...\nIn these cases, you need to rerun lilo manually when you boot from the $GLOBAL_PREFIX device/partition.\n");
		}
	    }
	}
    }
}


sub ProtectGRUB {

    ## ProtectGRUB can password the GRUB prompt so that specifying special
    ## options to the kernel at the GRUB prompt requires a password.  

    ## Both of these protect lots of different attacks which boot the kernel
    ## under "special parameters."  One such very common practice is to boot
    ## Red Hat Linux in single user mode _without_a_password_.

    my $grub_password="";
    my @grub_config_files;

    # Should we password protect the lilo prompt?
    if ((&getGlobalConfig("BootSecurity","protectgrub") eq "Y") and(&getGlobalConfig("BootSecurity","protectgrub_password"))) { 
	$grub_password=&getGlobalConfig("BootSecurity","protectgrub_password");
    }


    #
    # Which media (hard disk/floppy) should we modify lilo on?
    #

    # If they want to modify the hard disk's grub file, make sure it exists.
#    if (&getGlobalConfig("BootSecurity","lilosub_drive") eq "Y") {
	if ( -e ${GLOBAL_PREFIX} . &getGlobal('FILE', "grub.conf")  ) {
	    push @grub_config_files,&getGlobal('FILE', "grub.conf");
	}
	else {
	    &ErrorLog("Couldn't modify hard drive's grub.conf -- couldn't");
	    &ErrorLog("find " . &getGlobal('FILE', "grub.conf") . "\n");
	}
#    }

    #
    # Make the modifications to the file now...
    #

    if ( ($grub_password) and (@grub_config_files)){
	
	&ActionLog("# sub ProtectGRUB\n");
	
        my $file;
	
	foreach $file (@grub_config_files) {

	    #
	    # Make the actual insertions
	    #

	    # Password protect the grub prompt
	    open GRUBPASS,"/root/GRUB-PASSWORD";
	    print GRUBPASS "$grub_password\n";
	    close GRUBPASS;
	    my $md5sum_command = &getGlobal('BIN', "md5sum");
	    open MD5,"$md5sum_command /root/GRUB-PASSWORD |";
	    my $md5password = <MD5>;
	    close MD5;
	    my $rm_command = &getGlobal('BIN', "rm");
	    `$rm_command /root/GRUB-PASSWORD`;
	    my $line="password -md5" . $md5password . "\n";
	    &B_prepend_line($file,'password',$line);

	    #
	    # Put good permissions on the file, especially since the lilo
	    # password needs to be hidden from view by users.
	    #
                  
	    &B_chmod(0600,"$file");
	    &B_chown(0,"$file");

	}
    }
}


sub SecureInittab {

    if ((&getGlobalConfig("BootSecurity","secureinittab") eq "Y") or (&getGlobalConfig("BootSecurity","passsum") eq "Y") ) {
	&ActionLog("# sub SecureInittab\n");

	# Do we want to disable Ctrl-Alt-Del rebooting of the system via
	# the line in /etc/inittab? 
	
	if (&getGlobalConfig("BootSecurity","secureinittab") eq "Y") {
	    &B_hash_comment_line(&getGlobal('FILE', "inittab"),":ctrlaltdel:");
	}
	
	# Require a password to boot in single user mode (runlevel S/1)
	# by adding a line for /sbin/sulogin to runlevel S in /etc/inittab.
	
	# Password protect single user mode
	
	if (&getGlobalConfig("BootSecurity","passsum") eq "Y")  {
	   
	    if (&getGlobal('BIN',"sulogin")) {
		my $file=$GLOBAL_PREFIX . &getGlobal('FILE', "inittab");
		my $line_to_insert_after=":initdefault:";
		my $line_to_insert="\n~~:S:wait:" . &getGlobal('BIN',"sulogin") . "\n";
		my $pattern = "S:wait:.*sulogin";
		&B_insert_line($file,$pattern,$line_to_insert,$line_to_insert_after);
	    }
	}
    }

}

sub DisableAutologin {

    # Another msec overlap requirement.
    # Disable Mandrake's autologin feature -- bad physical security risk.
    #
    if (&getGlobalConfig("BootSecurity","disable_autologin") eq "Y") {
	&ActionLog("# sub DisableAutologin\n");

	my $file = "/etc/sysconfig/autologin";

	&B_append_line($file,'^\s*AUTOLOGIN',"AUTOLOGIN=yes\n");
	&B_replace_line($file,'^\s*AUTOLOGIN',"AUTOLOGIN=yes\n");
    }
}

1;




