<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    catalog-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Handle media catalog items and lists.                   */
/*                                                                      */
/* ******************************************************************** */
/** @package catalog */

/** indexing/searching */
include_once("search-lucene-defs.php");
include_once("search-index-defs.php");
include_once("search-query-defs.php");
/** Filesystem access */
include_once("file-defs.php");

// -----------------------------------------------------------------------
/**
* A class which encpasulates an item which can be in the catalog.
* @package catalog
*/
class catalogitem extends RenderableObject {
  var $cat_id;               // Unique catalog record key for the catalog item
  var $cat_name = "";        // Name of this catalog item
  var $cat_desc = "";        // Full description of catalog item
  var $mime_type = "";       // Mime-type of catalog item
  var $mime_category = "";   // Mime category eg: document, movie, audio etc.
  var $keywords = "";        // Any keywords associated with it
  var $category = "";        // Optional user-defined category for item
  var $filesize = "";        // Size of catalog item in bytes
  var $filepath = "";        // Website URL path to the item (with leading '/')
  var $physicalpath = "";    // Full physical path on the filesystem
  var $fileexists = false;   // True if file exists at filepath
  var $width = 0;            // Pixel width of this item
  var $height = 0;           // Pixel height of this item
  var $uploaded_ts = 0;      // Unix timestamp of datetime item was added to catalog
  var $uploaded = "";        // Datetime uploaded, NICE_FULLDATETIME format
  var $errmsg = "";          // Contains error message if invalid
  var $valid = false;        // True if item was retreived successfully
  var $newcat = false;       // True if we just created this story

  // .....................................................................
  /** Constructor
  * @param integer $id Optional unique catalog item ID
  */
  function catalogitem($id=false) {
    // The all-important catalog ID..
    $this->cat_id = $id;
    if (!$id) {
      $this->newcat = true;
      $this->valid = true;
    }
    // Further processing for existing items..
    if (!$this->newcat) {
      // Attempt to get the catlog item
      $this->get();
    }

  } // catalogitem
  // .....................................................................
  /** Get current or nominated catalog item definition from the database.
  * @param integer $id Optional unique catalog item ID to get
  */
  function get($id=false) {
    global $RESPONSE;
    $res = false;
    if ($id) $this->cat_id = $id;
    if ($this->cat_id) {
      $catQ = dbrecordset("SELECT * FROM ax_catalog WHERE cat_id=$this->cat_id");
      if ($catQ->hasdata) {
        $this->cat_name      = $catQ->field("cat_name");
        $this->cat_desc      = $catQ->field("cat_desc");
        $this->mime_type     = $catQ->field("mime_type");
        $this->mime_category = $catQ->field("mime_category");
        $this->keywords      = $catQ->field("keywords");
        $this->category      = $catQ->field("category");
        $this->filesize      = $catQ->field("filesize");
        $this->filepath      = $catQ->field("filepath");
        $this->width         = $catQ->field("width");
        $this->height        = $catQ->field("height");
        $this->uploaded_ts   = datetime_to_timestamp($catQ->field("upload_timestamp"));
        $this->uploaded      = timestamp_to_displaydate(NICE_FULLDATETIME, $this->uploaded_ts);
        // Update status of physical file..
        $this->physicalpath = $RESPONSE->site_docroot . $this->filepath;
        $this->fileexists = file_exists($this->physicalpath);
        $res = true;
        $this->newcat = false;
      }
      else $this->errmsg = "No record of catalog item: $this->cat_id";
    }
    else $this->errmsg = "No catalog ID was given";
    // Did we succeed..?
    $this->valid = $res;
    return $res;
  } // get
  // .....................................................................
  /** Save current catalog item definition to the database. Inserts
  * if brand new, else performs an update.
  */
  function save() {
    global $RESPONSE;
    $res = false;
    if ($this->newcat || ($this->cat_id && $this->valid)) {
      if ($this->newcat) {
        $this->cat_id = get_next_sequencevalue("seq_cat_id", "ax_catalog", "cat_id");
        $this->uploaded_ts = time();
        $this->uploaded = timestamp_to_displaydate(NICE_FULLDATETIME, $this->uploaded_ts);
        $cup = new dbinsert("ax_catalog");
        $cup->set("cat_id", $this->cat_id);
        $this->newcat = false;
      }
      else {
        $cup = new dbupdate("ax_catalog");
        $cup->where("cat_id=$this->cat_id");
      }
      $cup->set("cat_name",         $this->cat_name);
      $cup->set("cat_desc",         $this->cat_desc);
      $cup->set("mime_type",        $this->mime_type);
      $cup->set("mime_category",    $this->mime_category);
      $cup->set("keywords",         $this->keywords);
      $cup->set("category",         $this->category);
      $cup->set("filesize",         $this->filesize);
      $cup->set("filepath",         $this->filepath);
      $cup->set("width",            $this->width);
      $cup->set("height",           $this->height);
      $cup->set("upload_timestamp", timestamp_to_datetime($this->uploaded_ts));
      $res = $cup->execute();
      if ($res) {
        // Index to search engine..
        $this->index();
        // Update status of physical file..
        $this->physicalpath = $RESPONSE->site_docroot . $this->filepath;
        $this->fileexists = file_exists($this->physicalpath);
      }
    }
    return $res;
  } // save
  // .....................................................................
  /**
  * Remove the catalog item from the database and disk. This method
  * normally tries to remove the physical file first, and if that succeeds
  * it removes the database record. If $deletefile is false then the file
  * will be left and only the DB record deleted.
  * @param boolean $deletefile If true the physical file will be deleted
  * @return boolean True if the operation succeeded, else false.
  */
  function delete($deletefile=true) {
    global $RESPONSE, $IMAGESDIR;
    $deleted = false;
    $err = false;
    if ($this->cat_id && $this->valid) {
      // First, deal to the physical file..
      if ($deletefile) {
        if (file_exists($this->physicalpath)) {
          // Avoid deleting images which are part of the standard issue..
          if (substr($this->filepath, 0, strlen($IMAGESDIR)) != $IMAGESDIR) {
            if (!is_writeable($this->physicalpath) || !unlink($this->physicalpath)) {
              $err = true;
            }
          }
        }
      }
      // Delete the database record now..
      if (!$err) {
        $dcat = new dbdelete("ax_catalog");
        $dcat->where("cat_id=$this->cat_id");
        $res = $dcat->execute();
        if ($res) {
          // Remove from search engine index..
          $this->unindex();
          $this->valid = false;
          $deleted = true;
        }
      }
    }
    return $deleted;
  }  // delete
  // .....................................................................
  /**
  * Index this catalog item to the search engine.
  * If it exists already, index entry for this item is replaced.
  */
  function index() {
    global $SE_AVAILABLE;
    if ($this->valid && $SE_AVAILABLE) {
      $I = new searchengine_indexer();
      // Make sure these fields are stored..
      $I->define_field("catname", "Text", STORED);
      $I->define_field("category", "Text", STORED);
      $I->define_field("mimecat", "Text", STORED);
      $I->define_field("uploaded", "Date", STORED);
      $I->define_field("path", "text", STORED);
      // Index the fields..
      $I->index_field("Id",       "CAT_" . $this->cat_id);
      $I->index_field("keywords", $this->keywords);
      $I->index_field("catname",  $this->cat_name);
      $I->index_field("category", $this->category);
      $I->index_field("mimecat",  $this->mime_category);
      $I->index_field("uploaded", $this->uploaded_ts);
      $I->index_field("path",     $this->filepath);
      // Index the content..
      $allcontent[] = $this->cat_name;
      $allcontent[] = $this->cat_desc;
      $allcontent[] = $this->keywords;
      $I->index_content("CAT_" . $this->cat_id, implode(" ", $allcontent));
      $I->execute();
    }
  } // index
  // .....................................................................
  /** Remove this catalog item from the search engine index. */
  function unindex() {
    global $SE_AVAILABLE;
    if ($this->valid && $SE_AVAILABLE) {
      $UI = new searchengine_unindexer();
      $UI->unindex("CAT_" . $this->cat_id);
      $UI->execute();
    }
  } // unindex
  // .....................................................................
  /** Return the catalog item as a clickable icon.
  * @param boolean $autostart Movies etc. if true start automatically
  */
  function AsIcon($showcontrols=false, $autostart=false, $loop=false) {
    $icon = "";
    $tooltip = basename($this->filepath);
    // Set commonsense values for movies..
    if (!$showcontrols) {
      $autostart = true;
      $loop = true;
    }
    switch ($this->mime_category) {
      case "movie":
        $media = new MediaObject($this->filepath, $this->width, $this->height, $autostart, $loop, $showcontrols);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "audio":
        $media = new MediaObject($this->filepath, $this->width, $this->height, $autostart, $loop, $showcontrols);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "flash":
        $media = new FlashObject($this->filepath, $this->width, $this->height, $autostart, $loop);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "document":
        $media = new DocumentObject($this->filepath, $this->width, $this->height);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      case "image":
        $media = new img($this->filepath, $this->cat_name);
        $media->AsIcon($tooltip);
        $icon = $media->render();
        break;
      default:
        $icon = $this->AsLink();
    } // switch
    return $icon;
  } // AsIcon
  // .....................................................................
  /** Return the catalog item as image, a clickable icon, or otherwise a link.
  * @param boolean $autostart Movies etc. if true start automatically
  */
  function Insitu($showcontrols=false, $autostart=false, $loop=false) {
    $catmedia = "";
    $tooltip = basename($this->filepath);
    // Set commonsense values for movies..
    if (!$showcontrols) {
      $autostart = true;
      $loop = true;
    }
    switch ($this->mime_category) {
      case "movie":
        $media = new MediaObject($this->filepath, $this->width, $this->height, $autostart, $loop, $showcontrols);
        $catmedia = $media->render();
        break;
      case "flash":
        $media = new FlashObject($this->filepath, $this->width, $this->height, $autostart, $loop);
        $catmedia = $media->render();
        break;
      case "image":
        $media = new img(
          $this->filepath,
          str_replace(" ", "", $this->cat_name),
          ($this->cat_desc != "" ? $this->cat_desc : $this->cat_name)
          );
        $catmedia = $media->render();
        break;
      default:
        $catmedia = $this->AsIcon();
    } // switch
    return $catmedia;
  } // In-situ
  // .....................................................................
  /** Return the catalog item as a clickable hyperlink. */
  function AsLink() {
    $catlink = new anchor($this->filepath, $this->cat_name);
    $catlink->set_linkover_text( basename($this->filepath) );
    return $catlink->render();
  } // AsLink
  // .....................................................................
  /**
  * Create this catalog item from a media file on disk. The media item
  * should be located at a physical path on disk somewhere. It will be grabbed,
  * moved to a new location, and the item record saved to the DB.
  * NOTE: this may be an existing catalogitem, OR a newly created one. This
  * is not determined by this routine, but must be set up before calling this
  * method. The save() method then does whatever is necessary.
  * @param string $srcpath The full physical filesystem path to the source media
  * @param string $destpath The full physical path of where to move the media
  * @param string $filepath The relative path of the media, to save on DB record
  * @param string $mimetype The mime type of the file, eg: "image/jpeg"
  * @param string $name Media file associated name
  * @param string $category Media file associated category
  * @param string $desc Media file full description
  * @param string $keywords Media file associated keywords
  * @return boolean True if all ok, else false
  */
  function create(
      $srcpath, $destpath, $filepath,
      $mimetype="",
      $width=0, $height=0,
      $name="", $category="", $desc="", $keywords=""
  ) {
    global $RESPONSE;

    // Valid flag..
    $this->valid = false;

    // Move file to destination & create our catalog record..
    if (is_readable($srcpath)) {

      // Determine mime-type
      if ($mimetype == "") {
        $mimetype = mimetype_from_filename($srcpath);
      }
      // Determine mime-category..
      $mime_category = mimetype_category($mimetype);

      // Size of this file..
      $filesize = filesize($srcpath);

      // Image metrics..
      if ($mime_category == "image") {
        $imginfo = getimagesize($srcpath);
        $width  = $imginfo[0];
        $height = $imginfo[1];
      }
      // Move to destination, if different to source location..
      if (realpath($destpath) != realpath($srcpath)) {
        if (copy($srcpath, $destpath) === false) {
          $errmsgs[] = "failed to copy new media to its destination: $srcpath --> $destpath";
        }
      }
      // Store file in database..
      if ($name == "") {
        $name = basename($srcpath);
      }
      $this->cat_name = $name;
      $this->cat_desc = $desc;
      $this->keywords = $keywords;
      $this->category = $category;
      $this->mime_category = $mime_category;
      $this->mime_type = $mimetype;
      $this->filesize = $filesize;
      $this->filepath = $filepath;
      $this->physicalpath = $destpath;
      $this->fileexists = file_exists($destpath);
      $this->width = $width;
      $this->height = $height;
      if ($this->save()) {
        $this->valid = true;
      }
      else {
        $errmsgs[] = "failed to save catalog item $this->cat_id to database.";
      }
    }
    // Return the status..
    return $this->valid;

  } // create
  // .....................................................................
  /**
  * Process an uploaded media file, and define this catalog item to be
  * the newly uploaded file. Assuming a valid upload is performed, this
  * catalog item will be added to the database, and the file stahsed
  * in the media directory. This method is provided to allow for easy
  * handling of upload form submission particularly for the Axyl media
  * catalog. Ie. use this if you have a form which is just for uploading
  * new images, movies etc. to the Axyl catalog.
  * @param string $name Media file associated name
  * @param string $category Media file associated category
  * @param string $desc Media file full description
  * @param string $keywords Media file associated keywords
  * @return array Error messages, if any occurred
  */
  function upload($name="", $category="", $desc="", $keywords="") {
    global $CATALOGDIR, $RESPONSE;
    global $_overwrite;

    $this->valid = false;
    $errmsgs = array();
    $up = new fileupload();
    if ($up->uploaded_count >= 1) {
      // Process uploaded file..
      $file_mime = $up->mimetype;

      // The relative URL path we will save on the DB record..
      $filepath = "$CATALOGDIR/$up->filename";
      // The physical path on the filesystem..
      $destpath = $RESPONSE->site_docroot . $filepath;

      // Check pre-existence/overwrite flag..
      if (!file_exists($destpath) || isset($_overwrite)) {
        // Determine mime_category..
        $mime_category = mimetype_category($file_mime);
        if ($mime_category == "") {
          $file_mime = mimetype_from_filename($up->filename);
          $mime_category = mimetype_category($file_mime);
          if ($mime_category == "") {
            $errmsgs[] = "That file is of a type unknown to the system [$file_mime].";
          }
        }
        // Carry on if legal mime_category..
        if ($mime_category != "") {
          // Store file in catalog..
          $srcpath = tempnam("/tmp", "CATA");
          $tempdir = dirname($srcpath);
          $tempfname = basename($srcpath);
          if ($up->store($tempdir, $tempfname)) {

            // If we are doing this from a microsite, then possibly
            // assign the category, if not already assigned..
            if ($RESPONSE->microsites_mode == MICROSITES_ENABLED) {
              if ($RESPONSE->microsite_detected != "" && $category == "") {
                $category = $RESPONSE->microsite_detected;
              }
            }
            // Move new media file into place, and save data to DB..
            $this->create(
                    $srcpath, $destpath, $filepath,
                    $file_mime,
                    0, 0,
                    $name, $category, $desc, $keywords
                    );
            // Tidy up temporary file..
            if (is_writeable($srcpath)) {
              unlink($srcpath);
            }
          }
          else {
            $errmsgs[] = $up->error_message();
          }
        }
        else {
          $errmsgs[] = "No content type was specified.";
        }
      }
      else {
        $errmsgs[] = "File exists. Check the box to overwrite.";
        $overwrite_chk = true;
      }
    }
    else {
      $errmsgs[] = "Nothing was uploaded.";
    }
    return $errmsgs;
  } // upload
  // .....................................................................
  /**
  * Render the catalog item. We render it as either and icon or a link,
  * both being clickable to view the content.
  * @param string $mode Mode of rendering: 'icon' or 'link'.
  * @return string HTML rendering of this catalog item in given mode
  */
  function html($mode="link") {
    $s = "";
    switch ($mode) {
      case "icon":
        $s = $this->AsIcon();
        break;
      case "insitu":
        $s = $this->Insitu(false);
        break;
      default: // case "link"
        $s = $this->AsLink();
    } // switch
    return $s;
  } // html

} // catalogitem class

// -----------------------------------------------------------------------
/**
* This class encapsulates a media catalog, which is a collection of
* catalogitem objects.
* @package catalog
*/
class catalog extends RenderableObject {
  /** The array of catalogitem objects in this catalog */
  var $catalogitems = array();
  /** The array of categories (optional) */
  var $categories = array();
  // .....................................................................
  /** Constructor */
  function catalog($catalogitems=false) {
    if (is_array($catalogitems)) {
      $this->catalogitems = $catalogitems;
    }
  } // catalog
  // .....................................................................
  /** Return the count of items in the catalog currently.
  * @return integer Count of items in the catalog.
  */
  function itemcount() {
    return count($this->catalogitems);
  }
  // .....................................................................
  /**
  * Define the list of user categories which is used to categorise
  * the catalog items in this catalog. This is submitted as an
  * associative array with a single word string as key, and a multi-word
  * string as the value (category description).
  * @param array $cats Array of catalog categories: ID => description
  */
  function set_categories($categories) {
    if (is_array($categories)) {
      $this->categories = $categories;
    }
  } // define_categories
  // .....................................................................
  /**
  * Add catalog item to the catalog. The catalog will add the new item
  * by cat_id. If that ID already exists it is overwritten.
  * @param object $catitem The catalog item to add
  */
  function additem($catitem) {
    if (is_object($catitem) && is_a($catitem, "catalogitem")) {
      $this->catalogitems[$catitem->cat_id] = $catitem;
    }
  } // additem
  // .....................................................................
  /**
  * Find a catalogitem by filepath. Just trawls the array of catalog
  * items in this catalog comparing filepaths. Returns false if not found
  * else a copy of the catalogitem object.
  * @param string $filepath Website URL that image is located at.
  * @return mixed False if we didn't match a filepath, else catalog item copy.
  */
  function get_catalogitem_by_filepath($filepath) {
    $found = false;
    foreach ($this->catalogitems as $catitem) {
      if ($catitem->filepath == $filepath) {
        $found = $catitem;
        break;
      }
    }
    return $found;
  } // get_catalogitem_by_filepath
  // .....................................................................
  /**
  * Perform a search for catalog items. This method will use the search
  * engine if it is available and keywords are provided, otherwise it will
  * just go to the database. The end result is that this catalog is
  * populated with the results of the search, having been cleared out
  * beforehand.
  * @param string $keywords Keywords to search for
  * @param array  $mimecats Array of content types (mime categories)
  * @param array  $categories Array of media user-categories
  * @param strong $sortby Sort order for results
  */
  function search($keywords="", $mimecats="", $categories="", $sortby="") {
    global $SE_AVAILABLE;

    // Initialise catalog..
    $this->catalogitems = array();

    // KEYWORD SEARCH..
    if ($keywords != "" && $SE_AVAILABLE) {
      $thesearch = new searchengine_search();

      // CONTENT TYPES
      if (is_array($mimecats) && count($mimecats) > 0) {
        $terms = array();
        foreach ($mimecats as $mimecat) {
          // The nullstring represents 'Any' or 'All'..
          if ($mimecat != "") $terms[] = $mimecat;
        }
        if (count($terms) > 0) {
          $thesearch->must_match( "mimecat:(" . implode(" ", $terms) . ")" );
        }
      }

      // CATEGORIES
      $thesearch->does_not_match( "category:sitecontent" );
      if (is_array($categories) && count($categories) > 0) {
        $terms = array();
        foreach ($categories as $cat) {
          // The nullstring represents 'Any' or 'All'..
          if ($cat != "") $terms[] = $cat;
        }
        if (count($terms) > 0) {
          $thesearch->must_match( "category:(" . implode(" ", $terms) . ")" );
        }
      }

      // KEYWORDS
      // Keywords matching. We search the keywords fields, and
      // also the default text field..
      if ($keywords != "") {
        $words = explode(" ", $keywords);
        $terms = array();
        foreach ($words as $word) {
          if ($word != "") $terms[] = "$word^2";
        }
        $keyword_terms = implode(" ", $terms);

        $terms = array();
        foreach ($words as $word) {
          if ($word != "") $terms[] = $word;
        }
        $text_terms = implode(" ", $terms);
        // Construct the search terms..
        $thesearch->must_match( "(keywords:($keyword_terms) ($text_terms))" );
      }

      // SORT ORDER
      switch ($sortby) {
        case "rank":
          $thesearch->set_sortorder("RANK");
          break;
        case "uploaded":
          $thesearch->set_sortorder("uploaded:Date:Desc");
          break;
        case "catname":
          $thesearch->set_sortorder("catname");
          break;
        case "mimecat":
          $thesearch->set_sortorder("mimecat,uploaded:Date:Desc");
          break;
        case "category":
          $thesearch->set_sortorder("category,uploaded:Date:Desc");
          break;
        default:
          $thesearch->set_sortorder("uploaded:Date:Desc");
          break;
      } // switch

      // RETURN FIELDS
      $thesearch->set_returnfields("catname,category,mimecat,uploaded:Date,path");
      $thesearch->execute();

      // Populate the catalog..
      if ($thesearch->hitcount() > 0) {
        foreach ($thesearch->hit as $hit) {
          $ci = new catalogitem();
          $bits = explode("_", $hit["Id"]);
          $ci->cat_id = $bits[1];
          $ci->cat_name = $hit["catname"];
          $ci->category = $hit["category"];
          $ci->mime_category = $hit["mimecat"];
          $ci->uploaded_ts = $hit["uploaded"];
          $ci->uploaded = timestamp_to_displaydate(NICE_FULLDATETIME, $ci->uploaded_ts);
          $ci->filepath = $hit["path"];
          $ci->valid = true;
          $ci->newcat = false;
          $this->catalogitems[$ci->cat_id] = $ci;
        }
      }

    } // got keywords
    else {
      $cQ  = new dbselect("ax_catalog");
      $wheres = array();

      // CONTENT TYPES
      if (is_array($mimecats) && count($mimecats) > 0) {
        // The nullstring represents 'Any' or 'All'..
        if (!in_array("", $mimecats)) {
          $wheres[] = "mime_category IN ('" . implode("','", $mimecats) . "')";
        }
      }

      // CATEGORIES
      if (is_array($categories) && count($categories) > 0) {
        // The nullstring represents 'Any' or 'All'..
        if (!in_array("", $categories)) {
          $wheres[] = "category IN ('" . implode("','", $categories) . "')";
        }
      }
      if (count($wheres) > 0) {
        $cQ->where( implode(" AND ", $wheres) );
      }

      // SORT ORDER
      switch ($sortby) {
        case "uploaded":
          $cQ->orderby("upload_timestamp DESC");
          break;
        case "catname":
          $cQ->orderby("cat_name");
          break;
        case "mimecat":
          $cQ->orderby("mime_category");
          break;
        case "category":
          $cQ->orderby("category");
          break;
        default:
          $cQ->orderby("upload_timestamp DESC");
          break;
      } // switch

      $cQ->execute();
      if ($cQ->hasdata) {
        do {
          $ci = new catalogitem();
          $ci->cat_id = $cQ->field("cat_id");
          $ci->cat_name = $cQ->field("cat_name");
          $ci->category = $cQ->field("category");
          $ci->mime_category = $cQ->field("mime_category");
          $ci->uploaded_ts = datetime_to_timestamp($cQ->field("upload_timestamp"));
          $ci->uploaded = timestamp_to_displaydate(NICE_FULLDATETIME, $ci->uploaded_ts);
          $ci->filepath = $cQ->field("filepath");
          $ci->valid = true;
          $ci->newcat = false;
          $this->catalogitems[$ci->cat_id] = $ci;
        } while ($cQ->get_next());
      }
    }

  } // search
  // .....................................................................
  /** Return the HTML for this catalog. Returns the list of catalog items
  * as an HTML table.
  * @return string HTML table containing the whole catalog
  */
  function html() {
    global $RESPONSE, $LIBDIR;
    return "";
  } // html

} // catalog class

// -----------------------------------------------------------------------
?>