#include "includes.h"

/*
** This file contains the functions which print the header and a 
** statistics line for every defined flag.                
**                                                  
** New functions (to print new statistics) can be added to this file
** as follows:                                                           
**       1. Add a new entry to the function definition table at the end
**          of this file (funcdef[]).                                 
**       2. Add the defined 'printhead' and 'printline' functions.   
**                                                                  
** If certain counters are required which are not yet collected, the
** fetchdef.c file of the data-collector should be updated as well. 
**                                                                  
** Function 'printhead':                                          
**       Called with one parameter:                          
**          1. Structure holding the release- and version-info of kernel.
**
**       Return value: not relevant.                             
**
**       This function is assumed to print the header of a statistics
**       list behind the time-stamp, which has already been printed 
**       before the 'printhead' function is called. So the header may contain
**       70 characters maximum.
**                                                                
** Function 'printline':                                         
**       Called with six parameters:                          
**          1. Number of seconds     passed since the previous sample.
**          2. Number of clock-ticks passed since the previous sample.
**          3. Number of clock-ticks per second for the target-system.
**          4. Number of processors available in the target-system
**          5. Structure holding the release- and version-info of kernel.
**          6. Time-stamp string to be used when printing more than one line.
**
**       Return value:     true  (1) in case the relevant counters were
**                                   succesfully read and printed
**                         false (0) in case the relevant counters could
**                                   not be obtained; the reaction on   
**                                   this return-value is that a message
**                                   is printed and the 'printline' is  
**                                   never called again.                
**                                                                      
** The 'printline' function can get the required counters via the    
** function 'getcset' (see file atsar.c).                               
** ---------------------------------------------------------------------
** Author:      Gerlof Langeveld - AT Computing, Nijmegen, Holland
** E-mail:      gerlof@ATComputing.nl
** Date:        Januar  1995
** LINUX-port:  Februar 1999
** 
** $Log: funcdef.c,v $
** Revision 1.22  2001/03/16 09:38:17  gerlof
** Reimplement FTP/HTTP counters.
**
** Revision 1.21  2001/03/14 14:03:00  root
** Changed numbers of disks 2.2
**
** Revision 1.20  2001/03/14 11:08:58  gerlof
**  TTY statistics added.
**
** Revision 1.19  2001/03/12 14:57:18  gerlof
** IPv6 counters.
**
** Revision 1.18  2001/02/26 15:34:14  gerlof
** Show load-averages in the default atsar-output.
**
** Revision 1.17  2001/02/26 15:27:11  gerlof
** Use counters of type 'double' i.s.o. long for HTTP and FTP.
**
** Revision 1.16  2000/11/07 09:21:10  gerlof
** Support for modified per-partition statistics.
**
** Revision 1.15  1999/10/18 12:57:56  gerlof
** Separate counters for FTP-input and -output.
**
** Revision 1.14  1999/10/18 08:58:53  gerlof
** Error-message when using -F or -H flag in combination with an interval.
**
** Revision 1.13  1999/09/23 14:38:47  gerlof
** Function partline did not return 1 for success
**
** Revision 1.12  1999/09/22 11:30:47  gerlof
** modified headerline of page/swap counters (-p)
**
** Revision 1.11  1999/09/17 11:26:14  gerlof
** Changed header-line of disk-output: sect i.s.o. blk
**
** Revision 1.10  1999/09/17 09:38:08  gerlof
** Extra counter (kbytes/s) in output of _ftp_
**
** Revision 1.9  1999/09/16 09:26:33  gerlof
** Minor code-restyling.
**
** Revision 1.8  1999/08/31 09:33:36  gerlof
** Implement support for WEB-servers.
** Porting to version 2.0.
**
** Revision 1.7  1999/08/30 13:54:29  gerlof
** Code-cleanup rpc-call statistics
**
** Revision 1.6  1999/08/27 14:05:04  gerlof
** Added NFS-statistics.
**
** Revision 1.5  1999/08/26 06:55:23  gerlof
** Code-cleanup, new calling-parameter for print-line functions (number of
** cpu's), new flags -W, -T and -L introduced and reorganized output per flag.
**
** Revision 1.4  1999/05/18 08:31:03  gerlof
** Back-port from version 2.2 to 2.0.
**
** Revision 1.3  1999/05/11 09:08:08  gerlof
** Add support for disk-partition statistics and kernel-tables.
**
** Revision 1.2  1999/05/10 10:12:24  gerlof
** New flag -v and changed output for flag -p.
**
** Revision 1.1  1999/05/05 11:40:11  gerlof
** Initial revision
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public License as published by the
** Free Software Foundation; either version 2, or (at your option) any
** later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU General Public License for more details.
*/

static char rcsid[] = "$Id: funcdef.c,v 1.22 2001/03/16 09:38:17 gerlof Exp $";


/*
** Function proto-types
*/
int	getcset(char *, void **, void **);


/*
** CPU statistics
*/
void
cpuhead(struct osrel *osr)
{
	printf("cpu %%usr %%sys %%nice %%idle pswch/s runq nrproc "
	       "lavg1 lavg5 avg15 _cpu_\n");
}

int
cpuline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register int		i;
	unsigned long		usrdelta, sysdelta, nicdelta;
	long			idldelta;
	struct genstat	 	*cur, *pre;


	/*
	** obtain the general statistics
	*/
	if ( !getcset("genstat", (void **)&cur, (void **)&pre) )
			return(0);

	/*
	** print overall statistics
	*/
	usrdelta = (cur->cpu_user   - pre->cpu_user)   / ncpu;
	sysdelta = (cur->cpu_system - pre->cpu_system) / ncpu;
	nicdelta = (cur->cpu_nice   - pre->cpu_nice)   / ncpu;
	idldelta = deltatic - (usrdelta+sysdelta+nicdelta);

	if (idldelta < 0)
		idldelta = 0;

	printf("all %4.0f %4.0f %5.0f %5.0f %7d %4d %6d %5.2f %5.2f %5.2f\n",
		(double) (usrdelta * 100) / deltatic,
		(double) (sysdelta * 100) / deltatic,
		(double) (nicdelta * 100) / deltatic,
		(double) (idldelta * 100) / deltatic,
		(cur->context_swtch - pre->context_swtch) / deltasec,
		 cur->nrrun,    cur->nrproc,
		 cur->loadavg1, cur->loadavg5, cur->loadavg15);

	/*
	** print per-cpu statistics
	*/
	if (ncpu > 1)
	{
		for (i=0; i < ncpu; i++)
		{
			usrdelta = cur->per_cpu_user[i] - pre->per_cpu_user[i];
			sysdelta = cur->per_cpu_system[i] -
						pre->per_cpu_system[i];
			nicdelta = cur->per_cpu_nice[i] - pre->per_cpu_nice[i];
			idldelta = deltatic - (usrdelta+sysdelta+nicdelta);
	
			if (idldelta < 0)
				idldelta = 0;

			printf("%s  %3d %4.0f %4.0f %5.0f %5.0f\n",
				tstamp, i,
				(double) (usrdelta * 100) / deltatic,
				(double) (sysdelta * 100) / deltatic,
				(double) (nicdelta * 100) / deltatic,
				(double) (idldelta * 100) / deltatic);
		}
	}

	return(1);
}


/*
** Memory- & swap-usage
*/
void
memhead(struct osrel *osr)
{
	printf("memtot memfree memshared  buffers   cached"
	       "      swptot swpfree  _mem_\n"             );
}

int
memline(time_t deltasec, time_t deltatic, time_t hz, 
				int ncpu, struct osrel *osr, char *tstamp)
{
	register unsigned long	tm, fm, sm, bm, cm, ts, fs;
	struct memstat	 	*mcur, *mpre;

	if ( !getcset("memstat", (void **)&mcur, (void **)&mpre) )
		return(0);

	tm = mcur->memory[MTOTAL]  / 1024;
	fm = mcur->memory[MFREE]   / 1024;
	sm = mcur->memory[MSHARED] / 1024;
	bm = mcur->memory[MBUFS]   / 1024;
	cm = mcur->memory[MCACHED] / 1024;
 
	ts = mcur->swap[MTOTAL]    / 1024;
	fs = mcur->swap[MFREE]     / 1024;

	printf("%5dM %6dK %8dK %7dK %7dK      %5dM %6dK\n",
		tm/1024, fm, sm, bm, cm,
		ts/1024, fs);

	return(1);
}

/*
** Paging & swapping statistics
*/
void
pagehead(struct osrel *osr)
{
	printf(" pagein/s pageout/s     swapin/s swapout/s     "
	       "    fork/s      _page_\n");
}

int
pageline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	si, so, pi, po, fo;
	struct genstat	*scur, *spre;
	struct tabstat	*pcur, *ppre;

	/*
	** obtain the general statistics
	*/
	if ( !getcset("genstat", (void **)&scur, (void **)&spre) )
			return(0);

	pi = scur->pgpgin   - spre->pgpgin;
	po = scur->pgpgout  - spre->pgpgout;
	si = scur->pswpin   - spre->pswpin;
	so = scur->pswpout  - spre->pswpout;

	if (getcset("tabstat", (void **)&pcur, (void **)&ppre) )
		fo = (pcur->totforks - ppre->totforks);
	else
		fo = 0;

	printf("%9.2f %9.2f    %9.2f %9.2f        %7.2f\n",
		(double) pi / deltasec,
		(double) po / deltasec,
		(double) si / deltasec,
		(double) so / deltasec,
		(double) fo / deltasec);

	return(1);
}


/*
** Interrupt statistics
*/
static char	usedirq[16];
static int	usedmax;

void
inthead(struct osrel *osr)
{
	register unsigned long	i;
	static char		firstcall = 1;
	static char		headerline[128];
	struct genstat		*cur,  *pre;

	/*
	** generate the header-line dynamically, only showing headers 
	** for the irq's which have been used so far
	*/
	if (firstcall)
	{
		/*
		** obtain the general statistics
		*/
		if ( !getcset("genstat", (void **)&cur, (void **)&pre) )
		{
			memcpy(headerline, "\n", 2);
			printf(headerline);
			firstcall = 0;
			return;
		}

		for (i=0; i < MAXIRQ; i++)
		{
			if (pre->irqs[i])
			{
				usedirq[usedmax] = i;
				if ( ++usedmax >= 16 )
					break;
			}
		}

		memcpy(headerline, "cpu  ", 5);

		for (i=0; i < usedmax; i++)
		{
			char tmpbuf[16];

			sprintf(tmpbuf, "iq%02d ", usedirq[i]);
			strcat(headerline, tmpbuf);
		}

		/*
		** try if a right-aligned marker still fits
		*/
		i = strlen(headerline);

		if (i < 61)
		{
			memset(&headerline[i], ' ', 61-i);
			strcat(headerline, "_intr/s_");
		}
		strcat(headerline, "\n");

		firstcall = 0;
	}

	printf(headerline);
}

int
intline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register unsigned long	i, j;
	struct genstat	 	*cur,  *pre;

	/*
	** obtain the general statistics
	*/
	if ( !getcset("genstat", (void **)&cur, (void **)&pre) )
		return(0);

	/*
	** print overall interrupts
	*/
	printf("all  ");

	for (i=0; i < usedmax; i++)
	{
		printf("%4d ", (cur->irqs[usedirq[i]] -
				pre->irqs[usedirq[i]]) / deltasec);
	}
	printf("\n");

	/*
	** print per-cpu interrupts
	*/
	if (ncpu > 1)
	{
		for (i=0; i < ncpu; i++)
		{
			printf("%s  %3d  ", tstamp, i);

			for (j=0; j < usedmax; j++)
			{
				printf("%4d ", (cur->per_irqs[i][usedirq[j]]-
						pre->per_irqs[i][usedirq[j]]) /
								deltasec);
			}
			printf("\n");
		}
			
	}

	return(1);
}

/*
** Kernel-parameter statistics
*/
void
tabhead(struct osrel *osr)
{
	printf("superb-sz inode-sz     file-sz    dquota-sz"
	       "    flock-sz      _curmax_\n"               );
}

int
tabline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	struct tabstat	 	*pcur, *ppre;

	if ( !getcset("tabstat", (void **)&pcur, (void **)&ppre) )
		return(0);

	printf("%4d/%-4d  %5d/%-5d  %4d/%-4d  %5d/%-5d  %4d/%-4d\n",
		pcur->cursuper, pcur->maxsuper,
		pcur->curinode, pcur->maxinode,
		pcur->curfiles, pcur->maxfiles,
		pcur->curdquot, pcur->maxdquot,
		pcur->curlocks, pcur->maxlocks);

	return(1);
}

/*
** Kernel-parameter statistics
*/
void
ttyhead(struct osrel *osr)
{
	printf("port   xmit/s   recv/s   frer/s  parer/s  ovrun/s    brk/s"
	       "      _tty_\n");
}

int
ttyline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register int		nr, i, j, lin;
	struct serialstat 	*tcur, *tpre;
	unsigned long		tx, rx, fe, pe, br, oe;
	static int			sampcnt = 0;

	if ( !( nr = getcset("ttystat", (void **)&tcur, (void **)&tpre)) )
		return(0);

	sampcnt++;

	for (i=j=lin=0; i < nr && j < nr && (tcur+i)->port; i++, j++)
	{
		/*
		** note that only the active tty's are registered;
		** when a new tty has been used for the first time in
		** the last sample, the position of the existing statistics
		** in the table might have been shifted related to the
		** previous sample
		*/
		if ( (tcur+i)->port != (tpre+j)->port)
		{
			/* resync */
			for (j=0; j < nr && (tpre+j)->port; j++)
			{
				if ( (tcur+i)->port == (tpre+j)->port)
					break;
			}
		}

		tx = (tcur+i)->tx - (tpre+j)->tx;
		rx = (tcur+i)->rx - (tpre+j)->rx;
		fe = (tcur+i)->fe - (tpre+j)->fe;
		pe = (tcur+i)->pe - (tpre+j)->pe;
		oe = (tcur+i)->oe - (tpre+j)->oe;
		br = (tcur+i)->br - (tpre+j)->br;

		if (sampcnt > 1 && tx+rx+fe+pe+oe+br == 0)
			continue;

		lin++;

		if (lin > 1)
			printf("%s  ", tstamp);

		printf("%4X %8.2f %8.2f %8.3f %8.3f %8.3f %8.3f\n",
			(tcur+i)->port,
			(double) tx / deltasec,
			(double) rx / deltasec,
			(double) fe / deltasec,
			(double) pe / deltasec,
			(double) oe / deltasec,
			(double) br / deltasec);
	}

	if (lin == 0)
		printf("\n");

	return(1);
}


/*
** Network-interface statistics
*/
void
ifhead(struct osrel *osr)
{
	printf("inpck/s otpck/s inbyt/s otbyt/s "
	       "incmpr/s otcmpr/s inmcst/s iface _if_\n"     );
}

int
ifline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	static int		sampcnt = 0;
	register int	i, j, nvalid;
	unsigned long	nrif, ip, op, ib, ob, ic, oc, im;
	struct ifstat 	*cur, *pre;

	if ( !(nrif = getcset("ifstat", (void **)&cur, (void **)&pre)) )
		return(0);

	sampcnt++;

	for (i=0, j=0, nvalid=0; i < nrif; i++, j=i)	/* per interface */
	{
		if ( (cur+i)->name[0] == 0 )
			break;		/* always returns 'true', even if  */
					/* no entries are found in this    */
					/* interval                        */

		/*
		** note that if-layers may be added or deleted since the
		** previous sample; this also means that the position
		** of the existing statistics in this table might have
		** been shifted since the previous sample
		*/
		if ( strcmp( (cur+i)->name, (pre+j)->name) != 0)
		{
			/* resync */
			for (j=0; j < nrif; j++)
			{
				if ( (pre+j)->name[0] == 0 )
					break;	/* not known in previous samp */

				if  ( strcmp( (cur+i)->name, (pre+j)->name) ==0)
					break;
			}

			sampcnt = 1;	/* show the new total list of if's */
		}

		ip = (cur+i)->rpack      - (pre+j)->rpack;
		op = (cur+i)->spack      - (pre+j)->spack;
		ib = (cur+i)->rbyte      - (pre+j)->rbyte;
		ob = (cur+i)->sbyte      - (pre+j)->sbyte;
		ic = (cur+i)->rcompr     - (pre+j)->rcompr;
		oc = (cur+i)->scompr     - (pre+j)->scompr;
		im = (cur+i)->rmultic    - (pre+j)->rmultic;

		/*
		** print for the first sample all interfaces which
		** are found; afterwards print only the interfaces
		** which were really active during the interval
		*/
		if (sampcnt > 1 && !ip && !op)
			continue;

		if (nvalid)	printf("%s  ", tstamp);

		nvalid++;

		printf("%7.1f %7.1f %7d %7d %8.2f %8.2f %8.2f %s\n", 
			(double) ip / deltasec,
			(double) op / deltasec,
			         ib / deltasec,
			         ob / deltasec,
			(double) ic / deltasec,
			(double) oc / deltasec,
			(double) im / deltasec,
			(cur+i)->name);
	}

	if (nvalid == 0)
		printf("\n");

	return(1);
}

void
IFhead(struct osrel *osr)
{
	printf("inerr/s oterr/s coll/s indrop/s otdrop/s "
	       "infram/s otcarr/s iface _if_\n"     );
}

int
IFline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	static int	sampcnt = 0;
	register int	i, j, nvalid;
	unsigned long	nrif, ip, op, ie, oe, co, id, od, ifi, ofi, ifr, oca;
	struct ifstat 	*cur, *pre;

	if ( !(nrif = getcset("ifstat", (void **)&cur, (void **)&pre)) )
		return(0);

	sampcnt++;

	for (i=0, j=0, nvalid=0; i < nrif; i++, j=i)	/* per interface */
	{
		if ( (cur+i)->name[0] == 0 )
			break;		/* always returns 'true', even if  */
					/* no entries are found in this    */
					/* interval                        */

		/*
		** note that if-layers may be added or deleted since the
		** previous sample; this also means that the position
		** of the existing statistics in this table might have
		** been shifted since the previous sample
		*/
		if ( strcmp( (cur+i)->name, (pre+j)->name) != 0)
		{
			/* resync */
			for (j=0; j < nrif; j++)
			{
				if ( (pre+j)->name[0] == 0 )
					break;	/* not known in previous samp */

				if  ( strcmp( (cur+i)->name, (pre+j)->name) ==0)
					break;
			}

			sampcnt = 1;	/* show the new total list of if's */
		}

		ip  = (cur+i)->rpack      - (pre+j)->rpack;
		op  = (cur+i)->spack      - (pre+j)->spack;
		ie  = (cur+i)->rerrs      - (pre+j)->rerrs;
		oe  = (cur+i)->serrs      - (pre+j)->serrs;
		co  = (cur+i)->scollis    - (pre+j)->scollis;
		id  = (cur+i)->rdrop      - (pre+j)->rdrop;
		od  = (cur+i)->sdrop      - (pre+j)->sdrop;
		ifi = (cur+i)->rfifo      - (pre+j)->rfifo;
		ofi = (cur+i)->sfifo      - (pre+j)->sfifo;
		ifr = (cur+i)->rframe     - (pre+j)->rframe;
		oca = (cur+i)->scarrier   - (pre+j)->scarrier;

		/*
		** print for the first sample all interfaces which
		** are found; afterwards print only the interfaces
		** which were really active during the interval
		*/
		if (sampcnt > 1 && !ip && !op)
			continue;

		if (nvalid)	printf("%s  ", tstamp);

		nvalid++;

		printf("%7.2f %7.2f %6.2f %8.2f %8.2f %8.2f %8.2f %s\n", 
			(double) ie  / deltasec,
			(double) oe  / deltasec,
			(double) co  / deltasec,
			(double) id  / deltasec,
			(double) od  / deltasec,
			(double) ifr / deltasec,
			(double) oca / deltasec,
			(cur+i)->name);
	}

	if (nvalid == 0)
		printf("\n");

	return(1);
}

/*
** IP version 4 statistics
*/
void
ipv4head(struct osrel *osr)
{
	printf("inrecv/s outreq/s indeliver/s forward/s "
	       "reasmok/s fragcreat/s    _ip_\n"         );
}

int
ipv4line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	ir, or, id, fw, ro, fc;
	struct netstat 	*cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	ir = cur->ip.IpInReceives  - pre->ip.IpInReceives;
	or = cur->ip.IpOutRequests - pre->ip.IpOutRequests;
	id = cur->ip.IpInDelivers  - pre->ip.IpInDelivers;
	fw = cur->ip.IpForwarding  - pre->ip.IpForwarding;
	ro = cur->ip.IpReasmOKs    - pre->ip.IpReasmOKs;
	fc = cur->ip.IpFragCreates - pre->ip.IpFragCreates;

	printf("%8.1f %8.1f %11.1f %9.1f %9.1f %11.1f\n", 
		(double) ir / deltasec,
		(double) or / deltasec,
		(double) id / deltasec,
		(double) fw / deltasec,
		(double) ro / deltasec,
		(double) fc / deltasec);

	return(1);
}

void
IPv4head(struct osrel *osr)
{
	printf("in: dsc/s hder/s ader/s unkp/s ratim/s rfail/s "
	       "ot: dsc/s nort/s  _ip_\n"         );
}

int
IPv4line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	ihe, iae, iup, ids, ods, onr, rto, rfl;
	struct netstat 	*cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	ids = cur->ip.IpInDiscards  	- pre->ip.IpInDiscards;
	ihe = cur->ip.IpInHdrErrors  	- pre->ip.IpInHdrErrors;
	iae = cur->ip.IpInAddrErrors 	- pre->ip.IpInAddrErrors;
	iup = cur->ip.IpInUnknownProtos	- pre->ip.IpInUnknownProtos;
	rto = cur->ip.IpReasmTimeout  	- pre->ip.IpReasmTimeout;
	rfl = cur->ip.IpReasmFails  	- pre->ip.IpReasmFails;
	ods = cur->ip.IpOutDiscards  	- pre->ip.IpOutDiscards;
	onr = cur->ip.IpOutNoRoutes  	- pre->ip.IpOutNoRoutes;

	printf("    %5.1f %6.1f %6.1f %6.1f %7.1f %7.1f     %5.1f %6.1f\n", 
		(double) ids / deltasec,
		(double) ihe / deltasec,
		(double) iae / deltasec,
		(double) iup / deltasec,
		(double) rto / deltasec,
		(double) rfl / deltasec,
		(double) ods / deltasec,
		(double) onr / deltasec);

	return(1);
}


/*
** TCP version 4 statistics
*/
void
tcphead(struct osrel *osr)
{
	printf("insegs/s otsegs/s actopen/s pasopen/s  "
	       "nowopen  socknow sockmax _tcp_\n"         );
}

int
tcpline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long		is, os, ao, po, sn=0, sm=0;
	struct netstat 		*cur,  *pre;
	struct sockstat 	*scur, *spre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	if ( getcset("sockstat", (void **)&scur, (void **)&spre) )
	{
		sn = scur->tcpnow;
		sm = scur->tcpmax;
	}

	is = cur->tcp.TcpInSegs        - pre->tcp.TcpInSegs;
	os = cur->tcp.TcpOutSegs       - pre->tcp.TcpOutSegs;
	ao = cur->tcp.TcpActiveOpens   - pre->tcp.TcpActiveOpens;
	po = cur->tcp.TcpPassiveOpens  - pre->tcp.TcpPassiveOpens;

	printf("%8.1f %8.1f %9.1f %9.1f  %7d  %7d %7d\n",
		(double) is / deltasec,
		(double) os / deltasec,
		(double) ao / deltasec,
		(double) po / deltasec,
		cur->tcp.TcpCurrEstab,
		sn, sm);

	return(1);
}

void
TCPhead(struct osrel *osr)
{
	printf("inerr/s  retrans/s  attfail/s  "
	       "estabreset/s  outreset/s         _tcp_\n");
}

int
TCPline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long		ie, rs, af, er, or;
	struct netstat 		*cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	ie = cur->tcp.TcpInErrs        - pre->tcp.TcpInErrs;
	rs = cur->tcp.TcpRetransSegs   - pre->tcp.TcpRetransSegs;
	af = cur->tcp.TcpAttemptFails  - pre->tcp.TcpAttemptFails;
	er = cur->tcp.TcpEstabResets   - pre->tcp.TcpEstabResets;
	or = cur->tcp.TcpOutRsts       - pre->tcp.TcpOutRsts;

	printf("%7.1f  %9.1f  %9.1f  %12.1f  %10.1f\n",
		(double) ie / deltasec,
		(double) rs / deltasec,
		(double) af / deltasec,
		(double) er / deltasec,
		(double) or / deltasec);

	return(1);
}


/*
** UDP version 4 statistics
*/
void
udpv4head(struct osrel *osr)
{
	printf("indgram/s otdgram/s   inerr/s  noport/s    "
	       "     socknow sockmax _udp_\n"         );
}

int
udpv4line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	id, ot, ie, np, sn, sm;
	struct netstat 	*cur,  *pre;
	struct sockstat	*scur, *spre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	if ( getcset("sockstat", (void **)&scur, (void **)&spre) )
	{
		sn = scur->udpnow;
		sm = scur->udpmax;
	}

	id = cur->udp.UdpInDatagrams  - pre->udp.UdpInDatagrams;
	ot = cur->udp.UdpOutDatagrams - pre->udp.UdpOutDatagrams;
	ie = cur->udp.UdpInErrors     - pre->udp.UdpInErrors;
	np = cur->udp.UdpNoPorts      - pre->udp.UdpNoPorts;


	printf("%9.1f %9.1f   %7.2f %9.2f         %7d %7d\n",
		(double) id / deltasec,
		(double) ot / deltasec,
		(double) ie / deltasec,
		(double) np / deltasec,
		sn, sm);

	return(1);
}

/*
** ICMP version 4 statistics
*/
void
icmpv4head(struct osrel *osr)
{
	printf("intot/s outtot/s  inecho/s inerep/s  "
	       "otecho/s oterep/s         _icmp_\n"   );
}


int
icmpv4line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	it, ot, ie, oe, ir, or;
	struct netstat  *cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	it = cur->icmp.IcmpInMsgs  	- pre->icmp.IcmpInMsgs;
	ot = cur->icmp.IcmpOutMsgs 	- pre->icmp.IcmpOutMsgs;
	ie = cur->icmp.IcmpInEchos 	- pre->icmp.IcmpInEchos;
	oe = cur->icmp.IcmpOutEchos	- pre->icmp.IcmpOutEchos;
	ir = cur->icmp.IcmpInEchoReps	- pre->icmp.IcmpInEchoReps;
	or = cur->icmp.IcmpOutEchoReps	- pre->icmp.IcmpOutEchoReps;

	printf("%7.1f %8.1f  %8.2f %8.2f  %8.2f %8.2f\n", 
		(double) it / deltasec, 
		(double) ot / deltasec,
		(double) ie / deltasec,
		(double) ir / deltasec,
		(double) oe / deltasec,
		(double) or / deltasec);

	return(1);
}


void
ICMPv4head(struct osrel *osr)
{
	printf("ierr/s isq/s ird/s idu/s ite/s "
	       "oerr/s osq/s ord/s odu/s ote/s  _icmp_\n");
}

int
ICMPv4line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	ier, oer, idu, odu, ite, ote, isq, osq, ird, ord;
	struct netstat  *cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	ier = cur->icmp.IcmpInErrors      	- pre->icmp.IcmpInErrors;
	idu = cur->icmp.IcmpInDestUnreachs	- pre->icmp.IcmpInDestUnreachs;
	ite = cur->icmp.IcmpInTimeExcds   	- pre->icmp.IcmpInTimeExcds;
	isq = cur->icmp.IcmpInSrcQuenchs  	- pre->icmp.IcmpInSrcQuenchs; 
	ird = cur->icmp.IcmpInRedirects   	- pre->icmp.IcmpInRedirects;
	oer = cur->icmp.IcmpOutErrors		- pre->icmp.IcmpOutErrors;
	odu = cur->icmp.IcmpOutDestUnreachs	- pre->icmp.IcmpOutDestUnreachs;
	ote = cur->icmp.IcmpOutTimeExcds   	- pre->icmp.IcmpOutTimeExcds;
	osq = cur->icmp.IcmpOutSrcQuenchs  	- pre->icmp.IcmpOutSrcQuenchs; 
	ord = cur->icmp.IcmpOutRedirects   	- pre->icmp.IcmpOutRedirects;

	printf("%6.2f %5.2f %5.2f %5.2f %5.2f %6.2f %5.2f %5.2f %5.2f %5.2f\n", 
		(double) ier / deltasec,
		(double) isq / deltasec,
		(double) ird / deltasec,
		(double) idu / deltasec,
		(double) ite / deltasec,
		(double) oer / deltasec,
		(double) osq / deltasec,
		(double) ord / deltasec,
		(double) odu / deltasec,
		(double) ote / deltasec);

	return(1);
}


/*
** IP version 6 statistics
*/
void
ipv6head(struct osrel *osr)
{
	printf("inrecv/s outreq/s inmc/s outmc/s indeliv/s "
	       "reasmok/s fragcre/s  _ip6_\n");
}

int
ipv6line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	ir, or, im, om, id, ro, fc;
	struct netstat 	*cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	ir = cur->ip6.Ip6InReceives   - pre->ip6.Ip6InReceives;
	or = cur->ip6.Ip6OutRequests  - pre->ip6.Ip6OutRequests;
	im = cur->ip6.Ip6InMcastPkts  - pre->ip6.Ip6InMcastPkts;
	om = cur->ip6.Ip6OutMcastPkts - pre->ip6.Ip6OutMcastPkts;
	id = cur->ip6.Ip6InDelivers   - pre->ip6.Ip6InDelivers;
	ro = cur->ip6.Ip6ReasmOKs     - pre->ip6.Ip6ReasmOKs;
	fc = cur->ip6.Ip6FragCreates  - pre->ip6.Ip6FragCreates;

	printf("%8.1f %8.1f %6.1f %7.1f %9.1f %9.1f %9.1f\n", 
		(double) ir / deltasec,
		(double) or / deltasec,
		(double) im / deltasec,
		(double) om / deltasec,
		(double) id / deltasec,
		(double) ro / deltasec,
		(double) fc / deltasec);

	return(1);
}

void
IPv6head(struct osrel *osr)
{
	printf("in: dsc/s hder/s ader/s unkp/s ratim/s rfail/s "
	       "ot: dsc/s nort/s _ip6_\n");
}

int
IPv6line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	ihe, iae, iup, ids, ods, onr, rto, rfl;
	struct netstat 	*cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	ids = cur->ip6.Ip6InDiscards  		- pre->ip6.Ip6InDiscards;
	ihe = cur->ip6.Ip6InHdrErrors  		- pre->ip6.Ip6InHdrErrors;
	iae = cur->ip6.Ip6InAddrErrors 		- pre->ip6.Ip6InAddrErrors;
	iup = cur->ip6.Ip6InUnknownProtos	- pre->ip6.Ip6InUnknownProtos;
	rto = cur->ip6.Ip6ReasmTimeout  	- pre->ip6.Ip6ReasmTimeout;
	rfl = cur->ip6.Ip6ReasmFails  		- pre->ip6.Ip6ReasmFails;
	ods = cur->ip6.Ip6OutDiscards  		- pre->ip6.Ip6OutDiscards;
	onr = cur->ip6.Ip6OutNoRoutes  		- pre->ip6.Ip6OutNoRoutes;

	printf("    %5.1f %6.1f %6.1f %6.1f %7.1f %7.1f     %5.1f %6.1f\n", 
		(double) ids / deltasec,
		(double) ihe / deltasec,
		(double) iae / deltasec,
		(double) iup / deltasec,
		(double) rto / deltasec,
		(double) rfl / deltasec,
		(double) ods / deltasec,
		(double) onr / deltasec);

	return(1);
}


/*
** TCP version 6 statistics
*/
void
tcpv6head(struct osrel *osr)
{
	printf("                                       "
	       "       socknow sockmax  _tcp6_\n"         );
}

int
tcpv6line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long		sn=0, sm=0;
	struct sockstat 	*scur, *spre;

	if ( getcset("sockstat", (void **)&scur, (void **)&spre) )
	{
		sn = scur->tcp6now;
		sm = scur->tcp6max;
	}

	printf("                                       "
	       "       %7d %7d\n", sn, sm);

	return(1);
}

/*
** UDP version 6 statistics
*/
void
udpv6head(struct osrel *osr)
{
	printf("indgram/s otdgram/s   inerr/s  noport/s    "
	       "   socknow sockmax  _udp6_\n"         );
}

int
udpv6line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	id, ot, ie, np, sn=0, sm=0;
	struct netstat 	*cur,  *pre;
	struct sockstat	*scur, *spre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	if ( getcset("sockstat", (void **)&scur, (void **)&spre) )
	{
		sn = scur->udp6now;
		sm = scur->udp6max;
	}

	id = cur->udp6.UdpInDatagrams  - pre->udp6.UdpInDatagrams;
	ot = cur->udp6.UdpOutDatagrams - pre->udp6.UdpOutDatagrams;
	ie = cur->udp6.UdpInErrors     - pre->udp6.UdpInErrors;
	np = cur->udp6.UdpNoPorts      - pre->udp6.UdpNoPorts;

	printf("%9.1f %9.1f   %7.2f %9.2f       %7d %7d\n",
		(double) id / deltasec,
		(double) ot / deltasec,
		(double) ie / deltasec,
		(double) np / deltasec,
		sn, sm);

	return(1);
}

/*
** ICMP version 6 statistics
*/
void
icmpv6head(struct osrel *osr)
{
	printf("intot/s outtot/s inerr/s innsol/s innadv/s "
	       "otnsol/s otnadv/s  _icmp6_\n"   );
}

int
icmpv6line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	it, ot, ier, ins, ina, ons, ona;
	struct netstat  *cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	it  = cur->icmp6.Icmp6InMsgs  	- pre->icmp6.Icmp6InMsgs;
	ot  = cur->icmp6.Icmp6OutMsgs 	- pre->icmp6.Icmp6OutMsgs;
	ier = cur->icmp6.Icmp6InErrors 	- pre->icmp6.Icmp6InErrors;
	ins = cur->icmp6.Icmp6InNeighborSolicits
				- pre->icmp6.Icmp6InNeighborSolicits;
	ina = cur->icmp6.Icmp6InNeighborAdvertisements
				- pre->icmp6.Icmp6InNeighborAdvertisements;
	ons = cur->icmp6.Icmp6OutNeighborSolicits
				- pre->icmp6.Icmp6OutNeighborSolicits;
	ona = cur->icmp6.Icmp6OutNeighborAdvertisements
				- pre->icmp6.Icmp6OutNeighborAdvertisements;

	printf("%7.1f %8.1f %7.2f %8.2f %8.2f %8.2f %8.2f\n", 
		(double) it  / deltasec, 
		(double) ot  / deltasec,
		(double) ier / deltasec,
		(double) ins / deltasec,
		(double) ina / deltasec,
		(double) ons / deltasec,
		(double) ona / deltasec);

	return(1);
}


void
ICMPv6head(struct osrel *osr)
{
	printf("iecho/s ierep/s oerep/s idu/s odu/s ird/s ord/s ite/s "
	       "ote/s   _icmp6_\n");
}

int
ICMPv6line(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	unsigned long	iec, ier, oer, idu, odu, ird, ord, ite, ote;
	struct netstat  *cur, *pre;

	if ( !getcset("netstat", (void **)&cur, (void **)&pre) )
		return(0);

	iec = cur->icmp6.Icmp6InEchos         - pre->icmp6.Icmp6InEchos;
	ier = cur->icmp6.Icmp6InEchoReplies   - pre->icmp6.Icmp6InEchoReplies;
	oer = cur->icmp6.Icmp6OutEchoReplies  - pre->icmp6.Icmp6OutEchoReplies;
	idu = cur->icmp6.Icmp6InDestUnreachs  - pre->icmp6.Icmp6InDestUnreachs;
	odu = cur->icmp6.Icmp6OutDestUnreachs - pre->icmp6.Icmp6OutDestUnreachs;
	ird = cur->icmp6.Icmp6InRedirects     - pre->icmp6.Icmp6InRedirects;
	ord = cur->icmp6.Icmp6OutRedirects    - pre->icmp6.Icmp6OutRedirects;
	ite = cur->icmp6.Icmp6InTimeExcds     - pre->icmp6.Icmp6InTimeExcds;
	ote = cur->icmp6.Icmp6OutTimeExcds   	- pre->icmp6.Icmp6OutTimeExcds;

	printf("%7.2f %7.2f %7.2f %5.2f %5.2f %5.2f %5.2f %5.2f %5.2f\n", 
		(double) iec / deltasec,
		(double) ier / deltasec,
		(double) oer / deltasec,
		(double) idu / deltasec,
		(double) odu / deltasec,
		(double) ird / deltasec,
		(double) ord / deltasec,
		(double) ite / deltasec,
		(double) ote / deltasec);

	return(1);
}



/*
** Disk statistics
*/
void
dkhead(struct osrel *osr)
{
	printf("device            read/s rdKb/s   "
	       "write/s wrKb/s        rdwr/s _disk_\n");
}

int
dkline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register int	i, j;
	unsigned long	lin, rw, rn, rb, wn, wb;
	struct genstat	*cur,  *pre;

	/*
	** obtain the general statistics
	*/
	if ( !getcset("genstat", (void **)&cur, (void **)&pre) )
		return(0);

	for (i=j=lin=0; i < MAXDISK &&
		(cur->dk_drive_maj[i] || cur->dk_drive_min[i]); i++, j++)
	{
		if (cur->dk_drive_tot[i] == 0)
			continue;

		/*
		** note that only the active disks are shown;
		** when a disk has been used for the first time in
		** the last sample, the position of the existing statistics
		** in the table might have been shifted related to the
		** previous sample
		*/
		if ( cur->dk_drive_maj[i] != pre->dk_drive_maj[j] ||
		     cur->dk_drive_min[i] != pre->dk_drive_min[j]   )
		{
			/* resync */
			for (j=0; j < MAXDISK; j++)
			{
				if (cur->dk_drive_maj[i] ==
						pre->dk_drive_maj[j] &&
		     		    cur->dk_drive_min[i] ==
						pre->dk_drive_min[j]   )
					break;
			}
		}

		if ( (rw = cur->dk_drive_tot[i]  - pre->dk_drive_tot[j]) <= 0)
			continue;

		rn = cur->dk_drive_rio[i]  - pre->dk_drive_rio[j];
		wn = cur->dk_drive_wio[i]  - pre->dk_drive_wio[j];
		rb = cur->dk_drive_rblk[i] - pre->dk_drive_rblk[j];
		wb = cur->dk_drive_wblk[i] - pre->dk_drive_wblk[j];

		if (lin > 0)
			printf("%s  ", tstamp);

		printf("disk%03d-%03d   %10.2f %6.2f   %7.2f %6.2f    %10.2f\n",
			cur->dk_drive_maj[i], cur->dk_drive_min[i],
			(double) rn / deltasec,
			(double) rb / deltasec / 2.0,
			(double) wn / deltasec,
			(double) wb / deltasec / 2.0,
			(double) rw / deltasec);

		lin++;
	}

	if (lin == 0)
		printf("\n");

	return(1);
}

/*
** Disk-partition statistics
*/
void
parthead(struct osrel *osr)
{
	printf("partition   busy  read/s Kbyt/r   "
	       "write/s Kbyt/w  avque avserv _part_\n");
}

int
partline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register int	i, nr, lin;
	unsigned long	io_msecs, rd, wr, rs, ws;
	double		busy, avq, avs, avkr, avkw;
	char		lastchar, dskname[32];
	struct dkstat 	*cur,  *pre;
	unsigned long	deltams = deltatic * 1000 / hz;


	if ( (nr = getcset("partstat", (void **)&cur, (void **)&pre)) == 0)
		return(0);

	for (i=0, lin=0; i < nr; i++, cur++, pre++)
	{
		if ( cur->readblocks == pre->readblocks &&
		     cur->writblocks == pre->writblocks   )
			continue;	/* no activity on this partition */

		if ( (cur->major != pre->major || cur->minor != pre->minor) &&
		     (pre->major != 0          || pre->minor != 0         )   )
			continue;	/* probably a new partition popped up during interval */

		/*
		** this partition was active during last interval
		*/
		if (lin > 0)
			printf("%s  ", tstamp);

		rd = cur->readblocks  - pre->readblocks;
		wr = cur->writblocks  - pre->writblocks;
		rs = cur->readsectors - pre->readsectors;
		ws = cur->writsectors - pre->writsectors;

		if (rd > 0)
			avkr = (double) rs / rd / 2.0;
		else
			avkr = 0.0;

		if (wr > 0)
			avkw = (double) ws / wr / 2.0;
		else
			avkw = 0.0;

		lastchar = cur->name[strlen(cur->name)-1]; 

		/*
		** print info for entire disk
		*/
		if ( !isdigit(lastchar) )
		{
			io_msecs = cur->rdwr_msecs - pre->rdwr_msecs;

			if ( (busy = io_msecs * 100.0 / deltams) > 100.0)
				busy = 100.0;

			avs = (double) io_msecs / (rd+wr);

			if (io_msecs > 0)
				avq = (cur->avq - pre->avq) / (double) io_msecs;
			else
				avq = 0.0;

			sprintf(dskname, "%s (%d-%d)", cur->name, cur->major, cur->minor);

			printf("%-11s %3.0f%% %7.2lf %6.1lf %9.2lf %6.1lf %6.2lf %6.2lf ms\n",
				dskname,
				busy,
				(double) rd * 1000.0 / deltams,
				avkr,
				(double) wr * 1000.0 / deltams,
				avkw,
				avq,
				avs);
		}
		/*
		** print info for a partition
		*/
		else
		{
			printf("%-11s      %7.2lf %6.1lf %9.2lf %6.1lf\n",
				cur->name,
				(double) rd * 1000.0 / deltams,
				avkr,
				(double) wr * 1000.0 / deltams,
				avkw);
		}

		lin++;
	}

	if (lin == 0)
		printf("\n");

	return(1);
}

/*
** NFS general statistics
*/
void
nfshead(struct osrel *osr)
{
	printf("svrpc/s clrpc/s   sudp/s stcp/s stcon/s   "
	       "cudp/s ctcp/s ctcon/s _nfs_\n");
}

int
nfsline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	struct nfsstat 	*cur, *pre;
	unsigned long	sr, sn, su, st, sc, cr, cn, cu, ct, cc;

	if ( !getcset("nfsstat", (void **)&cur, (void **)&pre) )
		return(0);

	sr  = cur->sv_rpccnt      - pre->sv_rpccnt;
	sn  = cur->sv_netcnt      - pre->sv_netcnt;
	su  = cur->sv_netudpcnt   - pre->sv_netudpcnt;
	st  = cur->sv_nettcpcnt   - pre->sv_nettcpcnt;
	sc  = cur->sv_nettcpconn  - pre->sv_nettcpconn;

	cr  = cur->cl_rpccnt      - pre->cl_rpccnt;
	cn  = cur->cl_netcnt      - pre->cl_netcnt;
	cu  = cur->cl_netudpcnt   - pre->cl_netudpcnt;
	ct  = cur->cl_nettcpcnt   - pre->cl_nettcpcnt;
	cc  = cur->cl_nettcpconn  - pre->cl_nettcpconn;

	printf("%7.2f %7.2f %8.2f %6.2f %7.3f %8.2f %6.2f %7.3f\n", 
		(double) sr / deltasec,
		(double) cr / deltasec,
		(double) su / deltasec,
		(double) st / deltasec,
		(double) sc / deltasec,
		(double) cu / deltasec,
		(double) ct / deltasec,
		(double) cc / deltasec);

	return(1);
}

/*
** NFS error statistics
*/
void
NFShead(struct osrel *osr)
{
	printf("svbadfmt/s svbadauth/s svbadclnt/s  "
	       "clretrans/s clauthrefresh/s _nfs_\n");
}

int
NFSline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	struct nfsstat 	*cur, *pre;
	unsigned long	sf, sa, sc, cr, ca;

	if ( !getcset("nfsstat", (void **)&cur, (void **)&pre) )
		return(0);

	sf  = cur->sv_rpcbadfmt       - pre->sv_rpcbadfmt;
	sa  = cur->sv_rpcbadauth      - pre->sv_rpcbadauth;
	sc  = cur->sv_rpcbadclnt      - pre->sv_rpcbadclnt;
	cr  = cur->cl_rpcretrans      - pre->cl_rpcretrans;
	ca  = cur->cl_rpcauthrefresh  - pre->cl_rpcauthrefresh;

	printf("%10.3f %11.3f %11.3f  %11.3f %15.3f\n", 
		(double) sf / deltasec,
		(double) sa / deltasec,
		(double) sc / deltasec,
		(double) cr / deltasec,
		(double) ca / deltasec);

	return(1);
}

/*
** NFS additional server statistics
*/
void
nfsshead(struct osrel *osr)
{
	printf("rchit/s rcmiss/s %%hit fhstal/s iord/s iowr/s "
	       "racach thr tlast/s _nfs_\n");
}

int
nfssline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	struct nfsstat 	*cur, *pre;
	unsigned long	rch, rcm, rcn, fst, flu, fan, fnd, fnn, ior, iow;
	unsigned long	thc, thl, ras;
	float		perc;

	if ( !getcset("nfsstat", (void **)&cur, (void **)&pre) )
		return(0);

	rch = cur->sv_rchits          - pre->sv_rchits;
	rcm = cur->sv_rcmisses        - pre->sv_rcmisses;
	rcn = cur->sv_rcnocache       - pre->sv_rcnocache;
	fst = cur->sv_fhstale         - pre->sv_fhstale;
	flu = cur->sv_fhlookup        - pre->sv_fhlookup;
	fan = cur->sv_fhanon          - pre->sv_fhanon;
	fnd = cur->sv_fhnocachedir    - pre->sv_fhnocachedir;
	fnn = cur->sv_fhnocachenondir - pre->sv_fhnocachenondir;
	ior = cur->sv_ioread          - pre->sv_ioread;
	iow = cur->sv_iowrite         - pre->sv_iowrite;
	thc = cur->sv_thcount;
	thl = cur->sv_thlastcnt       - pre->sv_thlastcnt;
	ras = cur->sv_rasize;

	if (rch+rcm > 0)
		perc = (double) rch * 100 / (rch + rcm);
	else
		perc = 0.0;

	printf("%7.2f %8.2f %4.0f %8.3f %5.1fK %5.1fK %5dK %3d %7.3f\n",
		(double) rch / deltasec,
		(double) rcm / deltasec,
		         perc,
		(double) fst / deltasec,
		(double) ior / deltasec / 1024,
		(double) iow / deltasec / 1024,
		         ras / 1024,
		         thc,
		(double) thl / deltasec);
}


/*
** RPC statistics
*/
void
rpchead(struct osrel *osr)
{
	printf("gat sat lku rln  rd wrc  wr cre  rm "
	       "rnm lnk sln mkd rmd rdd fst _rpc_\n");
}

int
rpcline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register int	i, lin=0;
	struct nfsstat 	*cur, *pre;
	double 		stot, ctot;

	if ( !getcset("nfsstat", (void **)&cur, (void **)&pre) )
		return(0);

	stot = cur->sv_rpccnt - pre->sv_rpccnt;
	ctot = cur->cl_rpccnt - pre->cl_rpccnt;

	/*
	** print server rpc-call statistics (version 2 and 3)
	*/
	if (stot > 0)
	{
	 	lin += prirpcs(cur->sv_proc2, pre->sv_proc2,
					 stot, "%srv2", lin, tstamp);

	 	lin += prirpcs(cur->sv_proc3, pre->sv_proc3,
					 stot, "%srv3", lin, tstamp);
	}

	/*
	** print client rpc-call statistics (version 2 and 3)
	*/
	if (ctot > 0)
	{
	 	lin += prirpcs(cur->cl_proc2, pre->cl_proc2,
					 ctot, "%cln2", lin, tstamp);

	 	lin += prirpcs(cur->cl_proc3, pre->cl_proc3,
					 ctot, "%cln3", lin, tstamp);
	}

	if (lin == 0)
		printf("\n");

	return(1);
}

/*
** print one line with percentages about RPC-call usage
*/
#define	NNULL            0
#define	NGETATTR         1
#define	NSETATTR         2
#define	NROOT            3
#define	NLOOKUP          4
#define	NREADLINK        5
#define	NREAD            6
#define	NWRITEC          7
#define	NWRITE           8
#define	NCREATE          9
#define	NREMOVE          10
#define	NRENAME          11
#define	NLINK            12
#define	NSYMLINK         13
#define	NMKDIR           14
#define	NRMDIR           15
#define	NREADDIR         16
#define	NSTATFS          17
#define	NNUMRPC          18

int
prirpcs(unsigned long rpccur[], unsigned long rpcpre[],
			double tot, char *txt, int lines, char *tstamp)
{
	register int	i, sub;

	for (i=0, sub=0; i < NNUMRPC; i++)
		sub += rpccur[i] - rpcpre[i];

	if (sub == 0)
		return 0;

	if (lines)
		printf("%s  ", tstamp);

	printf("%3.0f %3.0f %3.0f %3.0f %3.0f %3.0f %3.0f %3.0f "
               "%3.0f %3.0f %3.0f %3.0f %3.0f %3.0f %3.0f %3.0f %s\n",
		(rpccur[NGETATTR]  - rpcpre[NGETATTR]) * 100 / tot,
		(rpccur[NSETATTR]  - rpcpre[NSETATTR]) * 100 / tot,
		(rpccur[NLOOKUP]   - rpcpre[NLOOKUP])  * 100 / tot,
		(rpccur[NREADLINK] - rpcpre[NREADLINK])* 100 / tot,
		(rpccur[NREAD]     - rpcpre[NREAD])    * 100 / tot,
		(rpccur[NWRITEC]   - rpcpre[NWRITEC])  * 100 / tot,
		(rpccur[NWRITE]    - rpcpre[NWRITE])   * 100 / tot,
		(rpccur[NCREATE]   - rpcpre[NCREATE])  * 100 / tot,
		(rpccur[NREMOVE]   - rpcpre[NREMOVE])  * 100 / tot,
		(rpccur[NRENAME]   - rpcpre[NRENAME])  * 100 / tot,
		(rpccur[NLINK]     - rpcpre[NLINK])    * 100 / tot,
		(rpccur[NSYMLINK]  - rpcpre[NSYMLINK]) * 100 / tot,
		(rpccur[NMKDIR]    - rpcpre[NMKDIR])   * 100 / tot,
		(rpccur[NRMDIR]    - rpcpre[NRMDIR])   * 100 / tot,
		(rpccur[NREADDIR]  - rpcpre[NREADDIR]) * 100 / tot,
		(rpccur[NSTATFS]   - rpcpre[NSTATFS])  * 100 / tot,
		txt);

	return 1;
}


#ifdef	WEBSUPPORT
/*
** FTP statistics
*/
void
ftphead(struct osrel *osr)
{
	printf("xfers/s kbytes/s   avg_time  avg_kbytes direct name"
	       "             _ftp_\n");
}

int
ftpline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	static int		sampcnt = 0;
	register int	i, n, lines;
	double			xfers;
	struct ftpstat 	*cur, *pre;

	extern char	*infile;        /* input file-name            */
	extern char	*interval;      /* number-of-seconds interval */

	/*
	** not supported when reading from pipe, i.e. starting with interval
	*/
	if (!infile && interval != NULL)
	{
		printf("interval for  ftp-counters:");
		return 0;
	}

	if ( (n = getcset("ftpstat", (void **)&cur, (void **)&pre)) == 0)
		return 0;

	sampcnt++;

	for (i=0, lines=0; i < n; i++, cur++, pre++)
	{
		/*
		** check output-activity
		*/
		if ( sampcnt == 1 || (xfers = cur->ocumxfer - pre->ocumxfer) > 0.0)
		{
			if (lines > 0)
				printf("%s  ", tstamp);
	
			printf("%7.3f %8.2f %8.2f s %11.3f output %s\n",
				(cur->ocumxfer    - pre->ocumxfer)     /deltasec,
				(cur->ocumkbytes  - pre->ocumkbytes)   /deltasec,
				(cur->ocumseconds - pre->ocumseconds)  /xfers,
				(cur->ocumkbytes  - pre->ocumkbytes)   /xfers,
				 cur->logname);
	
			lines++;
		}

		/*
		** check input-activity
		*/
		if ( sampcnt == 1 || (xfers = cur->icumxfer - pre->icumxfer) > 0.0)
		{
			if (lines > 0)
				printf("%s  ", tstamp);

			printf("%7.3f %8.2f %8.2f s %11.3f  input %s\n",
				(cur->icumxfer    - pre->icumxfer)    /deltasec,
				(cur->icumkbytes  - pre->icumkbytes)  /deltasec,
				(cur->icumseconds - pre->icumseconds) /xfers,
				(cur->icumkbytes  - pre->icumkbytes)  /xfers,
				 cur->logname);

			lines++;
		}
	}

	if (lines == 0)
		printf("\n");
	
	return 1;
}


/*
** HTTP statistics
*/
void
httphead(struct osrel *osr)
{
	printf(" head/s   get/s   put/s  post/s delete/s  name"
	       "                 _http_\n");
}

int
httpline(time_t deltasec, time_t deltatic, time_t hz,
				int ncpu, struct osrel *osr, char *tstamp)
{
	register int	i, n, lines;
	struct httpstat *cur, *pre;
	double 			head, gets, puts, post, dels;
	static int		sampcnt = 0;

	extern char	*infile;        /* input file-name            */
	extern char	*interval;      /* number-of-seconds interval */

	/*
	** not supported when reading from pipe, i.e. starting with interval
	*/
	if (!infile && interval != NULL)
	{
		printf("interval for http-counters:");
		return 0;
	}

	if ( (n = getcset("httpstat", (void **)&cur, (void **)&pre)) == 0)
		return 0;

	sampcnt++;

	for (i=0, lines=0; i < n; i++, cur++, pre++)
	{
		head = cur->cumhead - pre->cumhead;
		gets = cur->cumgets - pre->cumgets;
		puts = cur->cumputs - pre->cumputs;
		post = cur->cumpost - pre->cumpost;
		dels = cur->cumdels - pre->cumdels;

		if (sampcnt == 1 || head || gets || puts || post || dels)
		{
			if (lines > 0)
				printf("%s  ", tstamp);

			printf("%7.3f %7.3f %7.3f %7.3f %8.3f  %s\n",
				head  / deltasec,
				gets  / deltasec,
				puts  / deltasec,
				post  / deltasec,
				dels  / deltasec,
				cur->logname);

			lines++;
		}
	}

	if (lines == 0)
		printf("\n");

	return 1;
}
#endif

/*********************************************************************/
/* Function definition table.                                        */
/*                                                                   */
/* The layout of this table is as follows:                           */
/*     Column 1:                                                     */
/*        Boolean which indicates if the specified function is       */
/*        active during a run of 'atsar'. When 'atsar' is started,   */
/*        this boolean will be defined 'true' for all entries for    */
/*        which the command-line flag has been specified. Initially  */
/*        this column should contain 0 (false), unless this function */
/*        is always required.                                        */
/*        If no flags are specified for 'atsar', the first entry     */
/*        in this table is defined active (default flag).            */
/*                                                                   */
/*     Column 2:                                                     */
/*        Flag which can be used as command-line argument to         */
/*        select the function defined in this table-entry. Be sure   */
/*        that a unique character is choosen.                        */
/*        The flags 'f', 's', 'e', 'i' and 'A' are reserved!!        */
/*                                                                   */
/*     Column 3:                                                     */
/*        Entry-point of the 'printhead' function.                   */
/*                                                                   */
/*     Column 4:                                                     */
/*        Entry-point of the 'printline' function.                   */
/*                                                                   */
/*     Column 5:                                                     */
/*        Information about the statistics shown by the function     */
/*        specified by the table-entry. This text is printed as      */
/*        command-usage.                                             */
/*********************************************************************/
struct funcdef funcdef[] =
{
	0,  'u',	cpuhead,	cpuline,  	"cpu",
	0,  'd',	dkhead,		dkline,		"disk",
	0,  'D',	parthead,	partline,	"disk-partition",
	0,  'r',	memhead,	memline,	"memory & swap",
	0,  'p',	pagehead,	pageline,	"paging & swapping",
	0,  'I',	inthead,	intline,	"interrupts",
	0,  'v',	tabhead,	tabline,	"kernel-resources",
	0,  'y',	ttyhead,	ttyline,	"tty activity",
	0,  'l',	ifhead,		ifline,		"net-interf (general)",
	0,  'L',	IFhead,		IFline,		"net-interf (errors)",
	0,  'w',	ipv4head,	ipv4line,	"ip   v4    (general)",
	0,  'W',	IPv4head,	IPv4line,	"ip   v4    (errors)",
	0,  't',	tcphead,	tcpline,  	"tcp  v4    (general)",
	0,  'T',	TCPhead,	TCPline,  	"tcp  v4    (errors)",
	0,  'U',	udpv4head,	udpv4line,  	"udp  v4",
	0,  'm',	icmpv4head,	icmpv4line,	"icmp v4    (general)",
	0,  'M',	ICMPv4head,	ICMPv4line,	"icmp v4    (per type)",
	0,  'g',	ipv6head,	ipv6line,	"ip   v6    (general)",
	0,  'G',	IPv6head,	IPv6line,	"ip   v6    (errors)",
	0,  'j',	tcpv6head,	tcpv6line,  	"tcp  v6    (general)",
	0,  'h',	udpv6head,	udpv6line,  	"udp  v6",
	0,  'k',	icmpv6head,	icmpv6line,	"icmp v6    (general)",
	0,  'K',	ICMPv6head,	ICMPv6line,	"icmp v6    (per type)",
	0,  'N',	nfshead,	nfsline,	"nfs        (general)",
	0,  'E',	NFShead,	NFSline,	"nfs        (errors)",
	0,  'V',	nfsshead,	nfssline,	"nfs        (server)",
	0,  'R',	rpchead,	rpcline,	"nfs-rpc    (%calls)",

#ifdef	WEBSUPPORT
	0,  'F',	ftphead,	ftpline,	"ftp  traffic",
	0,  'H',	httphead,	httpline,	"http traffic",
#endif
};

int	funcnt = sizeof(funcdef)/sizeof(struct funcdef);
