/** \file src/type-dialog.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "type-dialog.hh"

#include <gtkmm/accelgroup.h>
#include <gtkmm/accelkey.h>
#include <gtkmm/cellview.h>
#include <gtkmm/combobox.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/icontheme.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/scale.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/stock.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
class OneUstring : public EditableColumnsBase {
/******************************************************************************/

public:
	static OneUstring& get() { static OneUstring the; return the; }

	Gtk::TreeModelColumn<ustring> it;

private:

	OneUstring()
	:	EditableColumnsBase(), it()
	{ add(it); }

}; /* class OneUstring */

/******************************************************************************/
class MagicColumns : public EditableColumnsBase {
/******************************************************************************/

public:

	static MagicColumns& get() { static MagicColumns the; return the; }
	
	Gtk::TreeModelColumn<const MagicMatch*> obj;
	Gtk::TreeModelColumn<const Magic*> group;
	Gtk::TreeModelColumn<ustring> type, value, mask, offset, priority;

private:

	MagicColumns()
	:	EditableColumnsBase(), obj(), group(),
		type(), value(), mask(), offset(), priority()
	{
		add(obj); add(group);
		add(type); add(value); add(mask); add(offset); add(priority);
	}

}; /* class MagicColumns */

/******************************************************************************/
class XMLColumns : public EditableColumnsBase {
/******************************************************************************/

public:

	static XMLColumns& get() { static XMLColumns the; return the; }
	
	Gtk::TreeModelColumn<ustring> namespace_uri, local_name;

private:

	XMLColumns()
	: EditableColumnsBase(), namespace_uri(), local_name()
	{	add(namespace_uri); add(local_name); }

}; /* class XMLColumns */

/******************************************************************************/
/* class TypeDialog                                                           */
/******************************************************************************/

TypeDialog::TypeDialog(const RefPtr<MimeDatabase>& database,
	MimeType* initial_type, Location target, bool renameable)
:	Gtk::Window(), m_updating(0),
	m_locations(NO_LOCATION), m_target(target),
	m_fixed_type(NULL), m_target_type(NULL),
	m_database(database), m_buttons(Gtk::BUTTONBOX_END, 12), s_changed(),
	m_notebook(), m_category_widget(NULL), m_category_id(), m_name_widget(NULL),
	m_description_entry(), m_location(), m_icon(),
	m_default_icon(), m_default_icon_select("..."),
	m_alias_store(Gtk::ListStore::create(OneUstring::get())),
	m_superclass_store(Gtk::ListStore::create(OneUstring::get())),
	m_glob_store(Gtk::ListStore::create(OneUstring::get())),
	m_magic_store(Gtk::TreeStore::create(MagicColumns::get())),
	m_xml_store(Gtk::ListStore::create(XMLColumns::get())),
	m_alias_view(m_alias_store, _("A_liases:")),
	m_superclass_view(m_superclass_store, _("_Parent types:")),
	m_glob_view(m_glob_store),
	m_magic_view(m_magic_store),
	m_xml_view(m_xml_store)
{
	set_border_width(12);
	set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG);
	set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
	
	Gtk::VBox *vbox = new Gtk::VBox(false, 12);
	add(*Gtk::manage(vbox));
	
	vbox->pack_start(m_notebook, true, true);
	
	Gtk::VBox *general = new Gtk::VBox(false, 12);
	general->set_border_width(12);
	m_notebook.append_page(*Gtk::manage(general), _("General"));
	
	Gtk::HBox *general_upper = new Gtk::HBox(false, 12);
	general->pack_start(*Gtk::manage(general_upper), false, false);

	m_icon.set_alignment(Gtk::ALIGN_LEFT, Gtk::ALIGN_TOP);
	general_upper->pack_start(m_icon, false, false);

	InfoTable *general_info = new InfoTable();
	general_upper->pack_start(*Gtk::manage(general_info), true, true);

	if (renameable) {
		Gtk::ComboBox *cb = new Gtk::ComboBox(CategoriesStore::get_selector());
		cb->signal_changed().connect
			(sigc::mem_fun(*this, &TypeDialog::on_category_changed));
		m_category_widget = cb;
		Gtk::Entry *entry = new Gtk::Entry();
		entry->signal_changed().connect
			(sigc::mem_fun(*this, &TypeDialog::on_name_changed));
		m_name_widget = entry;
	} else {
		Gtk::CellView *cv = new Gtk::CellView();
		cv->set_model(CategoriesStore::get_selector());
		m_category_widget = cv;
		Gtk::Label *label = new Gtk::Label(ustring(), Gtk::ALIGN_LEFT);
		label->set_selectable(true);
		m_name_widget = label;
	}
	m_category_widget->pack_start(CategoriesColumns::get().icon, false);
	m_category_widget->pack_start(CategoriesColumns::get().name, true);
	general_info->add_item(_("C_ategory:"),
		*CAST(m_category_widget, Gtk::Widget));
	
	Gtk::HBox *name_hbox = new Gtk::HBox(false, 0);
	m_category_id.set_selectable(true);
	name_hbox->pack_start(m_category_id, false, false);
	name_hbox->pack_start(*m_name_widget, true, true);
	general_info->add_item(_("_Name:"), *Gtk::manage(name_hbox),
		*m_name_widget);

	m_description_entry.signal_changed().connect
		(sigc::mem_fun(*this, &TypeDialog::on_description_changed));
	general_info->add_item(_("_Description:"), m_description_entry);

	m_location.set_selectable(true);
	m_location.set_alignment(Gtk::ALIGN_LEFT, Gtk::ALIGN_CENTER);
	general_info->add_item(_("Location:"), m_location);
	
	Gtk::HBox *icon_hbox = new Gtk::HBox(false, 6);

	m_default_icon.set_selectable(true);
	m_default_icon.set_alignment(Gtk::ALIGN_LEFT, Gtk::ALIGN_CENTER);
	m_default_icon.set_ellipsize(Pango::ELLIPSIZE_START);
	icon_hbox->pack_start(m_default_icon, true, true);
	
	m_default_icon_select.signal_clicked().connect
		(sigc::mem_fun(*this, &TypeDialog::on_select_default_icon));
	icon_hbox->pack_start(m_default_icon_select, false, false);

	general_info->add_item(_("Default _icon:"), *Gtk::manage(icon_hbox),
		m_default_icon_select);

	Gtk::HBox *related = new Gtk::HBox(false, 12);
	related->set_border_width(12);
	m_notebook.append_page(*Gtk::manage(related), _("Related types"));
	
	m_alias_view.m_view.append_column(ustring(), OneUstring::get().it);
	m_alias_view.signal_add().connect(sigc::bind(sigc::mem_fun
		(*this, &TypeDialog::on_alias_changed), Gtk::TreeIter()));
	m_alias_view.signal_remove().connect(sigc::mem_fun
		(*this, &TypeDialog::on_alias_changed));
	related->pack_start(m_alias_view, true, true);

	m_superclass_view.m_view.append_column(ustring(), OneUstring::get().it);
	m_superclass_view.signal_add().connect(sigc::bind(sigc::mem_fun
		(*this, &TypeDialog::on_superclass_changed), Gtk::TreeIter()));
	m_superclass_view.signal_remove().connect(sigc::mem_fun
		(*this, &TypeDialog::on_superclass_changed));
	related->pack_start(m_superclass_view, true, true);

	m_glob_view.m_view.append_column(_("Filename pattern"),
		OneUstring::get().it);
	m_glob_view.signal_add().connect(sigc::bind(sigc::mem_fun
		(*this, &TypeDialog::on_glob_changed), Gtk::TreeIter()));
	m_glob_view.signal_remove().connect(sigc::mem_fun
		(*this, &TypeDialog::on_glob_changed));
	m_notebook.append_page(m_glob_view, _("Filenames"));

	MagicColumns &magic_cols = MagicColumns::get();
	m_magic_view.m_view.append_column(_("Type"), magic_cols.type);
	m_magic_view.m_view.append_column(_("Priority"), magic_cols.priority);
	m_magic_view.m_view.append_column(_("Value"), magic_cols.value);
	m_magic_view.m_view.append_column(_("Offset"), magic_cols.offset);
	m_magic_view.m_view.append_column(_("Mask"), magic_cols.mask);
	m_magic_view.signal_add().connect(sigc::bind(sigc::mem_fun
		(*this, &TypeDialog::on_magic_changed), Gtk::TreeIter()));
	m_magic_view.signal_remove().connect(sigc::mem_fun
		(*this, &TypeDialog::on_magic_changed));
	m_notebook.append_page(m_magic_view, _("File contents"));

	m_xml_view.m_view.append_column(_("Namespace URI"),
		XMLColumns::get().namespace_uri);
	m_xml_view.m_view.append_column(_("Root element"),
		XMLColumns::get().local_name);
	m_xml_view.signal_add().connect(sigc::bind(sigc::mem_fun
		(*this, &TypeDialog::on_xml_changed), Gtk::TreeIter()));
	m_xml_view.signal_remove().connect(sigc::mem_fun
		(*this, &TypeDialog::on_xml_changed));
	m_notebook.append_page(m_xml_view, _("XML elements"));
	
	vbox->pack_start(m_buttons, false, false);
	
	Gtk::Button *help = new Gtk::Button(Gtk::Stock::HELP);
	help->signal_clicked().connect(sigc::mem_fun(*this, &TypeDialog::on_help));
	m_buttons.pack_start(*Gtk::manage(help), false, false);
	m_buttons.set_child_secondary(*help, true);

	show_all_children();

	if (initial_type != NULL)
		set_type(*initial_type);
	else
		reset_type();
	
	m_database->get_Override_pkg().signal_icons_changed().connect
		(sigc::mem_fun(*this, &TypeDialog::update_icon));
}

TypeDialog::~TypeDialog()
{
	delete m_name_widget;
	delete m_category_widget;
	delete m_target_type;
	delete m_fixed_type;
}

ustring
TypeDialog::get_full_name() {
	Gtk::TreeIter category;
	if cast(m_category_widget, Gtk::ComboBox, combo)
		category = combo->get_active();
	else if cast(m_category_widget, Gtk::CellView, cv)
		category = CategoriesStore::get_selector()->
			get_iter(cv->get_displayed_row());

	ustring name;
	if cast(m_name_widget, Gtk::Entry, entry)
		name = entry->get_text();
	else if cast(m_name_widget, Gtk::Label, label)
		name = label->get_text();

	if (!category || name.empty()) return ustring();
	return category->get_value(CategoriesColumns::get().id) + "/" + name;
}

void
TypeDialog::set_type(MimeType& type)
{
	m_locations = type.get_locations();

	delete m_fixed_type;
	if (m_locations & ~m_target)
		m_fixed_type = type.limit_locations(~m_target);
	else
		m_fixed_type = new MimeType(type.m_type, type.m_subtype);

	delete m_target_type;
	if (m_locations & m_target)
		m_target_type = type.limit_locations(m_target);
	else
		m_target_type = new MimeType(type.m_type, type.m_subtype);

	update();
}

void
TypeDialog::reset_type()
{
	delete m_fixed_type;
	m_fixed_type = new MimeType(ustring(), ustring());
	delete m_target_type;
	m_target_type = new MimeType(ustring(), ustring());
	m_locations = NO_LOCATION;
	update();
}

sigc::signal<void>
TypeDialog::signal_changed()
{	return s_changed; }

int
TypeDialog::get_notebook_page() const
{	return m_notebook.get_current_page(); }

void
TypeDialog::set_indirect_sensitive(bool sensitive_)
{	m_default_icon_select.set_sensitive(sensitive_); }

void
TypeDialog::update()
{
	++m_updating;
	
	Gtk::TreeIter category;
	RefPtr<CategoriesStore> cats = CategoriesStore::get_selector();
	FOREACH_BASE(Gtk::TreeIter, cats->children(), i)
		if (i->get_value(cats->m_cols.id) == m_fixed_type->m_type) {
			category = i;
			break;
		}
	if (category) {
		if cast(m_category_widget, Gtk::ComboBox, cb)
			cb->set_active(category);
		else if cast(m_category_widget, Gtk::CellView, cv)
			cv->set_displayed_row(cats->get_path(category));
	}
	m_category_id.set_text(m_fixed_type->m_type + "/");

	if cast(m_name_widget, Gtk::Entry, entry)
		entry->set_text(m_fixed_type->m_subtype);
	else if cast(m_name_widget, Gtk::Label, label)
		label->set_text(m_fixed_type->m_subtype);

	Message combined_desc = m_fixed_type->m_description;
	FOREACH(Message, m_target_type->m_description, i)
		combined_desc[i->first] = i->second;
	m_description_entry.set_text(combined_desc.get_translated());

	m_location.set_text(get_location_name(m_locations));
	
	update_icon();

	m_alias_store->clear();
	m_superclass_store->clear();
	m_glob_store->clear();
	m_magic_store->clear();
	m_xml_store->clear();

	accumulate_lists(*m_fixed_type, false);
	accumulate_lists(*m_target_type, true);

	m_magic_view.m_view.expand_all();
	--m_updating;
}

void
TypeDialog::update_icon()
{
	++m_updating;

	m_icon.clear();
	Gtk::IconTheme::get_default()->rescan_if_needed();
	m_icon.set_from_icon_name(m_fixed_type->get_icon_name(),
		Gtk::ICON_SIZE_DIALOG);

	try {
		m_default_icon.set_text(Glib::filename_display_name
			(m_database->get_Override_pkg().get_default_icon(*m_target_type)));
	} catch (std::invalid_argument) {
		m_default_icon.set_text(_("None"));
	}
	
	--m_updating;
}

void
TypeDialog::on_help()
{
	switch (get_notebook_page()) {
	case 0: activate_help(*this, "assogiate-modifying-properties"); break;
	case 1: activate_help(*this, "assogiate-modifying-related"); break;
	case 2: activate_help(*this, "assogiate-detection-filename"); break;
	case 3: activate_help(*this, "assogiate-detection-contents"); break;
	case 4: activate_help(*this, "assogiate-detection-xml"); break;
	default: activate_help(*this); break;
	}
}

void
TypeDialog::accumulate_lists(MimeType& type, bool removable)
{
	CONST_FOREACH(std::list<ustring>, type.m_aliases, i) {
		Gtk::TreeIter j = m_alias_store->append();
		j->set_value(OneUstring::get().removable, removable);
		j->set_value(OneUstring::get().it, *i);
	}

	CONST_FOREACH(std::list<ustring>, type.m_superclasses, i) {
		Gtk::TreeIter j = m_superclass_store->append();
		j->set_value(OneUstring::get().removable, removable);
		j->set_value(OneUstring::get().it, *i);
	}
	
	CONST_FOREACH(std::list<ustring>, type.m_glob_patterns, i) {
		Gtk::TreeIter j = m_glob_store->append();
		j->set_value(OneUstring::get().removable, removable);
		j->set_value(OneUstring::get().it, *i);
	}

	CONST_FOREACH(std::list<Magic*>, type.m_magics, i)
		CONST_FOREACH(std::list<MagicMatch*>, (*i)->submatches, j)
			append_magic(Gtk::TreeIter(), **j, **i, removable);

	CONST_FOREACH(std::list<XMLRoot>, type.m_xml_roots, i) {
		Gtk::TreeIter j = m_xml_store->append();
		j->set_value(XMLColumns::get().removable, removable);
		j->set_value(XMLColumns::get().namespace_uri, i->namespace_uri);
		j->set_value(XMLColumns::get().local_name, i->local_name);
	}
}

void
TypeDialog::append_magic(const Gtk::TreeIter& iter, const MagicMatch& match,
	const Magic& group, bool removable)
{
	Gtk::TreeIter sub = iter ? m_magic_store->append(iter->children())
		: m_magic_store->append();
	MagicColumns &cols = MagicColumns::get();
	
	sub->set_value(cols.removable, removable);
	sub->set_value(cols.obj, &match);
	sub->set_value(cols.group, &group);

	RefPtr<MagicTypesStore> types = MagicTypesStore::get();
	FOREACH_BASE(Gtk::TreeIter, types->children(), i)
		if (i->get_value(types->m_cols.type) == match.type) {
			sub->set_value(cols.type, i->get_value(types->m_cols.name));
			break;
		}

	sub->set_value(cols.value, match.value);
	sub->set_value(cols.mask, match.mask);
	sub->set_value(cols.offset, (match.last_offset == -1)
		? compose::ucompose1(match.first_offset)
		/* Translators: This message represents a range between two values. */
		: compose::ucompose(_("%1-%2"), match.first_offset, match.last_offset));
	sub->set_value(cols.priority, compose::ucompose1(group.priority));

	CONST_FOREACH(std::list<MagicMatch*>, match.submatches, i)
		append_magic(sub, **i, group, false /* can't remove submatches yet */);
}

void
TypeDialog::on_category_changed()
{
	if (m_updating) return;
	m_target_type->m_type = CAST(m_category_widget, Gtk::ComboBox)->
		get_active()->get_value(CategoriesColumns::get().id);
	m_category_id.set_text(m_target_type->m_type + "/");
	s_changed.emit();
}

void
TypeDialog::on_name_changed()
{
	if (m_updating) return;
	m_target_type->m_subtype = CAST(m_name_widget, Gtk::Entry)->get_text();
	s_changed.emit();
}

void
TypeDialog::on_description_changed()
{
	if (m_updating) return;
	m_target_type->m_description.set_from_user(m_description_entry.get_text());
	s_changed.emit();
}

void
TypeDialog::on_select_default_icon()
{
	Gtk::FileChooserDialog fcd(_("Select Default Icon"));
	fcd.add_button(Gtk::Stock::CLEAR, Gtk::RESPONSE_NO);
	fcd.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	fcd.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_YES);
	fcd.set_default_response(Gtk::RESPONSE_YES);

	Gtk::FileFilter images;
	images.set_name(_("All compatible image files"));
	images.add_pattern("*.svg");
	images.add_pattern("*.png");
	images.add_pattern("*.xpm");
	fcd.add_filter(images);

	Gtk::FileFilter all;
	all.set_name(_("All files"));
	all.add_pattern("*");
	fcd.add_filter(all);

	try {
		fcd.set_filename(m_database->get_Override_pkg().
			get_default_icon(*m_target_type));
	} catch (std::invalid_argument) {
		fcd.unselect_all();
		fcd.set_response_sensitive(Gtk::RESPONSE_NO, false);
	}
	
	int response = fcd.run();
	fcd.hide();

	try {
		switch (response) {
		case Gtk::RESPONSE_YES:
			m_database->get_Override_pkg().set_default_icon(*m_target_type,
				fcd.get_filename());
			break;
		case Gtk::RESPONSE_NO:
			m_database->get_Override_pkg().unset_default_icon(*m_target_type);
			break;
		default:
			return;
		}
	} catch (std::invalid_argument& e) {
		Gtk::MessageDialog warn(*this, _("Default icon not valid"),
			false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
		warn.set_secondary_text(compose::ucompose(_("The selected file \"%1\" "
			"is not valid as a file type icon."),
			Glib::filename_display_name(fcd.get_filename())));
		warn.run();
	} catch (MimeDatabaseUpdateError& e) {
		show_database_update_warning(e, this);
	} catch (MimeDatabaseWriteError& e) {
		Gtk::MessageDialog warn(*this, _("Could not modify default icon"),
			false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
		warn.set_secondary_text(compose::ucompose(_("The default icon for "
			"%1 could not be modified. It may be helpful to retry.\n\n(%2)"),
			m_target_type->get_full_name(), e.what()));
		warn.run();
	}

	update_icon();
}


void
TypeDialog::one_ustring_common(const Gtk::TreeIter& iter, const ustring& title,
	const ustring& label, std::list<ustring>& list,
	const RefPtr<Gtk::ListStore>& store)
{
	if (m_updating) return;
	if (iter) {
		list.remove(iter->get_value(OneUstring::get().it));
		s_changed.emit();
	} else {
		AddItemDialog adder(this, title);
		Gtk::Entry entry; adder.add_item(label, entry, true);
		int response = adder.run(); adder.hide();
		if (response == Gtk::RESPONSE_OK) {
			ustring value = entry.get_text();
			list.push_back(value);

			Gtk::TreeIter j = store->append();
			j->set_value(OneUstring::get().removable, true);
			j->set_value(OneUstring::get().it, value);

			s_changed.emit();
		}
	}
}

void
TypeDialog::on_alias_changed(const Gtk::TreeIter& iter)
{	one_ustring_common(iter, _("Add Alias"), _("Alias:"),
		m_target_type->m_aliases, m_alias_store); }

void
TypeDialog::on_superclass_changed(const Gtk::TreeIter& iter)
{	one_ustring_common(iter, _("Add Parent Type"), _("Parent type:"),
		m_target_type->m_superclasses, m_superclass_store); }

void
TypeDialog::on_glob_changed(const Gtk::TreeIter& iter)
{	one_ustring_common(iter, _("Add Filename Rule"), _("Pattern:"),
		m_target_type->m_glob_patterns, m_glob_store); }

void
TypeDialog::on_magic_changed(const Gtk::TreeIter& iter)
{
	if (m_updating) return;
	if (iter) {
		Magic *old = const_cast<Magic*>(
			CAST(iter->get_value(MagicColumns::get().group), const Magic));
		if (old != NULL) {
			m_target_type->m_magics.remove(old);
			m_magic_view.m_view.expand_all();
			s_changed.emit();
		}
	} else {
		AddItemDialog adder(this, _("Add File Contents Rule"));

		Gtk::ComboBox type(MagicTypesStore::get());
		type.pack_start(MagicTypesColumns::get().name);
		adder.add_item(_("_Type:"), type, true);

		Gtk::HScale priority
			(*Gtk::manage(new Gtk::Adjustment(50.0, 0.0, 100.0)));
		priority.set_digits(0);
		priority.set_value_pos(Gtk::POS_LEFT);
		adder.add_item(_("_Priority:"), priority, true);

		Gtk::Entry thevalue;
		adder.add_item(_("_Value:"), thevalue, true);

		Gtk::HBox offsets(false, 6);

		offsets.pack_start(*Gtk::manage(new Gtk::Label(_("from"))),
			false, false);

		Gtk::SpinButton first_offset
			(*Gtk::manage(new Gtk::Adjustment(0.0, 0.0, INT_MAX)));
		first_offset.set_numeric(true);
		offsets.pack_start(first_offset, true, true);
		adder.add_required(first_offset);

		Gtk::CheckButton have_last_offset(_("to"));
		offsets.pack_start(have_last_offset, false, false);

		Gtk::SpinButton last_offset
			(*Gtk::manage(new Gtk::Adjustment(0.0, 0.0, INT_MAX)));
		last_offset.set_numeric(true);
		last_offset.set_sensitive(false);
		offsets.pack_start(last_offset, true, true);
		
		have_last_offset.signal_toggled().connect(sigc::bind(sigc::ptr_fun
			(&TypeDialog::update_sensitivity),
			&have_last_offset, &last_offset));

		offsets.show_all_children();
		adder.add_item(_("_Offsets:"), offsets, first_offset, false);

		Gtk::Entry mask;
		adder.add_item(_("_Mask:"), mask, false);

		int response = adder.run(); adder.hide();
		if (response == Gtk::RESPONSE_OK) {			
			MagicMatch *match = new MagicMatch();
			match->type = type.get_active()->
				get_value(MagicTypesColumns::get().type);
			match->value = thevalue.get_text();
			match->mask = mask.get_text();
			match->first_offset = first_offset.get_value_as_int();
			match->last_offset = have_last_offset.get_active()
				? last_offset.get_value_as_int() : -1;

			Magic *group = new Magic();
			group->priority = int(priority.get_value());
			group->submatches.push_back(match);
			m_target_type->m_magics.push_back(group);

			append_magic(Gtk::TreeIter(), *match, *group, true);

			m_magic_view.m_view.expand_all();
			s_changed.emit();
		}
	}
}

void
TypeDialog::on_xml_changed(const Gtk::TreeIter& iter)
{
	if (m_updating) return;
	if (iter) {
		XMLRoot old;
		old.namespace_uri = iter->get_value(XMLColumns::get().namespace_uri),
		old.local_name = iter->get_value(XMLColumns::get().local_name);
		m_target_type->m_xml_roots.remove(old);
		s_changed.emit();
	} else {
		AddItemDialog adder(this, _("Add XML Element Rule"));
		Gtk::Entry ns_uri, local_name;
		adder.add_item(_("_Namespace URI:"), ns_uri, true);
		adder.add_item(_("_Root element:"), local_name, false);
		int response = adder.run(); adder.hide();
		if (response == Gtk::RESPONSE_OK) {
			XMLRoot value;
			value.namespace_uri = ns_uri.get_text();
			value.local_name = local_name.get_text();
			m_target_type->m_xml_roots.push_back(value);

			Gtk::TreeIter j = m_xml_store->append();
			j->set_value(XMLColumns::get().removable, true);
			j->set_value(XMLColumns::get().namespace_uri, value.namespace_uri);
			j->set_value(XMLColumns::get().local_name, value.local_name);

			s_changed.emit();
		}
	}
}

void
TypeDialog::update_sensitivity(const Gtk::ToggleButton* source,
	Gtk::Widget* target)
{	if (source && target) target->set_sensitive(source->get_active()); }

/******************************************************************************/
/* class NewTypeDialog                                                        */
/******************************************************************************/

NewTypeDialog::NewTypeDialog(const RefPtr<MimeDatabase>& database)
:	TypeDialog(database, NULL, NO_LOCATION, true), m_new(Gtk::Stock::OK)
{
	set_title(_("New Type"));
	set_indirect_sensitive(false);

	Gtk::Button *cancel = new Gtk::Button(Gtk::Stock::CANCEL);
	cancel->add_accelerator("clicked", get_accel_group(), GDK_Escape,
		Gdk::ModifierType(0), Gtk::AccelFlags(0));
	cancel->signal_clicked().connect(sigc::mem_fun(*this, &Gtk::Window::hide));
	m_buttons.pack_start(*Gtk::manage(cancel), false, true);

	m_new.property_can_default() = true;
	m_new.set_sensitive(false);
	m_new.signal_clicked().connect
		(sigc::mem_fun(*this, &NewTypeDialog::on_new_type));
	m_buttons.pack_start(m_new, false, true);
	set_default(m_new);
	
	signal_changed().connect
		(sigc::mem_fun(*this, &NewTypeDialog::on_changed));

	show_all_children();
}

void
NewTypeDialog::on_help()
{
	if (get_notebook_page() == 0)
		activate_help(*this, "assogiate-creating-type");
	else
		TypeDialog::on_help();
}

void
NewTypeDialog::on_changed()
{
	bool sensitive_ = !get_full_name().empty();
	set_indirect_sensitive(sensitive_);
	if (sensitive_) update_icon();
	m_new.set_sensitive(sensitive_);
}

void
NewTypeDialog::on_new_type()
{	
	try {
		if (m_database->get_types().find(m_target_type->get_full_name()) !=
			m_database->get_types().end())
			throw MimeTypeExistsError();
		m_database->get_Override_pkg().add_type(*m_target_type);
	} catch (MimeTypeExistsError) {
		show_error(compose::ucompose(_("A file type with the name %1 already "
			"exists. Select a different name and try again."),
			m_target_type->get_full_name()));
		return;
	} catch (MimeDatabaseUpdateError& e) {
		show_database_update_warning(e, this);
	} catch (MimeDatabaseLoadError& e) {
		die_on_database_load_error(e);
	} catch (MimeDatabaseWriteError& e) {
		show_error(compose::ucompose(_("The file types database could not be "
			"modified. It may be helpful to retry.\n\n(%1)"), e.what()));
		return;
	}
	
	hide();
}

void
NewTypeDialog::show_error(const ustring& explanation)
{
	Gtk::MessageDialog warn(*this, _("Could not create type"), false,
		Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
	warn.set_secondary_text(explanation);
	warn.run();
}

/******************************************************************************/
/* class EditTypeDialog                                                       */
/******************************************************************************/

EditTypeDialog::EditTypeDialog(const RefPtr<MimeDatabase>& database,
	MimeType& type, Location target)
:	TypeDialog(database, &type, target,
		(type.get_locations() & ~target) == NO_LOCATION),
	m_throttle_changed(misc::Throttle::DELAY, 3.0), m_old_name()
{
	set_title(compose::ucompose(_("Edit Type %1"), type.get_full_name()));

	Gtk::Button *close = new Gtk::Button(Gtk::Stock::CLOSE);
	close->add_accelerator("clicked", get_accel_group(), GDK_Escape,
		Gdk::ModifierType(0), Gtk::AccelFlags(0));
	close->signal_clicked().connect(sigc::mem_fun(*this, &Gtk::Window::hide));
	m_buttons.pack_start(*Gtk::manage(close), false, true);

	signal_changed().connect(sigc::mem_fun(*this, &EditTypeDialog::on_changed));

	show_all_children();
}

EditTypeDialog::~EditTypeDialog()
{	m_throttle_changed.set(misc::Throttle::OPEN); }

void
EditTypeDialog::set_type(MimeType& type)
{
	m_old_name = type.get_full_name();
	set_title(compose::ucompose(_("Edit Type %1"), m_old_name));
	TypeDialog::set_type(type);
}

void
EditTypeDialog::on_changed()
{
	set_indirect_sensitive(!get_full_name().empty());
	m_throttle_changed.queue
		(sigc::mem_fun(*this, &EditTypeDialog::real_on_changed), true);
}

void
EditTypeDialog::real_on_changed()
{
	/* prevent redundant <comment/>s in Override.xml */
	FOREACH(Message, m_fixed_type->m_description, i) {
		Message::iterator j = m_target_type->m_description.find(i->first);
		if (j != m_target_type->m_description.end() && i->second == j->second)
			m_target_type->m_description.erase(i->first);
	}

	try {
		m_database->get_Override_pkg().replace_type(*m_target_type, m_old_name);
	} catch (MimeDatabaseUpdateError& e) {
		show_database_update_warning(e, this);
	} catch (MimeDatabaseLoadError& e) {
		die_on_database_load_error(e);
	} catch (MimeDatabaseWriteError& e) {
		Gtk::MessageDialog warn(*this, _("Could not modify type"), false,
			Gtk::MESSAGE_WARNING, Gtk::BUTTONS_CLOSE, true);
		warn.set_secondary_text(compose::ucompose(_("The file types database "
			"could not be modified. It may be helpful to retry.\n\n(%1)"),
			e.what()));
		warn.run();

		MimeTypeMap::iterator j = m_database->get_types().find(m_old_name);
		if (j != m_database->get_types().end())
			set_type(*j->second.first); /* restore real values */
	}
}

} /* namespace assoGiate */
