------------------------------------------------------------------------------
--                                                                          --
--                      GNAT METRICS TOOLS COMPONENTS                       --
--                                                                          --
--                 M E T R I C S . S O U R C E _ T A B L E                  --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                     Copyright (C) 2002-2008, AdaCore                     --
--                                                                          --
-- GNAT Metrics Toolset  is free software;  you can  redistribute it and/or --
-- modify it under terms of the  GNU General Public License as published by --
-- the Free Software Foundation;  either version 2, or (at your option) any --
-- later version.  GNAT Metrics Toolset is  distributed in the hope that it --
-- will be useful, but  WITHOUT ANY WARRANTY; without even the implied war- --
-- ranty of  MERCHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the --
-- GNU General Public License for more details.  You should have received a --
-- copy of the  GNU General Public License distributed with  GNAT; see file --
-- COPYING.  If not,  write to the  Free Software  Foundation,  51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- GNAT Metrics Toolset is maintained by AdaCore (http://www.adacore.com).  --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the source file table - the table containing the
--  information about the source files to be processed and the state of their
--  processing

with ASIS_UL.Metrics.Definitions; use ASIS_UL.Metrics.Definitions;

package METRICS.Source_Table is

   Low_SF_Bound   : constant := 0;
   High_SF_Bound : constant := 999_999;
   --  Almost 1_000_000 source files for one run of the tool

   type SF_Id is range Low_SF_Bound .. High_SF_Bound;

   No_SF_Id    : constant SF_Id := Low_SF_Bound;
   First_SF_Id : constant SF_Id := No_SF_Id + 1;

   Total_Sources : Natural;
   Sources_Left  : Natural;
   --  Counters used to form and output progress information.

   type SF_Status is (
      Waiting,
      --  Waiting for processing
      Not_A_Legal_Source,
      --  The file does not contain the compilable source

      Error_Detected,
      --  When computing global metrics for this source, some errors has been
      --  detected, so the results of metrics computing are not safe

      Processed);
      --  Global metrics have been successfully computed for this file

   function Present (SF : SF_Id) return Boolean;
   --  Checks that SF is not is equal to No_SF_Id

   function File_Find (Full_SF_Name : String) return SF_Id;
   --  Returns the Id of the file with full name Full_SF_Name stored in the
   --  files table. Returns No_SF_Id if the table does not contain such file.

   procedure Set_New_SF_Record;
   --  Sets the initial values for the new source file record
   --  ??? Do we need this?

   procedure Add_Source_To_Process
     (Fname       : String;
      No_Argument : out Boolean);
   --  Fname is treated as the name of the file to process by the metrics tool.
   --  If it is an empty string, this procedure set No_Argument ON (to stop
   --  iteration through tool parameters) and does else. Otherwise it tries to
   --  add this file to the table of files to process. The following checks are
   --  performed:
   --
   --  - first, this routine checks if Fname is the name of some existing file,
   --    and if it is not, generates the corresponding diagnosis and does
   --    nothing else;
   --
   --  - then, it checks if we already have stored a file with the same name.
   --    If we have the file with the same name, but from a different
   --    directory, the corresponding warning is generated, but the file is
   --    added to the file table (the situation when the metric tool is called
   --    to process files with the same name but from different directories
   --    looks strange, but this may be quite legal and reasonable). But if we
   --    have already stored in the list the name of exactly the same file, we
   --    generate the error message and do not change anything in the list of
   --    files.
   --
   --  At this stage we do not know if it is a compilable Ada source file.

   function Last_Source return SF_Id;
   --  Returns the Id of the last source stored in the source table. Returns
   --  No_SF_Id if there is no source file stored

   procedure Reset_Source_Iterator;
   function Next_Non_Processed_Source
     (Only_Bodies : Boolean := False)
      return        SF_Id;
   --  These two routines provide the iterator through the sources stored in
   --  the sources table. Reset_Source_Iterator resets the iterator to the
   --  ID of the first source stored in the table. Next_Non_Processed_Source
   --  returns the Id if the next source file stored in the file table which
   --  has not been processed yet. If Only_Bodies parameter is set ON, only
   --  body files are considered. We are not 100% sure that this file indeed
   --  is an Ada body file, at this stage we consider any file which name has
   --  ".adb" suffix as a body file. No_SF_Id is returned if there is no such
   --  file in the file table. The idea behind is to process first all the body
   --  files and for each body to process as much specs as possible using this
   --  body tree file, to minimize a set of trees needed to be created to
   --  process all the sources.

   ------------------------------------------------
   -- General source file access/update routines --
   ------------------------------------------------

   function Source_Name       (SF : SF_Id) return String;
   function Short_Source_Name (SF : SF_Id) return String;
   function Source_Status     (SF : SF_Id) return SF_Status;
   function Source_Out_File   (SF : SF_Id) return String;
   function Suffixless_Name   (SF : SF_Id) return String;
   --  Returns the file name with no directory information and with
   --  no suffix (if any).

   function Source_Name_For_Output (SF : SF_Id) return String;
   --  Returns the file name to be included in the metrics output. Depending
   --  on the options set for the tool, it may or may not contain the path
   --  information

   procedure Set_Source_Status (SF : SF_Id; S : SF_Status);

   ----------------------------------------------
   -- Access/update routines for metric values --
   ----------------------------------------------

   --  Line metrics:
   function Get_All_Lines      (SF : SF_Id) return Metric_Count;
   function Get_Code_Lines     (SF : SF_Id) return Metric_Count;
   function Get_Comment_Lines  (SF : SF_Id) return Metric_Count;
   function Get_EOL_Comments   (SF : SF_Id) return Metric_Count;
   function Get_Blank_Lines    (SF : SF_Id) return Metric_Count;

   procedure Set_All_Lines      (SF : SF_Id; Value : Metric_Count);
   procedure Set_Code_Lines     (SF : SF_Id; Value : Metric_Count);
   procedure Set_Comment_Lines  (SF : SF_Id; Value : Metric_Count);
   procedure Set_EOL_Comments   (SF : SF_Id; Value : Metric_Count);
   procedure Set_Blank_Lines    (SF : SF_Id; Value : Metric_Count);

end METRICS.Source_Table;
