------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                   G N A T C H E C K . C O M P I L E R                    --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2005-2008, AdaCore                     --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Handling; use Ada.Characters.Handling;
with Ada.Strings;             use Ada.Strings;
with Ada.Strings.Fixed;       use Ada.Strings.Fixed;
with Ada.Text_IO;             use Ada.Text_IO;

with System.Rident;

with ASIS_UL.Common;          use ASIS_UL.Common;
with ASIS_UL.Misc;            use ASIS_UL.Misc;
with ASIS_UL.Output;          use ASIS_UL.Output;
with ASIS_UL.Source_Table;    use ASIS_UL.Source_Table;
with ASIS_UL.Strings;         use ASIS_UL.Strings;

with Gnatcheck.Diagnoses;     use Gnatcheck.Diagnoses;

package body Gnatcheck.Compiler is

   subtype Option_Parameter is Natural;

   OFF : constant Option_Parameter := 0;
   ON  : constant Option_Parameter := 1;
   --  Flag indicating if the given option is ON or OFF. We can not use Boolean
   --  flags, because some style options and restrictions have additional
   --  numeric parameter

   ----------------------------------------------------
   -- Data structures and routines for style options --
   ----------------------------------------------------

   type Style_Option_Type is
     (Not_A_Style_Option,
      Ident_Level, --  `'1-9'
      'a',
      'A',
      'b',
      'c',
      'd',
      'e',
      'f',
      'h',
      'i',
      'I',
      'k',
      'l',
      'L',  --  `Lnnn'
      'M',  -- `Mnnn'
      'n',
      'o',
      'p',
      'r',
      's',
      'S',
      't',
      'u',
      'x',
      --  options that are not stored in gnatcheck structures
      'm', --  equivalent to M79
      'N', --  Turn everything OFF
      'g', --  Turns ON a GNAT-specific check -gnaty3aAbcdefhiIklmnprsStux
      'y'  --  Turns ON a standard style -gnaty3aAbcefhiklmnprst
      );

   subtype Specific_Style_Options is Style_Option_Type
     range Ident_Level .. 'x';

   type Style_Options_Array is array (Specific_Style_Options)
     of Option_Parameter;

   Style_Options        : Style_Options_Array := (others => OFF);
   Style_Options_Backup : Style_Options_Array;

   Style_To_Character : constant array (Specific_Style_Options) of Character :=
     (ASCII.NUL,
      'a',
      'A',
      'b',
      'c',
      'd',
      'e',
      'f',
      'h',
      'i',
      'I',
      'k',
      'l',
      'L',  --  `Lnnn'
      'M',  -- `Mnnn'
      'n',
      'o',
      'p',
      'r',
      's',
      'S',
      't',
      'u',
      'x');

   procedure Process_Style_Options (Opts : String; Success : out Boolean);
   --  Parses the parameter treating it as a parameter(s) of -gnaty option and
   --  fills Style_Options. Success is set OFF is any problem is detected
   --  during parameter parsing, in this case the setting of -gnaty options
   --  is not changed as a result of the call to this routine

   function To_Style_Option_Type (Ch : Character) return Style_Option_Type;
   --  Converts character literals into corresponding values of
   --  Style_Option_Type.

   ------------------------------------------------------
   -- Data structures and routines for warning options --
   ------------------------------------------------------

   --  First, warning options that do not use the dot notation

   type Warning_Option_Type is
     (Not_A_Warning_Option,  --  unknown option

      --  Individual checks that can be ON or OFF
      'a',
      'b',
      'c',
      'd',
      'f',
      'g',
      'h',
      'i',
      'j',
      'k',
      'l',
      'm',
      'o',
      'p',
      'q',
      'r',
      't',
      'u',
      'v',
      'w',
      'x',
      'y',
      'z',
      --  General-purpose warning switches
      'e',
      'n',
      's');

   subtype Valid_Warning_Options     is Warning_Option_Type range 'a' .. 's';
   subtype Warning_Individual_Checks is Warning_Option_Type range 'a' .. 'z';

   type Warning_Options_Array is array (Warning_Individual_Checks) of Boolean;
   --  For warning options we can use boolean array, because we do not
   --  have an additional numeric parameter for any warning control option

   Warning_Options : Warning_Options_Array := (others => False);
   --  That is, the initial state is that we turn OFF all the specific kinds
   --  of warnings.

   Warning_Options_Backup : Warning_Options_Array;

   type Warning_Option_Value is record
      Opt : Warning_Option_Type;
      Val : Boolean;
   end record;
   --  This is a result of analyzing the parameter of -gnatw option

   function To_Warning_Option_Value
     (Ch :   Character)
      return Warning_Option_Value;
   --  Analyzes a character that is treated as a parameter of -gnatw option
   --  and returns the corresponding setting of the corresponding option.
   --  (for example, returns ('a', True) for 'a' and ('a', False) for 'A')
   --  If the argument can not be a parameter of -gnatw option, returns
   --  (Not_A_Warning_Option, ...)

   function To_Lower_Warning_Option
     (Opt  : Valid_Warning_Options)
      return Character;
   function To_Upper_Warning_Option
     (Opt  : Warning_Individual_Checks)
      return Character;
   --  These two functions convert the argument into the corresponding
   --  character value.

   --  Options that use dot notation:

   type Dot_Warning_Option_Type is
     (Not_A_Dot_Warning_Option,  --  unknown option

      --  Individual checks that can be ON or OFF
      'a',
      'c',
      'o',
      'p',
      'w',
      'x',
      --  The last one turs ON absolutely all warnings
      'e');

   subtype Dot_Valid_Warning_Options     is Dot_Warning_Option_Type
     range 'a' .. 'e';
   subtype Dot_Warning_Individual_Checks is Dot_Warning_Option_Type
     range 'a' .. 'x';

   Dot_Warning_Opt_Iterator : Dot_Valid_Warning_Options;

   type Dot_Warning_Options_Array is array (Dot_Warning_Individual_Checks)
     of Boolean;
   --  For warning options using the dot notation we also can use boolean array
   --  because we do not have an additional numeric parameter for any
   --  warning control option that uses the dot notation

   Dot_Warning_Options : Dot_Warning_Options_Array :=
     (Dot_Warning_Individual_Checks => False);
   --  That is, the initial state is that we turn off all the infividual checks
   --  controlled by dot notation option. We do not have the position
   --  corresponding to '.e' option, if this option is specified, we set ON all
   --  the individual warnings.

   Dot_Warning_Options_Backup : Dot_Warning_Options_Array;

   type Dot_Warning_Option_Value is record
      Opt : Dot_Warning_Option_Type;
      Val : Boolean;
   end record;
   --  This is a result of analyzing the parameter of -gnatw option that uses
   --  the dot notation

   function To_Dot_Warning_Option_Value
     (Ch :   Character)
      return Dot_Warning_Option_Value;
   --  Analyzes a character that is treated as a parameter of -gnatw option
   --  and returns the corresponding setting of the corresponding option.
   --  (for example, returns ('a', True) for 'a' and ('a', False) for 'A')
   --  If the argument can not be a parameter of -gnatw option, returns
   --  (Not_A_Warning_Option, ...)

   function To_Lower_Dot_Warning_Option
     (Opt  : Dot_Valid_Warning_Options)
      return Character;

   function To_Upper_Dot_Warning_Option
     (Opt  : Dot_Warning_Individual_Checks)
      return Character;
   --  These two functions convert the argument into the corresponding
   --  character value.

   function Get_Dot_Warning_Option
     (Opt  : Dot_Warning_Individual_Checks)
      return String;
   --  Returns the string of the form '-gnatw.X" where 'X' represents the
   --  current setting of Opt

   procedure Process_Warning_Options (Opts : String; Success : out Boolean);
   --  Parses the parameter treating it as a parameter(s) of -gnatw option and
   --  fills Warning_Options and dot_Warning_Options arrays

   procedure Set_All_Indivitual_Checks (Val : Boolean);
   --  Sets all the individual checks ON or OFF depending on Val ???

   ---------------------------------------------------------
   -- Data structures and routines for restriction checks --
   ---------------------------------------------------------

   package Gnatcheck_Restrictions is new System.Rident;
   use Gnatcheck_Restrictions;
   --  We cannot use the instantiation of System.Rident in System.Restrictions
   --  because of the pragma Discard_Names that does not allow to use
   --  Restriction_Id'Value when analysing gnatcheck restriction parameters.

   type Restriction_State is record
      Active : Boolean;
      Param  : Option_Parameter;
   end record;
   --  We can not use Option_Parameter here, because some restructions (e.g.
   --  Max_Task_Entries) may be active and may have zero parameter

   Restriction_Setting : array (All_Restrictions) of Restriction_State :=
     (others => (False, OFF));
   --  This array represents only restrictions that are values of
   --  System.Rident.Restriction_Id. But we need to process restrictions that
   --  are not incuded in values of this type.

   type Special_Restriction_Id is
      (Not_A_Special_Restriction_Id,
       No_Dependence);

   subtype All_Special_Restrictions is Special_Restriction_Id range
     No_Dependence .. No_Dependence;
   --  All special restrictions, excluding Not_A_Special_Restriction_Id.

   Special_Restriction_Setting : array (All_Special_Restrictions)
     of Boolean := (others => False);
   --  This array only indicates if a given special restriction is ON or OFF,
   --  we cannot store any restriction parameter information, because
   --  parameter format is restriction-specific

   package Forbidden_Units_Dictionary is new Simple_String_Dictionary
     (Dictionary_Name => "Forbidden units dictionary");

   -------------------------------
   -- Analyze_Compiler_Warnings --
   -------------------------------

   procedure Analyze_Compiler_Warnings (Compiler_Out : Temp_File_Name) is
      Next_Line     : String (1 .. 1024);
      Line_Len      : Positive;
      Comp_Out_File : File_Type;

      procedure Analyze_Warning (Msg : String);
      --  Ananlyses one line containing the compiler warning. Inserts the
      --  warning messages into gnatcheck diagnoses table.

      procedure Analyze_Warning (Msg : String) is
         SF       : SF_Id;
         Line_Num : Natural;
         Col_Num  : Natural;
         --  Coordinates of the warning message

         Diag     : String_Loc;
         --  We store the whole warning message generated by the compiler as is
         --  This would result in some considerable duplications, but what
         --  would be better approach here ???

         Compiler_Message_Kind : Compiler_Message_Kinds :=
           Not_A_Compiler_Nessage;

         Idx      :          Positive := Msg'First;
         Last_Idx : constant Positive := Msg'Last;
         Word_End :          Positive;
      begin
         --  We assume the following compiler warning fromat:
         --
         --   file_name:line_num:column_num: message
         --
         --  What about instantiation chains????

         for J in Idx .. Last_Idx loop
            if Msg (J) = ':' then
               Word_End := J - 1;
               exit;
            end if;
         end loop;

         SF := File_Find (Msg (Idx .. Word_End), Use_Short_Name => True);

         if not Present (SF) then
            --  This source is not an argument of this check
            return;
         end if;

         Idx := Word_End + 2;
         Line_Num := 0;

         while Msg (Idx) /= ':' loop
            Line_Num :=
              Line_Num * 10 +
                (Character'Pos (Msg (Idx)) - Character'Pos ('0'));
            Idx := Idx + 1;
         end loop;

         Idx := Idx + 1;

         Col_Num := 0;

         while Msg (Idx) /= ':' loop
            Col_Num :=
              Col_Num * 10 + (Character'Pos (Msg (Idx)) - Character'Pos ('0'));
            Idx := Idx + 1;
         end loop;

         Idx := Idx + 2;
         --  Now Idx should point to the first character of the warning message

         case Msg (Idx) is
            when  '(' =>
               --  (style)
               Compiler_Message_Kind := Style;
            when  'w' =>
               --  warning, plain warning or restriction warning?
               Compiler_Message_Kind := General_Warning;

               if Idx + 9 < Last_Idx
                 and then
                  Msg (Idx + 7 .. Idx + 9) = ": v"
               then
                  Compiler_Message_Kind := Restriction;
               end if;

            when  others =>
               null;
               pragma Assert (False);
         end case;

         Diag := Enter_String (Msg (Idx .. Last_Idx));

         Store_Compiler_Message
           (In_SF        => SF,
            Line_Num     => Line_Num,
            Col_Num      => Col_Num,
            Message      => Diag,
            Message_Kind => Compiler_Message_Kind);

      end Analyze_Warning;

   begin
      Open (File => Comp_Out_File,
            Mode => In_File,
            Name => Compiler_Out);

      while not End_Of_File (Comp_Out_File) loop
         Get_Line (Comp_Out_File, Next_Line, Line_Len);
         Analyze_Warning (Next_Line (1 .. Line_Len));
      end loop;

      Close (Comp_Out_File);
   exception
      when Ex : others =>
         Error
           ("unknown bug detected when analyzing compiler warnings");
         Error_No_Tool_Name
           ("Please submit bug report to report@adacore.com");
         Report_Unhandled_Exception (Ex);
         raise Fatal_Error;
   end Analyze_Compiler_Warnings;

   function Compiler_Out_File_Name_String return String is
   begin

      if Analyze_Compiler_Output then
         return Compiler_Out_File_Name;
      else
         return "";
      end if;

   end Compiler_Out_File_Name_String;

   -------------------------------------
   -- Create_Restriction_Pragmas_File --
   -------------------------------------

   procedure Create_Restriction_Pragmas_File is
      RPF : File_Type;
   begin
      Create (File => RPF,
              Mode => Out_File,
              Name => Restriction_Config_File);

      for R in All_Restrictions loop

         if Restriction_Setting (R).Active then
            Put (RPF, "pragma Restriction_Warnings (");
            Put (RPF, R'Img);

            if R not in All_Boolean_Restrictions then
               Put (RPF, " =>"  & Restriction_Setting (R).Param'Img);
            end if;

            Put (RPF, ");");

            New_Line (RPF);

         end if;

      end loop;

      for R in Special_Restriction_Setting'Range loop

         if Special_Restriction_Setting (R) then

            case R is

               when No_Dependence =>
                  Forbidden_Units_Dictionary.Reset_Iterator;

                  while not Forbidden_Units_Dictionary.Done loop
                     Put
                       (RPF,
                        "pragma Restriction_Warnings (No_Dependence => ");
                     Put_Line
                       (RPF,
                        Forbidden_Units_Dictionary.Next_Entry & ");");

                  end loop;

            end case;

         end if;

      end loop;

      Close (RPF);
   end Create_Restriction_Pragmas_File;

   ----------
   -- Done --
   ----------

   function Done return Boolean is
   begin
      --  We do not pring out -gnatw.e, so:
      return Dot_Warning_Opt_Iterator = 'e';
   end Done;

   ----------------------------
   -- Get_Dot_Warning_Option --
   ----------------------------

   function Get_Dot_Warning_Option
     (Opt  : Dot_Warning_Individual_Checks)
      return String
   is
   begin

      if Dot_Warning_Options (Opt) then
         return "-gnatw." & To_Lower_Dot_Warning_Option (Opt);
      else
         return "-gnatw." & To_Upper_Dot_Warning_Option (Opt);
      end if;

   end Get_Dot_Warning_Option;

   ----------------------
   -- Get_Style_Option --
   ----------------------

   function Get_Style_Option return String is
      Result     : String (1 .. Style_Options'Length + 12);
      Result_Len : Positive := 1;
      Val_Image  : String (1 .. 3);
      Val_Idx    : Positive;
      Tmp        : Option_Parameter;
   begin
      Result (1 .. 6) := "-gnaty";
      Result_Len := 6;

      for J in Style_Options'Range loop

         if Style_Options (J) /= OFF then

            case J is
               when Ident_Level =>
                  Result_Len := Result_Len + 1;
                  Result (Result_Len) :=
                    Character'Val (Character'Pos ('0') + Style_Options (J));

               when 'L' | 'M'  =>

                  Tmp     := Style_Options (J);
                  Val_Idx := Val_Image'Last + 1;

                  while Tmp /= 0 loop
                     Val_Idx := Val_Idx - 1;
                     Val_Image (Val_Idx) :=
                       Character'Val (Character'Pos ('0') +  Tmp rem 10);
                     Tmp := Tmp / 10;
                  end loop;

                  Result_Len := Result_Len + 1;
                  Result (Result_Len) := Style_To_Character (J);

                  Result_Len := Result_Len + (Val_Image'Last - Val_Idx + 1);
                  Result
                   (Result_Len - (Val_Image'Last - Val_Idx) .. Result_Len) :=
                    Val_Image (Val_Idx .. Val_Image'Last);

               when others =>
                  Result_Len := Result_Len + 1;
                  Result (Result_Len) := Style_To_Character (J);
            end case;

         end if;

      end loop;

      return Result (1 .. Result_Len);
   end Get_Style_Option;

   ------------------------
   -- Get_Warning_Option --
   ------------------------

   function Get_Warning_Option return String is
      Result     : String (1 .. Warning_Options'Length + 6);
      --  "+ 6" is for "-gnatw"

      Idx : Positive := 6;
   begin
      Result (1 .. 6) := "-gnatw";

      for J in Warning_Individual_Checks loop

         if not Warning_Options (J) then
            Idx := Idx + 1;
            Result (Idx) := To_Upper_Warning_Option (J);
         end if;

      end loop;

      for J in Warning_Individual_Checks loop

         if Warning_Options (J) then
            Idx := Idx + 1;
            Result (Idx) := To_Lower_Warning_Option (J);
         end if;

      end loop;

      return Result;
   end Get_Warning_Option;

   -----------------------------
   -- Next_Dot_Warning_Option --
   -----------------------------

   function Next_Dot_Warning_Option return String is
      Result : constant String :=
        Get_Dot_Warning_Option (Dot_Warning_Opt_Iterator);
   begin
      Dot_Warning_Opt_Iterator :=
        Dot_Valid_Warning_Options'Succ (Dot_Warning_Opt_Iterator);

      return Result;
   end Next_Dot_Warning_Option;

   -------------------------------
   -- Print_Active_Restrictions --
   -------------------------------

   procedure Print_Active_Restrictions (Ident_Level : Natural := 0) is
      Bool_Tmp : Boolean := True;
   begin

      for R in Restriction_Setting'Range loop

         if Restriction_Setting (R).Active then
            Report_No_EOL (Proper_Case (R'Img), Ident_Level);

            if R not in All_Boolean_Restrictions then
               Report (" =>"  & Restriction_Setting (R).Param'Img);
            else
               Report_EOL;
            end if;

         end if;

      end loop;

      for R in Special_Restriction_Setting'Range loop

         if Special_Restriction_Setting (R) then
            Report_No_EOL (Proper_Case (R'Img), Ident_Level);

            case R is
               when No_Dependence =>
                  Report_No_EOL (" => ");

                  Forbidden_Units_Dictionary.Reset_Iterator;

                  while not Forbidden_Units_Dictionary.Done loop

                     if Bool_Tmp then
                        Report (Forbidden_Units_Dictionary.Next_Entry);
                        Bool_Tmp := False;
                     else
                        Report
                          ("No_Dependence => " &
                           Forbidden_Units_Dictionary.Next_Entry,
                           Ident_Level);
                     end if;

                  end loop;

            end case;

         end if;

      end loop;

   end Print_Active_Restrictions;

   -------------------------------
   -- Process_Restriction_Param --
   -------------------------------

   procedure Process_Restriction_Param
     (Parameter : String;
      Enable    : Boolean)
   is
      Param        : constant String  := Trim (Parameter, Both);
      First_Idx    : constant Natural := Param'First;
      Last_Idx     :          Natural := Param'Last;
      Arg_Present  :          Boolean := False;
      R_Id         :          Restriction_Id;
      Special_R_Id :          Special_Restriction_Id;
      R_Val        :          Option_Parameter;
   begin
      --  Param should have the format
      --
      --   restriction_parameter_identifier[ => restriction_parameter_argument]
      --
      --  We assume that it can be spaces around '=>'

      --  First, try to define the restriction name.

      for J in First_Idx + 1 .. Last_Idx loop

         if Param (J) = ' '
            or else Param (J) = '='
         then
            Last_Idx := J - 1;
            exit;
         end if;

      end loop;

      begin
         R_Id := Restriction_Id'Value (Param (First_Idx .. Last_Idx));
      exception
         when Constraint_Error =>
            R_Id := Not_A_Restriction_Id;
      end;

      if R_Id = Not_A_Restriction_Id then

         begin
            Special_R_Id :=
              Special_Restriction_Id'Value (Param (First_Idx .. Last_Idx));
         exception
            when Constraint_Error =>
               Special_R_Id := Not_A_Special_Restriction_Id;
         end;

      end if;

      if R_Id = Not_A_Restriction_Id
        and then
         Special_R_Id = Not_A_Special_Restriction_Id
      then
         Error ("wrong restriction identifier : " &
                 Param (First_Idx .. Last_Idx) & ", ignored");
         return;
      end if;

      --  Check if we have a restriction_parameter_argument, and if we do,
      --  set First_Idx to the first character after '=>'

      for J in Last_Idx + 1 .. Param'Last - 2 loop

         if Param (J) = '=' then

            if J <= Param'Last - 2
               and then Param (J + 1) = '>'
            then
               Arg_Present := True;
               Last_Idx := J + 2;
               exit;
            else
               Error ("wrong structure of restriction rule parameter " &
                      Param & ", ignored");
               return;
            end if;

         end if;

      end loop;

      if not Enable then

         if R_Id in All_Restrictions then
            Restriction_Setting (R_Id).Active := False;
         else
            Special_Restriction_Setting (Special_R_Id) := False;
            --  We may need to correct stored parameters of some restrictions

            if Arg_Present then

               case Special_R_Id is
                  when No_Dependence =>
                     Forbidden_Units_Dictionary.Remove_From_Dictionary
                       (Trim (Param (Last_Idx .. Param'Last), Both));

                  when others =>
                     null;
               end case;

            end if;
         end if;

         return;
      end if;

      if R_Id in All_Boolean_Restrictions then

         if Arg_Present then
            Error ("RESTRICTIONS rule parameter: " & Param &
                   " can not contain expression, ignored");
         else
            Restriction_Setting (R_Id).Active := Enable;
         end if;

      elsif R_Id /= Not_A_Restriction_Id then

         if not Arg_Present then
            Error ("RESTRICTIONS rule parameter: " & Param &
                    " should contain an expression, ignored");
            return;
         else
            begin
               R_Val :=
                 Option_Parameter'Value
                   (Trim (Param (Last_Idx .. Param'Last), Both));
            exception
               when Constraint_Error =>
                  Error ("wrong restriction parameter expression in " &
                          Param & ", ignored");
               return;
            end;

         end if;

         Restriction_Setting (R_Id).Active := Enable;
         Restriction_Setting (R_Id).Param  := R_Val;

      else
         --  If we are here, R_Id = Not_A_Restriction_Id, therefore
         --  Special_R_Id /= Not_A_Special_Restriction_Id

         case Special_R_Id is
            when No_Dependence =>

               if not Arg_Present then
                  Error ("RESTRICTIONS rule parameter: " & Param &
                          " should contain an unit name, ignored");
                  return;
               end if;

               Special_Restriction_Setting (Special_R_Id) := True;
               Forbidden_Units_Dictionary.Add_To_Dictionary
                 (Trim (Param (Last_Idx .. Param'Last), Both));

            when Not_A_Special_Restriction_Id =>
               null;
               pragma Assert (False);
         end case;
      end if;

   end Process_Restriction_Param;

   -------------------------------
   -- Process_Style_Check_Param --
   -------------------------------

   procedure Process_Style_Check_Param (Param  : String) is
      Success   : Boolean;
   begin

      if To_Lower (Param) = "all_checks" then
         Process_Style_Options ("y", Success);
      else
         Process_Style_Options (Param, Success);
      end if;

      if not Success then
         Error ("wrong parameters of style_checks option - " & Param &
                 ", ignored");
      end if;

   end Process_Style_Check_Param;

   ---------------------------
   -- Process_Style_Options --
   ---------------------------

   procedure Process_Style_Options (Opts : String; Success : out Boolean) is
      Idx            :          Positive                := Opts'First;
      Last_Idx       : constant Natural                 := Opts'Last;
      Turn_Checks_ON : Option_Parameter range OFF .. ON := ON;
      Val : Natural;

   begin

      Success := True;
      Style_Options_Backup := Style_Options;

      while Idx <= Last_Idx loop

         case Opts (Idx) is
            when '1' .. '9' =>
               if Turn_Checks_ON = ON then
                  Style_Options (Ident_Level) :=
                    Character'Pos (Opts (Idx)) - Character'Pos ('0');
               else
                  Style_Options (Ident_Level) := OFF;
               end if;

               Idx := Idx + 1;

            when 'a' |
                 'A' |
                 'b' |
                 'c' |
                 'd' |
                 'e' |
                 'f' |
                 'h' |
                 'i' |
                 'I' |
                 'k' |
                 'l' |
                 'n' |
                 'o' |
                 'p' |
                 'r' |
                 's' |
                 'S' |
                 't' |
                 'u' |
                 'x' =>

               Style_Options (To_Style_Option_Type (Opts (Idx))) :=
                 Turn_Checks_ON;
               Idx := Idx + 1;

            when 'm' =>

               if Turn_Checks_ON = ON then
                  Style_Options ('M') := 79;
               else
                  Style_Options ('M') := OFF;
               end if;

               Idx := Idx + 1;
            when 'y' =>

               if Turn_Checks_ON = ON then
                  Process_Style_Options ("3aAbcefhiklmnprst", Success);
                  Idx := Idx + 1;

                  pragma Assert (Success);
               else
                  Success       := False;
                  Style_Options := Style_Options_Backup;
                  exit;
               end if;

            when 'g' =>

               if Turn_Checks_ON = ON then
                  Process_Style_Options ("3aAbcdefhiIklmnprsStux", Success);
                  Idx := Idx + 1;

                  pragma Assert (Success);
               else
                  Success       := False;
                  Style_Options := Style_Options_Backup;
                  exit;
               end if;

            when  'N' =>
               --  -gnatyN
               if Turn_Checks_ON = ON then
                  Style_Options := (others => OFF);
                  Idx := Idx + 1;
               else
                  Success       := False;
                  Style_Options := Style_Options_Backup;
                  exit;
               end if;

            when 'L' | 'M' =>

               Idx := Idx + 1;

               if Idx > Last_Idx
                 or else
                  Opts (Idx) not in '0' .. '9'
               then
                  Success       := False;
                  Style_Options := Style_Options_Backup;
                  exit;
               end if;

               Val := 0;

               while Idx <= Last_Idx
                and then
                     Opts (Idx) in '0' .. '9'
               loop
                  Val := Val * 10 +
                         (Character'Pos (Opts (Idx)) - Character'Pos ('0'));

                  if Opts (Idx) = 'L' and then Val > 999 then
                     Success       := False;
                     Style_Options := Style_Options_Backup;
                     exit;
                  end if;
               end loop;

               if Turn_Checks_ON = ON then
                  Style_Options (To_Style_Option_Type (Opts (Idx))) := Val;
               else
                  Style_Options (To_Style_Option_Type (Opts (Idx))) := OFF;
               end if;

            when '+' =>
               Turn_Checks_ON := ON;
               Idx := Idx + 1;
            when '-' =>
               Turn_Checks_ON := OFF;
               Idx := Idx + 1;

            when others =>
               Success       := False;
               Style_Options := Style_Options_Backup;
               exit;
         end case;

      end loop;

   end Process_Style_Options;

   -----------------------------
   -- Process_Warning_Options --
   -----------------------------

   procedure Process_Warning_Options (Opts : String; Success : out Boolean) is
      Idx      : Positive          := Opts'First;
      Last_Idx : constant Positive := Opts'Last;

      Warning_Opt     : Warning_Option_Value;
      Dot_Warning_Opt : Dot_Warning_Option_Value;
   begin
      if Idx <= Last_Idx then
         Success := True;
         Warning_Options_Backup     := Warning_Options;
         Dot_Warning_Options_Backup := Dot_Warning_Options;
      else
         Success := False;
         return;
      end if;

      while Idx <= Last_Idx loop

         if Opts (Idx) = '.' then
            if Idx = Last_Idx then
               Success := False;
               return;
            else
               Idx := Idx + 1;
            end if;

            Dot_Warning_Opt := To_Dot_Warning_Option_Value (Opts (Idx));

            case  Dot_Warning_Opt.Opt is
               when Dot_Warning_Individual_Checks =>
                  Dot_Warning_Options (Dot_Warning_Opt.Opt) :=
                    Dot_Warning_Opt.Val;

               when 'e' =>
                  Set_All_Indivitual_Checks (True);
                  Dot_Warning_Options := (others => True);

               when Not_A_Dot_Warning_Option =>
                  Success := False;
                  exit;
            end case;
         else
            Warning_Opt := To_Warning_Option_Value (Opts (Idx));

            case  Warning_Opt.Opt is
               when Warning_Individual_Checks =>
                  Warning_Options (Warning_Opt.Opt) := Warning_Opt.Val;

                  if Warning_Opt.Opt = 'a' then
                     Set_All_Indivitual_Checks (Warning_Opt.Val);
                  end if;

               when 'e' =>
                  Error ("Warnings cannot be treated as error by gnatcheck, " &
                         "-gnatwe option ignored");
               when 'n' =>
                  null;
               when 's' =>
                  Set_All_Indivitual_Checks (False);
                  Dot_Warning_Options := (others => False);

               when Not_A_Warning_Option =>
                  Success := False;
                  exit;
            end case;

         end if;

         Idx := Idx + 1;
      end loop;

      if Success then
         Use_gnatw_Option := True;
      else
         Warning_Options     := Warning_Options_Backup;
         Dot_Warning_Options := Dot_Warning_Options_Backup;
      end if;
   end Process_Warning_Options;

   ---------------------------
   -- Process_Warning_Param --
   ---------------------------

   procedure Process_Warning_Param (Param  : String) is
      Success   : Boolean;
      Parameter : constant String := ASIS_Trim (Param);
   begin
      Process_Warning_Options (Parameter, Success);

      if not Success then
         Error ("wrong parameters of warnings option - " & Param &
                 ", ignored");
      end if;

   end Process_Warning_Param;

   ------------------------------------
   -- Reset_Dot_Warning_Opt_Iterator --
   ------------------------------------

   procedure Reset_Dot_Warning_Opt_Iterator is
   begin
      Dot_Warning_Opt_Iterator := 'a';
   end Reset_Dot_Warning_Opt_Iterator;

   -------------------------------
   -- Set_All_Indivitual_Checks --
   -------------------------------

   procedure Set_All_Indivitual_Checks (Val : Boolean) is
   begin
      Warning_Options (Warning_Individual_Checks) := (others => Val);
   end Set_All_Indivitual_Checks;

   -------------------------
   -- Set_Compiler_Checks --
   -------------------------

   procedure Set_Compiler_Checks is
   begin
      --  Use_gnaty_Option

      for J in Style_Options'Range loop

         if Style_Options (J) /= OFF then
            Use_gnaty_Option := True;
            exit;
         end if;

      end loop;

      --  Use_gnatw_Option ???

      --  Check_Restrictions

      for J in Restriction_Setting'Range loop

         if Restriction_Setting (J).Active then
            Check_Restrictions := True;
            exit;
         end if;

      end loop;

      if not Check_Restrictions then

         for J in Special_Restriction_Setting'Range loop

            if Special_Restriction_Setting (J) then
               Check_Restrictions := True;
               exit;
            end if;

         end loop;

      end if;

   end Set_Compiler_Checks;

   ---------------------------------
   -- To_Dot_Warning_Option_Value --
   ---------------------------------

   function To_Dot_Warning_Option_Value
     (Ch :   Character)
      return Dot_Warning_Option_Value
   is
      Arg_Ch     : Character := Ch;
      Result_Opt : Dot_Warning_Option_Type;
      Result_Val : Boolean;
   begin
      case Ch is
         when 'e' =>
            Result_Opt := 'e';
         when others =>
            Arg_Ch := To_Lower (Arg_Ch);

            begin
               Result_Opt :=
                 Dot_Warning_Individual_Checks'Value
                   ((1 => ''',
                     2 => Arg_Ch,
                     3 => '''));
            exception
               when Constraint_Error =>
                  Result_Opt := Not_A_Dot_Warning_Option;
            end;

            if Result_Opt /= Not_A_Dot_Warning_Option then
               Result_Val := Ch in 'a' .. 'z';
            end if;

      end case;

      return (Result_Opt, Result_Val);
   end To_Dot_Warning_Option_Value;

   ----------------------------------
   --  To_Lower_Dot_Warning_Option --
   ----------------------------------

   function To_Lower_Dot_Warning_Option
     (Opt  : Dot_Valid_Warning_Options)
      return Character
   is
   begin

      case Opt is
         when 'a' => return 'a';
         when 'c' => return 'c';
         when 'o' => return 'o';
         when 'p' => return 'p';
         when 'w' => return 'w';
         when 'x' => return 'x';
         when 'e' => return 'e';
      end case;

   end To_Lower_Dot_Warning_Option;

   -----------------------------
   -- To_Lower_Warning_Option --
   -----------------------------

   function To_Lower_Warning_Option
     (Opt  : Valid_Warning_Options)
      return Character
   is
   begin

      case Opt is
         when 'a' => return 'a';
         when 'b' => return 'b';
         when 'c' => return 'c';
         when 'd' => return 'd';
         when 'f' => return 'f';
         when 'g' => return 'g';
         when 'h' => return 'h';
         when 'i' => return 'i';
         when 'j' => return 'j';
         when 'k' => return 'k';
         when 'l' => return 'l';
         when 'm' => return 'm';
         when 'o' => return 'o';
         when 'p' => return 'p';
         when 'q' => return 'q';
         when 'r' => return 'r';
         when 't' => return 't';
         when 'u' => return 'u';
         when 'v' => return 'v';
         when 'w' => return 'w';
         when 'x' => return 'x';
         when 'y' => return 'y';
         when 'z' => return 'z';
         when 'e' => return 'e';
         when 'n' => return 'n';
         when 's' => return 's';
      end case;

   end To_Lower_Warning_Option;

   --------------------------
   -- To_Style_Option_Type --
   --------------------------

   function To_Style_Option_Type
     (Ch :   Character)
      return Style_Option_Type
   is
   begin
      return Style_Option_Type'Value (''' & Ch & ''');
   end To_Style_Option_Type;

   ---------------------------------
   -- To_Upper_Dot_Warning_Option --
   ---------------------------------

   function To_Upper_Dot_Warning_Option
     (Opt  : Dot_Warning_Individual_Checks)
      return Character
   is
   begin

      case Opt is
         when 'a' => return 'A';
         when 'c' => return 'C';
         when 'o' => return 'O';
         when 'p' => return 'P';
         when 'w' => return 'W';
         when 'x' => return 'X';
      end case;

   end To_Upper_Dot_Warning_Option;

   -----------------------------
   -- To_Upper_Warning_Option --
   -----------------------------

   function To_Upper_Warning_Option
     (Opt  : Warning_Individual_Checks)
      return Character
   is
   begin

      case Opt is
         when 'a' => return 'A';
         when 'b' => return 'B';
         when 'c' => return 'C';
         when 'd' => return 'D';
         when 'f' => return 'F';
         when 'g' => return 'G';
         when 'h' => return 'H';
         when 'i' => return 'I';
         when 'j' => return 'J';
         when 'k' => return 'K';
         when 'l' => return 'L';
         when 'm' => return 'M';
         when 'o' => return 'O';
         when 'p' => return 'P';
         when 'q' => return 'Q';
         when 'r' => return 'R';
         when 't' => return 'T';
         when 'u' => return 'U';
         when 'v' => return 'V';
         when 'w' => return 'W';
         when 'x' => return 'X';
         when 'y' => return 'Y';
         when 'z' => return 'Z';
      end case;

   end To_Upper_Warning_Option;

   -----------------------------
   -- To_Warning_Option_Value --
   -----------------------------

   function To_Warning_Option_Value
     (Ch :   Character)
      return Warning_Option_Value
   is
      Arg_Ch     : Character := Ch;
      Result_Opt : Warning_Option_Type;
      Result_Val : Boolean;
   begin

      case Ch is
         when 'e' =>
            Result_Opt := 'e';
         when 'n' =>
            Result_Opt := 'n';
         when 's' =>
            Result_Opt := 's';

         when others =>
            Arg_Ch := To_Lower (Arg_Ch);

            begin
               Result_Opt :=
                 Warning_Individual_Checks'Value
                   ((1 => ''',
                     2 => Arg_Ch,
                     3 => '''));
            exception
               when Constraint_Error =>
                  Result_Opt := Not_A_Warning_Option;
            end;

            if Result_Opt /= Not_A_Warning_Option then
               Result_Val := Ch in 'a' .. 'z';
            end if;

      end case;

      return (Result_Opt, Result_Val);
   end To_Warning_Option_Value;

end Gnatcheck.Compiler;
