/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/
#include "tMemManager.h"
#include "eSound.h"
#include "tError.h"
#include <string.h>
#include "tConfiguration.h"
#include "uMenu.h"
#include "eCamera.h"
//#include "tList.h"
#include <iostream.h>
#include <stdlib.h>
#include <eGrid.h>

//eGrid* eSoundPlayer::S_Grid = NULL;

#ifndef DEDICATED
static SDL_AudioSpec audio;
static bool sound_is_there=false;
#endif

// sound quality

#define SOUND_OFF 0
#define SOUND_LOW 1
#define SOUND_MED 2
#define SOUND_HIGH 3

#ifdef WIN32
static int buffer_shift=1;
#else
static int buffer_shift=0;
#endif

static tConfItem<int> bs("SOUND_BUFFER_SHIFT","Buffer size multiplier",buffer_shift);

static int sound_quality=SOUND_MED;
static tConfItem<int> sq("SOUND_QUALITY","sound quality [0=off, 3=high]",sound_quality);

static int sound_sources=10;
static tConfItem<int> ss("SOUND_SOURCES","number of sound sources to be heard at the "
		 "same time",sound_sources);
static REAL loudness_thresh=.01;
static int real_sound_sources;

static List<eSoundPlayer> se_globalPlayers;


void fill_audio(void *udata, Uint8 *stream, int len){
#ifndef DEDICATED
  real_sound_sources=0;
  int i;
  if (eGrid::CurrentGrid())
    for(i=eGrid::CurrentGrid()->Cameras().Len()-1;i>=0;i--)
      eGrid::CurrentGrid()->Cameras()(i)->SoundMix(stream,len);

  for(i=se_globalPlayers.Len()-1;i>=0;i--)
    se_globalPlayers(i)->Mix(stream,len,0,1,1);
    
  if (real_sound_sources>sound_sources+4)
    loudness_thresh+=.01;
  if (real_sound_sources>sound_sources+1)
    loudness_thresh+=.001;
  if (real_sound_sources<sound_sources-4)
    loudness_thresh-=.001;
  if (real_sound_sources<sound_sources-1)
    loudness_thresh-=.0001;
  if (loudness_thresh<0)
    loudness_thresh=0;
#endif
}


void se_SoundInit(){
#ifndef DEDICATED
  if (!sound_is_there && sound_quality!=SOUND_OFF){
    SDL_AudioSpec desired;

    switch (sound_quality){
    case SOUND_LOW:  desired.freq=11025; break;
    case SOUND_MED:  desired.freq=22050; break;
    case SOUND_HIGH: desired.freq=44100; break;
    default: desired.freq=22050; 
    }

    desired.format=AUDIO_S16;
    desired.samples=128;
    while (desired.samples <= desired.freq >> (6-buffer_shift))
	desired.samples <<= 1;
    desired.channels = 2;   
    desired.callback = fill_audio;
    desired.userdata = NULL;
    sound_is_there=(SDL_OpenAudio(&desired,&audio)>=0);
    if (sound_is_there && (audio.format!=AUDIO_S16 || audio.channels!=2)){
      se_SoundExit();
      // force emulation of 16 bit stereo
      audio.format=AUDIO_S16;
      audio.channels=2;
      sound_is_there=(SDL_OpenAudio(&audio,NULL)>=0);
      con << "Sorry. We need 16 bit stereo output (I'm too lazy to\n"
	"support other formats. Maybe if you ask really nice?).\n";
    }
    if (!sound_is_there)
      con << "Sound initialisation failed.\n";
    else{
      //for(int i=wavs.Len()-1;i>=0;i--)
      //wavs(i)->Init();
      con << "Sound initialized: 16 bit stereo at " << audio.freq << " Hz, buffer size "<< audio.samples << " samples.\n";
    }
  }
#endif
}

void se_SoundExit(){
#ifndef DEDICATED
  if (sound_is_there){
    con << "Disabling sound...\n";
    se_SoundPause(true);
    //    for(int i=wavs.Len()-1;i>=0;i--)
    //wavs(i)->Exit();
    SDL_CloseAudio();
    con << "done!\n";
  }
  sound_is_there=false;
#endif
}

#ifndef DEDICATED
static unsigned int locks;
#endif

void se_SoundLock(){
#ifndef DEDICATED
  if (!locks)
    SDL_LockAudio(); 
  locks++;
#endif
}

void se_SoundUnlock(){
#ifndef DEDICATED
  locks--;
  if (!locks)
    SDL_UnlockAudio(); 
#endif
}

void se_SoundPause(bool p){
#ifndef DEDICATED
  SDL_PauseAudio(p);
#endif
}

// ***********************************************************

eWavData::eWavData(const char * fileName,const char *alternative)
  :data(NULL),len(0){
  //wavs.Add(this,id);

  #ifndef DEDICATED

  alt=false;

  SDL_AudioSpec *result=SDL_LoadWAV(fileName,&spec,&data,&len);
  if (result!=&spec || !data){
    if (strlen(alternative)>0){
      result=SDL_LoadWAV(alternative,&spec,&data,&len);
      if (result!=&spec || !data){
	tERR_ERROR("Sound file " << alternative << " not found. Have you called "
		   "Armagetron from the right directory?");}
      else
	alt=true;
    }
    else{
      result=SDL_LoadWAV("sound/expl.wav",&spec,&data,&len);
      if (result!=&spec || !data){
	tERR_ERROR("Sound file sound/expl.wav not found. Have you called "
		   "Armagetron from the right directory?");}
      else
	len=0;
    }
    /*
    tERR_ERROR("Sound file " << fileName << " not found. Have you called "
    "Armagetron from the right directory?"); */
  }
  
  if (spec.format==AUDIO_S16)
    samples=len>>1;
  else if(spec.format==AUDIO_U8)
    samples=len;
  else
    tERR_ERROR("Sound file " << fileName << " has unsupported format. Sorry!");
  
  samples/=spec.channels;

#ifdef DEBUG
#ifdef LINUX
  con << "Sound file " << fileName << " loaded: ";
  switch (spec.format){
  case AUDIO_S16: con << "16 bit "; break;
  case AUDIO_U8: con << "8 bit "; break;
  default: con << "unknown "; break;
  }
  if (spec.channels==2)
    con << "stereo ";
  else
    con << "mono ";

  con << "at " << spec.freq << " Hz,\n";

  con << samples << " samples in " << len << " bytes.\n";
#endif
#endif
#endif
}

eWavData::~eWavData(){
#ifndef DEDICATED
  if (data){
    se_SoundLock();
    SDL_FreeWAV(data);
    data=NULL;
    len=0;
    se_SoundUnlock();
  }
#endif
}

bool eWavData::Mix(Uint8 *dest,Uint32 playlen,eAudioPos &pos,
		   REAL Rvol,REAL Lvol,REAL Speed,bool loop){
#ifndef DEDICATED
  playlen/=4;

  #define SPEED_FRACTION (1<<20)

  #define VOL_SHIFT 16
  #define VOL_FRACTION (1<<VOL_SHIFT)
  
  #define MAX_VAL ((1<<16)-1)
  #define MIN_VAL -(1<<16)

  // first, split the speed into the part before and after the decimal:
  if (Speed<0) Speed=0;

  // adjust for different sample rates:
  Speed*=spec.freq;
  Speed/=audio.freq;

  int speed=int(floor(Speed));
  int speed_fraction=int(SPEED_FRACTION*(Speed-speed));

  // secondly, make integers out of the volumes:
  int rvol=int(Rvol*VOL_FRACTION);
  int lvol=int(Lvol*VOL_FRACTION);


  bool goon=true;
  
  while (goon){
    if (spec.channels==2){
      if (spec.format==AUDIO_U8)
	while (playlen>0 && pos.pos<samples){
	  // fix endian problems for the Mac port, as well as support for other
	  // formats than  stereo...
	  int l=((short *)dest)[0];
	  int r=((short *)dest)[1];
	  r += (rvol*(data[(pos.pos<<1)  ]-128)) >> (VOL_SHIFT-8);
	  l += (lvol*(data[(pos.pos<<1)+1]-128)) >> (VOL_SHIFT-8);
	  if (r>MAX_VAL) r=MAX_VAL;
	  if (l>MAX_VAL) l=MAX_VAL;
	  if (r<MIN_VAL) r=MIN_VAL;
	  if (l<MIN_VAL) l=MIN_VAL;
	  
	  ((short *)dest)[0]=l;
	  ((short *)dest)[1]=r;
	  
	  dest+=4;
	  
	  pos.pos+=speed;
	  
	  pos.fraction+=speed_fraction;
	  while (pos.fraction>=SPEED_FRACTION){
	    pos.fraction-=SPEED_FRACTION;
	    pos.pos++;
	  }
	  
	  playlen--;
	}
      else{
	while (playlen>0 && pos.pos<samples){
	  int l=((short *)dest)[0];
	  int r=((short *)dest)[1];
	  r += (rvol*(((short *)data)[(pos.pos<<1)  ])) >> VOL_SHIFT;
	  l += (lvol*(((short *)data)[(pos.pos<<1)+1])) >> VOL_SHIFT;
	  if (r>MAX_VAL) r=MAX_VAL;
	  if (l>MAX_VAL) l=MAX_VAL;
	  if (r<MIN_VAL) r=MIN_VAL;
	  if (l<MIN_VAL) l=MIN_VAL;
	  
	  ((short *)dest)[0]=l;
	  ((short *)dest)[1]=r;
	  
	  dest+=4;
	  
	  pos.pos+=speed;
	  
	  pos.fraction+=speed_fraction;
	  while (pos.fraction>=SPEED_FRACTION){
	    pos.fraction-=SPEED_FRACTION;
	    pos.pos++;
	  }
	  playlen--;
	}
      }
    }
    else{
      if (spec.format==AUDIO_U8){
	while (playlen>0 && pos.pos<samples){
	  // fix endian problems for the Mac port, as well as support for other
	  // formats than  stereo...
	  int l=((short *)dest)[0];
	  int r=((short *)dest)[1];
	  int d=data[pos.pos]-128;
	  l += (lvol*d) >> (VOL_SHIFT-8);
	  r += (rvol*d) >> (VOL_SHIFT-8);
	  if (r>MAX_VAL) r=MAX_VAL;
	  if (l>MAX_VAL) l=MAX_VAL;
	  if (r<MIN_VAL) r=MIN_VAL;
	  if (l<MIN_VAL) l=MIN_VAL;
	  
	  ((short *)dest)[0]=l;
	  ((short *)dest)[1]=r;
	  
	  dest+=4;
	  
	  pos.pos+=speed;
	  
	  pos.fraction+=speed_fraction;
	  while (pos.fraction>=SPEED_FRACTION){
	    pos.fraction-=SPEED_FRACTION;
	    pos.pos++;
	  }
	  
	  playlen--;
	}
      }
      else
	while (playlen>0 && pos.pos<samples){
	  int l=((short *)dest)[0];
	  int r=((short *)dest)[1];
	  int d=((short *)data)[pos.pos];
	  l += (lvol*d) >> VOL_SHIFT;
	  r += (rvol*d) >> VOL_SHIFT;
	  if (r>MAX_VAL) r=MAX_VAL;
	  if (l>MAX_VAL) l=MAX_VAL;
	  if (r<MIN_VAL) r=MIN_VAL;
	  if (l<MIN_VAL) l=MIN_VAL;
	  
	  ((short *)dest)[0]=l;
	  ((short *)dest)[1]=r;
	  
	  dest+=4;
	  
	  pos.pos+=speed;
	  
	  pos.fraction+=speed_fraction;
	  while (pos.fraction>=SPEED_FRACTION){
	    pos.fraction-=SPEED_FRACTION;
	    pos.pos++;
	  }
	  playlen--;
	}
    }
    
    if (loop && pos.pos>=samples)
      pos.pos-=samples;
    else
      goon=false;
  }
#endif
  return (playlen>0);
  
}  

void eWavData::Loop(){
#ifndef DEDICATED
  Uint8 *buff2=tNEW(Uint8) [len];
  
  if (buff2){
    memcpy(buff2,data,len);
    Uint32 samples;
    
    if (spec.format==AUDIO_U8){
      samples=len;
      for(int i=samples-1;i>=0;i--){
	Uint32 j=i+((len>>2)<<1);
	if (j>=len) j-=len;

	REAL a=fabs(100*(j/REAL(samples)-.5));
	if (a>1) a=1;
	REAL b=1-a;

	data[i]=int(a*buff2[i]+b*buff2[j]);
      }
    }
    else if (spec.format==AUDIO_S16){
      samples=len>>1;
      for(int i=samples-1;i>=0;i--){

	/*
	REAL a=2*i/REAL(samples);
	if (a>1) a=2-a;
	REAL b=1-a;
	*/
	

	Uint32 j=i+((samples>>2)<<1);
	while (j>=samples) j-=samples;

	REAL a=fabs(100*(j/REAL(samples)-.5));
	if (a>1) a=1;
	REAL b=1-a;


	((short *)data)[i]=int(a*((short *)buff2)[i]+b*((short *)buff2)[j]);
      }
    }
    delete buff2;
  }

#endif
}


// ******************************************************************

void eAudioPos::Reset(int randomize){
#ifndef DEDICATED
  if (randomize){
    fraction=int(SPEED_FRACTION*(rand()/float(RAND_MAX)));
    pos=int(randomize*(rand()/float(RAND_MAX)));
  }
  else
    fraction=pos=0;
#endif
}



eSoundPlayer::eSoundPlayer(eWavData &w,bool l)
  :id(-1),wav(&w),loop(l){
  for(int i=MAX_VIEWERS-1;i>=0;i--)
    goon[i]=true;
}

eSoundPlayer::~eSoundPlayer(){}

bool eSoundPlayer::Mix(Uint8 *dest,
		      Uint32 len,
		      int viewer,
		      REAL rvol,
		      REAL lvol,
		      REAL speed){

  if (goon[viewer]){
    if (rvol+lvol>loudness_thresh){
      real_sound_sources++;
      return goon[viewer]=!wav->Mix(dest,len,pos[viewer],rvol,lvol,speed,loop);
    }
    else
      return true;
  }
  else
    return false;
}

void eSoundPlayer::Reset(int randomize){
  for(int i=MAX_VIEWERS-1;i>=0;i--){
    pos[i].Reset(randomize);
    goon[i]=true;
  }
}

void eSoundPlayer::End(){
  for(int i=MAX_VIEWERS-1;i>=0;i--){
    goon[i]=false;
  }
}


void eSoundPlayer::MakeGlobal(){
  se_globalPlayers.Add(this,id);
}


// ***************************************************************

uMenu Sound_menu("Sound Settings");

static uMenuItemInt sources_men
(&Sound_menu,"Sound Sources:",
 "Gives the approximate number of sound sources to be mixed; setting it too "
 "low will result in important sound sources (your own engine) to be "
 "constantly turned on and off, too high values eat away too much CPU power. "
 "Six seems to be a good value for low-power CPU's.",
 sound_sources,2,20,2);

static uMenuItemSelection<int> sq_men
(&Sound_menu,"Sound Quality:",
 "Selects the quality of the sound; currently, only 16 bit stereo modes are "
 "available.\n",
 sound_quality);


static uSelectEntry<int> a(sq_men,"Off","Disables sound output completely."
			   ,SOUND_OFF);
static uSelectEntry<int> b(sq_men,"Low",
			   "Sound output is 16 bit 11025 Hz stereo."
			   ,SOUND_LOW);
static uSelectEntry<int> c(sq_men,"Medium",
			   "Sound output is 16 bit 22050 Hz stereo."
			   ,SOUND_MED);
static uSelectEntry<int> d(sq_men,"High",
			   "Sound output is 16 bit 44100 Hz stereo."
			   ,SOUND_HIGH);

static uMenuItemSelection<int> bm_men
(&Sound_menu,"Buffer Length:",
 "Selects the size of the sound buffer.\n",
 buffer_shift);

static uSelectEntry<int> ba(bm_men,"Very Small","Latency of about 0.02s, but very high probability of artefacts.",-2);

static uSelectEntry<int> bb(bm_men,"Small","Latency below 0.04s, but high probability of artefacts.",-1);

static uSelectEntry<int> bc(bm_men,"Normal","Latency of about 0.1s, and may still produce artefacts.",0);

static uSelectEntry<int> bd(bm_men,"High","Latency of about 0.2s, probably no artifacts.",1);

static uSelectEntry<int> be(bm_men,"Very High","Latency of about 0.4s.",2);


void se_SoundMenu(){
  se_SoundPause(true);
  se_SoundLock();
  int oldsettings=sound_quality;
  int oldshift=buffer_shift;
  Sound_menu.Enter();
  if (oldsettings!=sound_quality || oldshift!=buffer_shift){
    se_SoundExit();
    se_SoundInit();
  }
  se_SoundUnlock();
  se_SoundPause(false);
}
