/* $Id: LuaEntity.cpp,v 1.7 2003/02/14 13:40:28 mat Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>

#include <Engine/Engine.h>

#include "luna.h"
#include "LuaEntity.h"
#include "LuaTable.h"

namespace Ark
{

   LuaEntity::LuaEntity(lua_State *L) :
      EngineEntity(GetEngine()->GetWorld())
   {}

   LuaEntity::~LuaEntity()
   {
      //cerr << "destroying lua entity!! \n";
   }

   /// Name, short name, identifier
   int
   LuaEntity::get_name (lua_State *L)
   {
      lua_pushstring (L, m_Name.c_str());
      return 1;
   }

   int
   LuaEntity::set_name (lua_State *L)
   {
      m_Name = lua_tostring(L, -1);
      lua_pop(L, 1);
      return 0;
   }

   int
   LuaEntity::get_shortname (lua_State *L)
   {
      lua_pushstring (L, m_ShortName.c_str());
      return 1;
   }

   int
   LuaEntity::set_shortname (lua_State *L)
   {
      m_ShortName = lua_tostring(L, -1);
      lua_pop(L, 1);
      return 0;
   }

   int
   LuaEntity::get_id (lua_State *L)
   {
      lua_pushnumber (L, m_ID);
      return 1;
   }


   /// Callbacks 
   int
   LuaEntity::add_timer (lua_State *L)
   {
      bool res = AddTimer
	 (static_cast<int>(lua_tonumber(L, -2)),
	  static_cast<scalar>(lua_tonumber(L, -1)));
    
      lua_pop (L,2);
      lua_pushbool (L, res);
      return 1;
   }
   
   int
   LuaEntity::add_message (lua_State *L)
   {
      Entity *dest;
      String msg;
	  std::vector<String> answerlist;

      lua_getobject (L, -3, &dest);
      msg = lua_tostring(L, -2);

      // table is in the stack at index -1
      lua_pushnil(L);  /* first key */
      while (lua_next(L, -2) != 0)
      {
         // `key' is at index -2 and `value' at index -1 
	 answerlist.push_back (lua_tostring(L, -1));
	 // removes `value'; keeps `index' for next iteration 
         lua_pop(L, 1); 
      }
      
      AddMessage (dest, msg, &answerlist);
      lua_pop(L, 3);
      return 0;
   }
  
   /// Visual
   int
   LuaEntity::psys_set (lua_State *L)
   {
      PsysSet
	 (static_cast<int>(lua_tonumber(L, -4)),
	  lua_tostring(L, -3),
	  lua_tostring(L, -2),
	  lua_tostring(L, -1));
    
      lua_pop (L,3);
      return 0;
   }

   int LuaEntity::psys_stop (lua_State *L)
   {
      PsysStop(static_cast<int>(lua_tonumber(L, -1)));
      lua_pop(L,1);
      return 0;
   }

   int LuaEntity::psys_exists (lua_State *L)
   {
      bool r = PsysExists(static_cast<int>(lua_tonumber(L, -1)));
      lua_pop (L,1);
      lua_pushbool (L, r);
      return 1;
   }
      
   int LuaEntity::set_model (lua_State *L)
   {
      SetModel(lua_tostring(L, -1));
      lua_pop (L,1);
      return 0;
   }

   int LuaEntity::get_model (lua_State *L)
   {
      Model *m = m_MState.GetModel();

      if (m)
	 lua_pushstring (L, m->Name().c_str());
      else
	 lua_pushnil(L);

      return 1;
   }
      
   int LuaEntity::play_anim (lua_State *L)
   {
      bool r = m_MState.Play (static_cast<ModelState::PlayMode>
			      ((int)lua_tonumber(L, -2)),
			      lua_tostring (L, -1));

      lua_pop (L,2);
      lua_pushbool (L, r);

      return 1;
   }

   int LuaEntity::play_sound (lua_State *L)
   {
      return 0;
   }
   
   int LuaEntity::play_music (lua_State *L)
   {
      std::string	file;
      Entity		*dest;
      
      file = lua_tostring(L, -1);
      lua_getobject (L, -2, &dest);
      std::cerr << "Playing " << file << " for entity " << dest->m_Name << std::endl;
      lua_pop(L, 2);
      
      // Voir pour le AddMessage : on envoit un message au player, receptionne
      // quelque part... 
      // le truc a part en question, il fait un Play, lequel va demander
      // au cache la class Music - besoin de 2 trucs, une classe Music toute bete
      // pour stocker mon mixmusic, charger la musique, et une autre pour la
      // jouer, i.e. ca sera la ou il y a Play.
      // nota: le loader doit etre enregistre ... donc classe abstraite puis
      // pointeur sur le bon machin Music pour que la dependance sur SDL_mixer
      // ne soit que sur le client...
      return 0;
   } 
      
   /// Position
   int LuaEntity::set_position (lua_State *L)
   {
      return 0;
   }

   int LuaEntity::set_entity_goal (lua_State *L)
   {
      Entity *dest;
      lua_getobject (L, -1, &dest);

      SetGoal (dest);
      return 0;
   }

   int LuaEntity::set_position_goal (lua_State *L)
   {
      return 0;
   }

   int LuaEntity::set_no_goal (lua_State *L)
   {
      SetGoal (NULL);
      return 0;
   }
   
   /**
    * Used to get the position of the entity.
    *
    * @return A table which is a vector of three values indexed by x, y and z.
    */
   int LuaEntity::get_position (lua_State *L)
   {
      // TODO: we should use a LuaVector3 but it is not yet implemented... ;-)
      LuaTable* position = LuaTable::createTable( L );
      position->addElement("x", m_MState.m_Position.X);
      position->addElement("y", m_MState.m_Position.Y);
      position->addElement("z", m_MState.m_Position.Z);

      // FIXME: memory leak TODO: oups, it does delete the table in lua state!
      // We have to modify LuaTable and add a boolean or a separate function to
      // delete the table from the lua state.
      //delete position;	// This doesn't delete the table in Lua state.

      return 1;	// We return 1 parameter
   }
      
   /// Flags
#define FLAGACCESSOR(func, flagvalue)                               \
      int LuaEntity::is_ ## func (lua_State *L)                     \
      {                                                             \
	 lua_pushbool (L, (m_Flags & (flagvalue)) ? true : false);  \
	 return 1;                                                  \
      }                                                             \
                                                                    \
      int LuaEntity::set_ ## func (lua_State *L)                    \
      {                                                             \
	 bool oldv = (m_Flags & (flagvalue)) ? true : false;        \
	 bool newv =  lua_tonumber(L, -1) != 0.0;       \
                                                                    \
	 if (newv) m_Flags |= flagvalue;                            \
	 else m_Flags &= ~flagvalue;                                \
                                                                    \
	 lua_pop (L,1);                                             \
	 lua_pushbool (L, oldv);                                    \
	 return 1;                                                  \
      }

   FLAGACCESSOR (aidriven, AIDRIVEN);
   FLAGACCESSOR (dead, DEAD);
   FLAGACCESSOR (visible, VISIBLE);
   FLAGACCESSOR (collider, COLLISION);
   FLAGACCESSOR (staticp, STATIC);

#undef FLAGACCESSOR

   
   /// Clear changed flag
   int LuaEntity::clear_changed (lua_State *L)
   {
      ClearChanged();
      return 0;
   }
      
   /// Entity binding.
   int LuaEntity::attach_to_entity (lua_State *L)
   {
      if (!lua_isnil (L, -2))
      {
	 Entity *dest;
      
	 lua_getobject (L, -2, &dest);
	 AttachToEntity (dest, lua_tostring (L, -1));
      }
      else
	 AttachToEntity (NULL, "");

      lua_pop(L, 2);
      return 0;
   }

   int LuaEntity::is_attached (lua_State *L)
   {
      lua_pushbool (L, IsAttached());
      return 1;
   }

   /// Entity management
   int LuaEntity::set_entries (lua_State *L)
   {
      std::istringstream stream( lua_tostring(L, -1) );

      lua_pop (L,1);
      GetEngine()->GetScript()->SetEntries
	 (this, "lua/setentries/buffer", stream);

      lua_pushbool (L, true);
      return 1;
   }

   int LuaEntity::set_callback_entries (lua_State *L)
   {
      lua_pushbool (L, true);
      return 1;
   }

   int LuaEntity::set_callback (lua_State *L)
   {
      m_Callback = lua_tostring(L, -1);
      lua_pop (L, 1);
      lua_pushbool (L, true);
      return 1;
   }

   int LuaEntity::get_callback (lua_State *L)
   {
      lua_pushstring (L, m_Callback.c_str());
      return 1;
   }

   ////////////////////////////////////////////////////////////////////////
   ////////////////////////////////////////////////////////////////////////
   ////////////////////////////////////////////////////////////////////////

   // === Functions to be implemented by script entities ===

   ////////////////////////////////////////////////////////////////////////
   ////////////////////////////////////////////////////////////////////////
   ////////////////////////////////////////////////////////////////////////

   /**
    * Called to set entity properties, EntryList is an association of
    * name with int/scalar/string values.
    */
   bool LuaEntity::SetEntries (Engine *engine, const Ark::EntryList &entries)
   {
      EntryList::const_iterator i;

      // Set callback
      i = entries.find ("callback");
      m_Callback = *i->second.d_str;
      
      return EngineEntity::SetEntries (engine, entries);
   }
      
   /**
    * Called to set entity callback properties, EntryList is an
    * association of name with int/scalar/string values.
    */
   bool
   LuaEntity::SetCallback (Engine *engine, const Ark::EntryList &entries)
   {
      return false;
   }

   // Push a global object function on the top of the stack,
   // (in the form callback_function), and check that it isn't 
   // nil.
   bool
   LuaEntity::push_check_objfunction (const String &fname)
   {
      lua_getglobal (g_LuaState, m_Callback.c_str());

      if (lua_isnil(g_LuaState, -1))
      { 
	 lua_pop (g_LuaState, 1);
	 return false;
      }

      lua_pushstring (g_LuaState, fname.c_str());
      lua_gettable (g_LuaState, -2);

      if (lua_isnil(g_LuaState, -1))
      { 
	 lua_pop (g_LuaState, 2);
	 return false;
      }

      lua_remove (g_LuaState, -2);
      push_object(g_LuaState);
      return true;
   }

   void
   LuaEntity::Create ()
   {
      if (!push_check_objfunction("on_create"))
	 return;
     
      lua_call (g_LuaState, 1, 0);
   }
      
   /**
    * Called when the entity has reached its goal.
    */
   bool
   LuaEntity::EvGoalReached ()
   {
      if (!push_check_objfunction("on_goal_reached"))
	 return false;
      lua_call (g_LuaState, 1, 0);

      //bool r = static_cast<bool>(lua_tonumber (g_LuaState, -1));
      //lua_pop (g_LuaState, 1);
      return true;
   }
	 
   /**
    * Called when a timer has expired. n is the timer id, as given to
    * the Timer() function.
    */
   bool
   LuaEntity::EvTimer (int n)
   {
      if (!push_check_objfunction("on_timer"))
	 return false;

      lua_pushnumber (g_LuaState, n);
      lua_call (g_LuaState, 2, 0);

      return true;
   }
	 	 
   /**
    * Called when someone speaks to this entity.
    */
   bool
   LuaEntity::EvTell ()
   {
      EntityMessage &em = m_Messages[0];

      if (!push_check_objfunction("on_tell"))
	 return false;

      static_cast<LuaEntity*> (em.m_Entity)->push_object(g_LuaState);
      lua_pushstring (g_LuaState, em.m_Message.c_str());

      size_t nanswers = em.m_AnswerList.size();
      for (size_t i = 0; i < nanswers; ++i)
	 lua_pushstring (g_LuaState, em.m_AnswerList[i].c_str());

      int retv = lua_call (g_LuaState, 3+nanswers, 1);
      bool r = lua_tonumber (g_LuaState, -1) != 0.0;
      lua_pop(g_LuaState, 1);

      // No error ? So remove message from the list.
      if (r && retv == 0)
	 m_Messages.erase(m_Messages.begin());

      return false;
   }

   /**
    * Called when a collision has occured.
    */
   void
   LuaEntity::EvHit (const EntityCollision &col)
   {
      if (!push_check_objfunction("on_simplehit"))
	 return;

      LuaEntity *collider = static_cast<LuaEntity*>(col.m_Collider);
      collider->push_object(g_LuaState);

      lua_pushbool (g_LuaState, col.m_Potential);
      int retv = lua_call (g_LuaState, 3, 1);

      bool r = lua_tonumber(g_LuaState, -1) != 0.0;
      if (retv != 0 || r == false) 
	 SetGoal (NULL);

      lua_pop(g_LuaState, 1);
   }


   const char LuaEntity::className[] = "Entity";
   const Luna<LuaEntity>::RegType LuaEntity::Register[] = 
   {
      {"get_name",             &LuaEntity::get_name},
      {"set_name",             &LuaEntity::set_name},
      {"get_shortname",        &LuaEntity::get_shortname},
      {"set_shortname",        &LuaEntity::set_shortname},
      {"get_id",               &LuaEntity::get_id},
      {"add_timer",            &LuaEntity::add_timer},
      {"add_message",          &LuaEntity::add_message},
      {"psys_set",             &LuaEntity::psys_set},
      {"psys_stop",            &LuaEntity::psys_stop},
      {"psys_exists",          &LuaEntity::psys_exists},
      {"set_model",            &LuaEntity::set_model},
      {"get_model",            &LuaEntity::get_model},
      {"play_anim",            &LuaEntity::play_anim},
      {"play_sound",           &LuaEntity::play_sound},
      {"play_music",           &LuaEntity::play_music},
      {"set_position",         &LuaEntity::set_position},
      {"get_position",         &LuaEntity::get_position},
      {"set_entity_goal",      &LuaEntity::set_entity_goal},
      {"set_position_goal",    &LuaEntity::set_position_goal},
      {"set_no_goal",          &LuaEntity::set_no_goal},
      {"is_aidriven",          &LuaEntity::is_aidriven},
      {"is_dead",              &LuaEntity::is_dead},
      {"is_visible",           &LuaEntity::is_visible},
      {"is_collider",          &LuaEntity::is_collider},
      {"is_static",            &LuaEntity::is_staticp},
      {"set_aidriven",         &LuaEntity::set_aidriven},
      {"set_dead",             &LuaEntity::set_dead},
      {"set_visible",          &LuaEntity::set_visible},
      {"set_collider",         &LuaEntity::set_collider},
      {"set_static",           &LuaEntity::set_staticp},
      {"clear_changed",        &LuaEntity::clear_changed},
      {"attach_to_entity",     &LuaEntity::attach_to_entity},
      {"is_attached",          &LuaEntity::is_attached},
      {"set_entries",          &LuaEntity::set_entries},
      {"set_callback_entries", &LuaEntity::set_callback_entries},
      {"set_callback",         &LuaEntity::set_callback},
      {"get_callback",         &LuaEntity::get_callback},
      {0}
   };

/******************************************************************************
 * Add an element to the list
 *
 * \param element The element to add.
 ******************************************************************************/
void LuaEntityList::addElement(LuaEntity& element)
{
  element.push_object(m_L);
  lua_rawseti(m_L, m_Index, ++m_LastIndex);
}

/******************************************************************************
 * Constructs a LuaEntityList from scratch. This creates a new table on
 * the stack.
 *
 * \param L The lua state where we are going to create the table.
 ******************************************************************************/
LuaEntityList* LuaEntityList::createTable(lua_State* L)
{
  lua_newtable(L);
  return new LuaEntityList(L, -1);
}

}

