/* $Id: ArkSystem.h,v 1.21 2003/03/23 20:09:25 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_SYSTEM_H
#define ARK_SYSTEM_H

#include <Ark/Ark.h>
#include <Ark/ArkString.h>

namespace Ark
{
   /// A message error level (for Report()).
   enum ErrorLevel
   {
      E_LOG,
      E_WARNING,
      E_ERROR,
      E_FATAL 
   };
   
   /// A pointer to a function called on exit.
   typedef void (*ExitFunc)();

   class Loaders;
   class FactoryList;

   class ARK_DLL_API System
   {
      protected:
	 /// Initializes Ark, etc..
	 System ();

	 /// Destroy the data used by the system
	 virtual ~System ();

      public:
	 /**
	  * \brief Init the library.
	  * \param argc Is a pointer to number of arguments
	  * \param argv Is a pointer to the argument list.
	  *
	  * This should be called before any library functions. It parses the
	  * command arguments and remove used ones. It reads default
	  * configuration files, and mounts the home directory
	  */
	 static void Init (int *argc, char ***argv);

	 /**
	  * Get the REAL library version (not the header version).
	  */
	 void GetVersion (int *major, int *minor, int *micro);

	 /**
	  * \brief This functions checks wether the header & library versions
	  * are the same.
	  * Parameters are optionnal since they should represent the header
	  * version values which are defaulted.
	  * \return and returns false if there are incompatibilities.
	  */
	 void CheckVersion (int major = VERSION_MAJOR,
			    int minor = VERSION_MINOR,
			    int micro = VERSION_MICRO);

	 //==================================================================

	 /**
	  * Check if the program was run with --verbose
	  * \return true if the library is in verbose mode
	  */
	 virtual bool IsVerbose () const = 0;

	 /// Return an environment string.
	 String GetEnv (const String &name);

	 /// Set an environment string.
	 bool SetEnv(const String &name, const String &value);

	 //==================================================================

	 /**
	  * Get the Config object for the given system
	  */
	 virtual Config *Cfg() = 0;

	 /**
	  * Get the FS object for the given system
	  */
	 virtual FileSystem *FS () = 0;

	 /**
	  * Get the loader list, used to retrieve objects from files.
	  */
	 virtual Loaders *GetLoaders () = 0;

	 /**
	  * Get the factory list, used to construct implementations of 
          * abstract interfaces.
	  */
	 virtual FactoryList *Factories() = 0;

	 //==================================================================

	 /**
	  * Register a function which will be called whenever the program
	  * exits (normally or not) or when an "error" signal is received.
	  * (ie SIGSEGV, SIGINT, SIGHUP, etc).
	  */
	 virtual void AtExit (ExitFunc handler) = 0;

	 //==================================================================

	 /** Tell the system there was an error/warning/etc
	  *  \param el is the error level
	  *  \param format is a format string, as for printf
	  */
	 void Report (ErrorLevel el, const char *format, ...);
	 
	 /// Tell the system there was an error/warning/etc (stdarg version).
	 void Report (ErrorLevel el, const char *format, va_list args);

	 /// Print a log message.
	 void Log (const char *format, ...);

	 /// Print a warning message.
	 void Warning (const char *format, ...);

	 /// Print an error message
	 void Error (const char *format, ...);

	 /// Print an error message (and exit).
	 void Fatal (const char *format, ...);
   };

/**
 * \class System
 * Return the global system (created when System::Init is called)
 */
   extern ARK_DLL_API System* Sys();

}

#endif
