# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package ArchZoom;

use Exporter;
use vars qw(@ISA @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(apply_config);

# global constants
use vars qw($ConfigFile %Config);
use ArchZoom::Constants;

use Arch::TempFiles qw(temp_root);
use Arch::Util qw(arch_backend);

BEGIN {
	my $dir = $FindBin::Bin || ".";
	foreach (
		"$dir/../../archzoom-data",
		"$dir/../archzoom-data",
		"$dir/archzoom-data",
		"$dir/../perllib/..",
	) {
		$ConfDir = ($DataDir = $_) . "/conf" if -d $_;
	}
	$ConfigFile ||= "$ConfDir/archzoom.conf";

	# default config if archzoom.conf is not found
	%Config = (
		regular_archives => 1,
		library_archives => 1,
		name_alias_redirection => 1,
		name_alias_resolving => 0,
		debug_enabled => 1,
		http_headers => 'Robots: nofollow, index, archive\n',
		content_charset => "",
		globcats_enabled => 0,
		annotate_enabled => 0,
		arch_ids_listing => 0,
		implicit_expanded_dir_max_files => 50,
		explicit_expanded_dir_max_files => 1000,
		inline_file_max_size => "",
		javascript_expanding => 1,
		javascript_tooltip => 1,
		auto_archive_expanding => 0,
		revision_bunch_size => 100,
		revision_bunch_additive => 10,
		revision_summary_max_length => 0,
		revisions_reverse => 0,
		file_syntax_highlighting => [],
		abrowse_compact => 0,
		abrowse_transpose => 0,
		abrowse_transpose_changeable => 1,
		template_dir => "templates",
		template_set => "default",
		layout_css => "default",
		color_css => "default",
		library_lookup => 0,
		auto_library_updating => 0,
		fallback_revision_library => "",
		auto_temp_trees => 1,
		arch_backend => "",
		home_dir => $ENV{HOME} || "",
		unix_path => $ENV{PATH} || "",
		temp_dir => "/tmp",
		error_file => -e "/dev/null"? "/dev/null": "",
		mime_types_file => "/etc/mime.types",
		tarball_downloading => 0,
		run_limit_number => 0,
		run_limit_timeout => 0,
		run_limit_file => "",
		cache_dir => "",
		archives_cache => "",
		globcats_cache => "",
		abrowse_cache => "",
		revisions_cache => "",
		nocache_enabled => 1,
		disabled_archive_regexp => "",
	);
}

sub load_config (;$) {
	my $file_name = shift || $ConfigFile;
	open CONFIG, "<$file_name" or do {
		warn "No $file_name file, using default config\n";
		return 0;
	};
	while (<CONFIG>) {
		chomp;
		s/(^|\s+)#.*//;  # skip comments
		next if /^\s*$/;
		next if /^\s*#/;
		my ($key, $value) = split(/\s*=\s*/, $_, 2);
		my $old_value = $Config{$key};
		unless (defined $old_value) {
			warn "Unknown key $key in $file_name ignored\n";
			next;
		}
		$value = [ split(/,\s*/, $value) ] if ref($old_value) eq 'ARRAY';
		$Config{$key} = $value;
	}
	close CONFIG;
	return 1;
}

sub apply_config_immediately () {
	# the logic is touchy, try to work in all cases
	$ENV{HOME} = $Config{'home_dir'} if $Config{'home_dir'};
	$ENV{HOME} ||= $ArchZoom::DefaultHomeDir;
	$ENV{HOME} = "" unless $ENV{HOME} && -d $ENV{HOME};

	# try to autodetect $HOME
	unless ($ENV{HOME}) {
		my $user = $ENV{USER} || $ENV{USERNAME} || getpwuid($>);
		$user = $1 if !$user && ($FindBin::Bin || "") =~ m|/home/([^/]+)/|;
		$ENV{HOME} = "/home/$user" if $user && -d "/home/$user";
	}

	$Config{'arch_backend'} =~ s@~/@$ENV{HOME}/@g;
	arch_backend($Config{'arch_backend'}) if $Config{'arch_backend'};

	# replace ~ and relative paths in file/dir names
	foreach (qw(
		template_dir
		fallback_revision_library
		unix_path
		temp_dir
		error_file
		mime_types_file
		run_limit_file
		cache_dir
	)) {
		my $dir = $DataDir;
		$dir = $StateDir if $_ eq "run_limit_file";
		$dir = $CacheDir if $_ eq "cache_dir";
		$Config{$_} =~ s@^([^~/\.])@$dir/$1@;
		$Config{$_} =~ s@~/@$ENV{HOME}/@g;
	}

	open(STDERR, ">>$Config{'error_file'}")
		if $Config{'error_file'} && !$ENV{DEBUG} && !-t STDERR;

	# this should be done before url flags are parsed, so work with $ENV
	$Config{template_set} = $1
		if ($ENV{QUERY_STRING} || $ARGV[1] || "") =~ /(?:^|[&?])template=([\w-]+)/
			&& (-d "$Config{template_dir}/$1" || $1 eq "minimal");
	# temporarily support the old name for backward compatibility
	$Config{template_set} = "plain" if $Config{template_set} eq "minimal";

	$Config{http_headers} = join('', map { $_ ne ""? "$_\n": "" }
		split(/^\s*|\s*\\n\s*/, $Config{http_headers}));
}

sub apply_config (%) {
	my %flags = @_;

	# some sane checks; unlike apply_config_immediately, this may die
	die "No valid \$HOME set and can't auto-detect it, define home_dir in $ConfigFile\n"
		unless $ENV{HOME};
	die "No $ENV{HOME}/.arch-params/ found\n" unless -d "$ENV{HOME}/.arch-params";

	$Config{'temp_dir'} = $ENV{TMPDIR} if $ENV{TMPDIR};
	temp_root($Config{'temp_dir'});

	$ENV{PATH} = $Config{'unix_path'} || $ENV{PATH} || "/usr/bin:/bin:/usr/local/bin";

	$Config{'revision_bunch_size'} = 100 if $Config{'revision_bunch_size'} <= 0;

	if ($Config{'inline_file_max_size'} =~ /^(\d+)(|B|KB|MB)$/i) {
		my $factor = { '' => 1, B => 1, KB => 1024, MB => 1024 * 1024 }->{uc($2)};
		$Config{'inline_file_max_size'} = $1 * $factor;
	} else {
		$Config{'inline_file_max_size'} = 1024 * 1024 * 1024;
	}

	if ($flags{css}) {
		$flags{css} = "default-bright" if $flags{css} eq "bright";
		if ($flags{css} =~ /^(\w+)(?:-(\w+))?/) {
			$flags{layout} ||= $1;
			$flags{color} ||= $2;
		}
	}

	my $css_dir = "$Config{template_dir}/$Config{template_set}/css";
	foreach ('layout', 'color') {
		my $opt = $_ . "_css";
		my $file_prefix = $_ eq 'color'? 'color-': '';
		$Config{$opt} = $flags{$_} if $flags{$_} && $flags{$_} =~ /^\w/;
		$Config{$opt} = "default"
			unless $Config{$opt} && -f "$css_dir/$file_prefix$Config{$opt}.css";
		$Config{$opt} = "css/$file_prefix$Config{$opt}.css";
	}

	$Config{content_charset} = $flags{charset}
		if $flags{charset} && $flags{charset} =~ /^[\w-]+$/;

	foreach (qw(
		archives_cache
		globcats_cache
		abrowse_cache
		revisions_cache
	)) {
		if ($Config{$_} =~ /^(\d+)[,;\s]+(\d+)[,;\s]+(.*)$/) {
			$Config{$_} = {
				max_size => $1, expiration => $2, dir => $3,
				can_create => 1, perl_data => 1,
			};
			$Config{$_}->{dir} =~ s@^([^~/\.])@($Config{'cache_dir'} || $CacheDir) . "/$1"@e;
			$Config{$_}->{dir} =~ s@^~/@$ENV{HOME}/@;
		} else {
			$Config{$_} = undef;
		}
	}
}

BEGIN {
	load_config();
	apply_config_immediately();
}

1;
