# 
# wd-check-for-patch-log.sh - check for a project tree patch log 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "*check for a patch log in a project tree\\n"
		printf "usage: wd-check-for-patch-log [options] [archive/]version [dir]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archie archive           specify the archive name\\n"
		printf " -e --errname errname          set a name for error messages\\n"
		printf "\\n"
		printf " -v                            be verbose\\n"
		printf "\\n"
		printf "Check that the project tree containing DIR (or the current\\n"
		printf "directory) has a patch log for VERSION.  If so, exit with status 0\\n"
		printf "otherwise with status 1.\\n"
		printf "\\n"
		printf "With a value for ERRNAME, if the log is missing, print a message\\n"
		printf "telling the user about \"add-log\".\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
errname=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "wd-check-for-patch-log: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "wd-check-for-patch-log: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    -e|--errname)	shift
    			if test $# -eq 0 ; then
			  printf "wd-check-for-patch-log: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -*)			printf "wd-check-for-patch-log: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -eq 0 -o $# -gt 2 ; then
  printf "usage: wd-check-for-patch-log [options] archive/version [dir]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

archive_version="$1"

if test $# -eq 2 ; then
  dir="$2"
else
  dir="."
fi

cd "$dir"
dir="`pwd`"

################################################################
# Sanity Check
# 

larch valid-package-name -e wd-check-for-patch-log --vsn -- "$archive_version"

archive=`larch parse-package-name -R "$archroot" -A "$archive" --arch "$archive_version"`
version=`larch parse-package-name -R "$archroot" -A "$archive" --package-version "$archive_version"`
category=`larch parse-package-name --basename $version`
branch=`larch parse-package-name --package $version`

cd "$dir"
wdroot="`larch tree-root --accurate`"

################################################################
# Check
# 

cd "$wdroot"
if test ! -e "{arch}/$category/$branch/$version/$archive/patch-log" ; then
  if test "x$errname" != x ; then
    printf "%s: no patch log for version\\n" "$errname" 1>&2
    printf "\\n" 1>&2
    printf "  The project tree:\\n" 1>&2
    printf "     %s\\n" "$wdroot" 1>&2
    printf "  for version:\\n" 1>&2
    printf "     %s\\n" "$version" 1>&2
    printf "\\n"
    printf "Try \"larch add-log --help\"\\n" 1>&2
    printf "\\n"
  fi
  exit 1
else
  exit 0
fi
