# 
# valid-id.sh - validate an id string
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "validate an id string\\n"
		printf "usage: valid-id [options] [--] id-string\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -e --errname name             use ERRNAME for error messages\\n"
		printf "\\n"
		printf "Check the validity of a user id string.  Exit with status 0\\n"
		printf "if it is ok, 1 otherwise.\\n"
		printf "\\n"
		printf "A valid id string is one that matches this regexp (ERE syntax)\\n"
		printf "\\n"
		printf "    ^[[:alnum:][:space:][:punct:]]*<RE2@RE2\.RE2>\$"
		printf "\\n"
		printf "where RE2 is:\\n"
		printf "\\n"
		printf "    [-.[:alnum:]]+\\n"
		printf "\\n"
		printf "By convention, it should have the form of an email address,\\n"
		printf "as in this example:\\n"
		printf "\\n"
		printf "	Jane Hacker <jane.hacker@gnu.org>\\n"
		printf "\\n"
		printf "If you are going to use remote databases, it is very important that\\n"
		printf "the portion between < and > be a string that is globally, uniquely\\n"
		printf "assigned to you as an individual.  arch uses that portion of your\\n"
		printf "id string (along with some other data) to generate file names\\n"
		printf "which must be unique.\\n" 
		printf "\\n"
		printf "If the id string is invalid, and an error name is provided\\n"
 		printf "(-e or --errname) print an error message before exiting.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

errname=

while test $# -ne 0 ; do

  case "$1" in 

    --)			shift
    			break
			;;

    -e|--errname)	shift
    			if test $# -eq 0 ; then
			  printf "valid-id: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -*)			printf "valid-id: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: valid-id [options] -- id-string\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

idstr="$1"

################################################################
# Check the Id String
# 

set +e
filtered="`printf \"%s\\n\" \"$idstr\" | grep -E \"^[[:alnum:][:space:][:punct:]]*<[-._[:alnum:]]+@[-._[:alnum:]]+\\.[-._[:alnum:]]+>\$\"`"
set -e

if test "x$filtered" = x ; then
  if test "x$errname" != x ; then
    printf "%s: invalid id string (%s)\\n" "$errname" "$idstr" 1>&2
  fi
  exit 1
fi

exit 0

