# 
# find-in-cache.sh - find a revision in the local cache
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "find a revision in the local cache\\n"
		printf "usage: find-in-cache [options] cachedir revision\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf "Look for a cached copy of REVISION in the project tree\\n"
		printf "which are immediate subdirectories of CACHEDIR.  If one is found,\\n"
		printf "print its location and exit with status 0.  Otherwise,\\n"
		printf "exit with status 1.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    --)			shift
			break
			;;

    -*)			printf "find-in-cache: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 2 ; then
  printf "usage: find-in-cache [options] cachedir revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

cachedir="$1"
shift

revision="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

larch valid-package-name -l -e find-in-cache -- "$revision"

archive="`larch parse-package-name --arch -R \"$archroot\" -A \"$archive\" \"$revision\"`"
category=`larch parse-package-name -b $revision`
branch=`larch parse-package-name $revision`
vsn=`larch parse-package-name -v $revision`
lvl=`larch parse-package-name -l $revision`

################################################################
# Look for Cached Copies
# 

here="`pwd`"
cd "$cachedir"
cachedir="`pwd`"

for wd in [=a-zA-Z+]* ; do
  for state in locked unlocked ; do
    if test -d "$wd/{arch}/++pristine-trees/$state/$category/$branch/$branch--$vsn/$archive/$branch--$vsn--$lvl" ; then
      cd "$wd/{arch}/++pristine-trees/$state/$category/$branch/$branch--$vsn/$archive/$branch--$vsn--$lvl"
      pwd
      exit 0
    fi
  done
done

exit 1

