# copy-from-library.sh: try to copy a revision from the library
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "try to copy a revision from the library\\n"
		printf "usage: copy-from-library [options] revision\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf "Look for a library copy of REVISION.  If one is found,\\n"
		printf "copy it to the current directory and exit with status 0.,\\n"
		printf "Otherwise exit with status 1.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

quiet=--quiet
report=--report
verbose=

while test $# -ne 0 ; do

  case "$1" in 

    --silent)	shift
    		quiet=
		report=
		verbose=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		;;

    --debug)	shift
    		larch heading "copy-from-library: debugging output enabled\\n"
    		set -x
		;;


    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-category: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-category: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    --)			shift
			break
			;;

    -*)			printf "copy-from-library: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: copy-from-library [options] revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

revision="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

tree_location="`larch library-find --silent -R \"$archroot\" -A \"$archive\" \"$revision\"`"

# This point only reached if the revision is in the library.
# 

here="`pwd`"

################################################################
# Greetings
# 
#  This is an internal command.

if test ! -z "$verbose" ; then 
  ARCH__OUTLINE_DEPTH="$ARCH__OUTLINE_DEPTH*"
  larch heading "copy-from-library\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "copy-from-library start time: %s\\n" "`date`"
  larch heading --sub "copying from: %s\\n" "$tree_location"
  larch heading --sub "copying to: %s\\n" "$here"
fi

################################################################
# Copy It
# 


if test ! -z "$report" ; then
  larch heading --sub "copying revision from library\\n" 
  larch heading --sub --sub "dir: %s\\n" "$tree_location"
fi

cat "$tree_location/,,index" \
| cut -f 1 \
| copy-file-list -- - "$tree_location" "$here"

if test ! -z "$verbose" ; then 
  larch heading --sub "copy-from-library finish time: %s\\n" "`date`"
fi


# tag: Tom Lord Thu Jan 24 19:57:10 2002 (library/copy-from-library.sh)
#
