# 
# set-manifest.sh - set the inventory manifest
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "set the inventory manifest from project tree\\n"
                printf "usage: set-manifest [options]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -f --force                    overwrite an existing manifest\\n"
                printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
                printf "\\n"
                printf "Set the inventory manifest of a project tree to the current\\n"
		printf "contents of the tree.\\n"
                printf "\\n"
                printf "The inventory manifest is a list of files previously recorded\\n"
                printf "as the expected file inventory.\\n"
                printf "\\n"
                printf "To record the current contents of a tree as the manifest,\\n"
                printf "use commands such as:\\n"
                printf "\\n"
                printf "See also \"larch manifest\" and \"larch check-manifest\".\\n"
                printf "\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

force=
dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "set-manifest: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    -f|--force)		shift
    			force=yes
			;;

    --)			shift
    			break
			;;
			
    -*)                 printf "set-manifest: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 0 ; then
  printf "usage: set-manifest [options]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root`"

manifest="$wdroot/{arch}/=manifest"

if test -z "$force" -a -e "$manifest" ; then

  printf "set-manifest: manifest already exists\\n" 1>&2
  printf "\\n" 1>&2
  printf "Try \"larch set-manifest --help\".\\n" 1>&2
  exit 1

fi


################################################################
# Store
# 

cd "$wdroot/{arch}" 

tmpfile="$wdroot/{arch}/,,manifest.$$"

rm -rf "$tmpfile"

bail()
{
  cd "$wdroot"
  rm -rf "$tmpfile"
  exit 1
}

finish()
{
  cd "$wdroot"
  rm -rf "$tmpfile"
  exit 0
}

trap "printf \"set-manifest: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

cd "$wdroot"
larch inventory --source --tags \
| sort -k 2 \
> "$tmpfile"

dangerous-rename "$tmpfile" "$wdroot/{arch}/=manifest"

