# 
# prepare-branch.sh: make a project tree for a new branch
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "create a project tree for a new branch\\n"
                printf "usage: prepare-branch [options] \\\\\\n"
		printf "                      [archive/]old-revision \\\\\\n"
		printf "                      [archive/]version \\\\\\n"
		printf "                      dir\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
                printf "Construct a new project tree, DIR, from OLD-REVISION.\\n"
		printf "\\n"
                printf "Make VERSION the default version for that tree.\\n"
		printf "\\n"
                printf "Initialize an empty log entry for commit to the new version\\n"
		printf "\\n"
		printf "This can be followed by \"larch finish-branch\" to create a new\\n"
		printf "branch in an archive.\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=
__restart=

while test $# -ne 0 ; do

  case "$1" in 

    ----restart) shift
    		__restart=----restart
		;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "prepare-branch: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "prepare-branch: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "prepare-branch: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;
			
    -*)                 printf "prepare-branch: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 3 ; then
  printf "usage: prepare-branch [options] \\\\\\n" 1>&2
  printf "                      [archive/]old-revision \\\\\\n" 1>&2
  printf "                      [archive/]version \\\\\\n" 1>&2
  printf "                      dir\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

old_revision="$1"
shift

new_version="$1"
shift

dir="$1"
shift



################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "`dirname \"$dir\"`"

parent="`pwd`"
dir="$parent/`basename \"$dir\"`"

cd "$parent"

larch valid-package-name -e prepare-branch --tolerant -- "$old_revision"
larch valid-package-name -e prepare-branch --tolerant -- "$new_version"

old_archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$old_revision\"`"
old_spec="`larch parse-package-name --non-arch \"$old_revision\"`"
new_archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$new_version\"`"
new_spec="`larch parse-package-name --non-arch \"$new_version\"`"

################################################################
# Greetings
# 

if test "(" -z "$__restart" -a ! -z "$quiet" ")" -o ! -z "$report" ; then
  larch heading "prepare-branch\\n"
  printf "arguments: %s\\n"  "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "prepare-branch start time: %s\\n" "`date`"
  larch heading --sub "old archive: %s\\n" "$old_archive"
  larch heading --sub "old revision spec: %s\\n" "$old_spec"
  larch heading --sub "new archive: %s\\n" "$new_archive"
  larch heading --sub "new version spec: %s\\n" "$new_spec"
  larch heading --sub "output directory: %s\\n" "$dir"
fi

################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$old_archive" ; then

  if test ! -z "$quiet" ; then
    larch heading --sub "restarting with connection to archive\\n"
  fi

  exec larch with-archive -A "$old_archive"  \
	    larch prepare-branch \
		 $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
		 ----restart \
		 "$old_archive/$old_spec" "$new_archive/$new_spec" "$dir"

fi

################################################################
# What Revision are we Branching From
# 

old_revision="`larch indicated-revision -e prepare-branch \"$old_archive/$old_spec\"`"
old_revision="`larch parse-package-name --non-arch \"$old_revision\"`"

if test ! -z "$quiet" ; then
  larch heading --sub "old revision: %s\\n" "$old_revision"
fi

################################################################
# What Version are we Branching To
# 

if larch valid-package-name --vsn "$new_spec" ; then
  new_version="$new_archive/$new_spec"
else
  new_version="`larch versions --reverse \"$new_archive/$new_spec\" | head -1`"
fi

if test -z "$new_version" ; then
  printf "\\n" 1>&2
  printf "prepare-branch: no versions found\\n" 1>&2
  printf "  archive: %s\\n" "$new_archive" 1>&2
  printf "  version spec: %s\\n" "$new_spec" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

if test ! -z "$quiet" ; then
  larch heading --sub "new version: %s\\n" "$new_version"
fi

################################################################
# Do It
# 

if test ! -z "$quiet" ; then
  larch heading --sub "fetching the BRANCH FROM revision\\n"
fi

larch nested --sub larch get \
		       $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
		       -A "$old_archive" \
		       "$old_revision" "$dir"

if test ! -z "$quiet" ; then
  larch heading --sub "switching new directory to branch version\\n"
fi

cd "$dir"
larch add-log "$new_version"
larch set-tree-version "$new_version"

if test ! -z "$quiet" ; then
  larch heading --sub "creating log file template\\n"
fi

log_template="`larch make-log`"
printf "branched from %s\\n" "$old_archive/$old_revision" >> "$log_template"
printf "\\n" >> "$log_template"

if test ! -z "$quiet" ; then
  printf "%s\\n" "$log_template" | larch body-indent --sub
fi

rm -f "{arch}/++branch-revision"

printf "%s\\n" "$old_archive/$old_revision" > "{arch}/++branch-revision"

if test ! -z "$quiet" ; then
  larch heading --sub "prepare-branch finish time: %s\\n" "`date`"
fi

# tag: Tom Lord Mon Dec 10 19:15:50 2001 (branching-and-merging/prepare-branch.sh)
#
