# make-sync-tree.sh: prepare a tree that will synchronize branches
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "prepare a tree that will synchronize branches\\n"
                printf "usage: get [options] [archive/]from-revision [archive/]to-version [dir]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
		printf "  --cache DIR                  cache directory for locally cached\\n"
                printf "                                 revisions\\n"
                printf "\\n"
                printf " --no-pristine                 don't save a pristine copy\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf "Extract FROM-REVISION from an archive, but modify its patch log and\\n"
		printf "tree-version to match the latest revision TO-VERSION.\\n"
		printf "\\n"
		printf "In effect, this merges FROM-REVISION into the latest revision of TO-VERSION\\n"
                printf "but in such a way that the resulting source tree exactly matches FROM-REVISION.\\n"
		printf "The tree can then be committed, creating a new revision in TO-VERSION with\\n"
		printf "no differences (apart from new patch logs and automatic ChangeLog entries) from\\n"
		printf "FROM-REVISION.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
cache_dir=

no_pristine=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --no-pristine)	shift
    			no_pristine=--no-pristine
			;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "get: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "get: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "get: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --cache)            shift
                        if test $# -eq 0 ; then
                          printf "get: --cache requires an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        cache_dir="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;

    -*)                 printf "get: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -lt 2 -o $# -gt 3 ; then
  printf "usage: get [options] [archive/]from-revision [archive/]to-version [dir]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

from_spec="$1"
shift

to_spec="$1"
shift

if test $# -eq 1 ; then
  parent="`dirname \"$1\"`"
  dirname="`basename \"$1\"`"
else
  parent="."
  dirname=
fi

################################################################
# Sanity Check and Process Defaults
# 
  
cd "$parent"
parent="`pwd`"

larch valid-package-name -e make-sync-tree -t -- "$from_spec"
larch valid-package-name -e make-sync-tree -t -- "$to_spec"

if test -z "$cache_dir" ; then
  cache_dir="$parent"
fi

################################################################
# Greetings
# 

if test ! -z "$quiet" ; then
  larch heading "make-sync-tree\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "make-sync-tree start time: %s\\n" "`date`"
  larch heading --sub "output dir: %s\\n" "$parent/$dirname"
  larch heading --sub "from revision spec: %s\\n" "$from_spec"
  larch heading --sub "to version spec: %s\\n" "$to_spec"
fi

################################################################
# Create a Temp Dir
# 

tmpdir="$parent/,,make-sync-tree.$$"

bail()
{
  rm -rf "$tmpdir"
  exit 1
}

trap "printf \"make-sync-tree: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

mkdir "$tmpdir"

################################################################
# Get the Revision of $from_spec
# 

cd "$tmpdir"
larch nested larch get -R "$archroot" -A "$archive" "$from_spec" from


################################################################
# Get the Revision of $to_spec
# 

cd "$tmpdir"
larch nested larch get -R "$archroot" -A "$archive" "$to_spec" to

################################################################
# Make Sure We Have an Output Directory Name
#

cd "$tmpdir"
if test -z "$dirname" ; then
  cd to
  dirname="`larch tree-version`"
  cd ..
  if test -e "../$dirname" ; then
    x=1;
    while test -e "../$dirname--sync.$x" ; do
      x=$(($x + 1))
    done
    dirname="../$dirname--sync.$x"
  fi
fi


################################################################
# Merge the Meta-Data
# 

cd "$tmpdir"
(cd "from/{arch}" ; find . -type f) \
| copy-file-list --no-overwrite -- - "from/{arch}" "to/{arch}"

################################################################
# Put the "to" Meta-Data in the "from" Tree
# 

cd "$tmpdir"
mv "from/{arch}" "old-{arch}"
mv "to/{arch}" from

################################################################
# Give the User the Completed Tree and Clean Up
# 

if test -e "../$dirname" ; then
  saved="++saved-$dirname.`date +%Y-%m-%d`.$$.bak"
  mv "../$dirname" "../$saved"
fi
mv from "../$dirname"
cd ..
rm -rf "$tmpdir"

# tag: Tom Lord Sun Feb 17 20:25:04 2002 (branching-and-merging/make-sync-tree.sh)
#
