# 
# make-archive.sh - make a new archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

###############################################################
# Format Version Identifier
# 
# These values are used to identify versions of the archive
# directory format.  Please do not change them.
# 
fmtvsn=1
fmtstr="Hackerlab arch archive directory, format version $fmtvsn."



################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "create a new archive directory\\n"
		printf "usage: make-archive [options] (-u | name) directory\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -r --readme file              save FILE as the README\\n"
	        printf "                               for this archive\\n"
		printf "\\n"
		printf " -u --update                   update an existing archive\\n"
		printf "\\n"
		printf "NAME is the global name for the archive.  It must be an\\n"
		printf "email address with a fully qualified domain name, optionally\\n"
		printf "followed by \"--\" and a string of letters, digits, periods\\n"
		printf "and dashes.\\n"
		printf "\\n"
		printf "Normally the archive directory must not already exist, but.\\n"
		printf "there is an exception:\\n"
		printf "\\n"
		printf "If -u or --update is specified, then attempt to bring\\n"
		printf "an existing archive up to date with respect to this\\n"
		printf "version of larch and update meta-info, such as the README.\\n"
		printf "file.\\n"
		printf "\\n"
		printf "If -u or --update is specified, the archive name must\\n"
		printf "not be specified.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 


readme=
update=0

while test $# -ne 0 ; do

  case "$1" in 

    -u|--update)	update=1
    			shift
			;;

    -r|--readme)	shift
			if test $# -eq 0 ; then
			  printf "make-archive: -R and --readme require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			readme="$1"
			shift
			;;

    -*)			printf "make-archive: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $update -eq 0 ; then

  if test $# -ne 2 ; then
    printf "usage: make-archive [options] name dir\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
  fi

  name="$1"
  dir="$2"

else

  if test $# -ne 1 ; then
    printf "usage: make-archive [options] -u dir\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
  fi

  dir="$1"
fi

cd "`dirname \"$dir\"`"
dir="`pwd`/`basename \"$dir\"`"

################################################################
# Sanity Check 
# 

if test $update -eq 0 ; then

  larch valid-archive-name -e "make-archive" -- "$name"

  if larch whereis-archive "$name" 1>&2 2> /dev/null ; then
    printf "make-archive: archive already registered\\n" 1>&2
    printf "\\n" 1>&2
    printf "  The archive %s\\n" "$name" 1>&2
    printf "  is already on your list of registered archives.\\n" 1>&2
    printf "  (See \"larch archives --help\".)\\n" 1>&2
    printf "\\n" 1>&2
    printf "  Either choose a different name, or unregister\\n" 1>&2
    printf "  the name you chose, but be careful not to create\\n" 1>&2
    printf "  two archives with the same name.  (See \"larch \\n" 1>&2
    printf "  register-archive --help\".)\\n" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi

else

  if test ! -d "$dir" -o ! -e "$dir/.archive-version" ; then
    printf "make-archive: not an archive directory (%s)\\n" "$dir" 1>&2
    exit 1
  fi

  oldvsn=`cat "$dir/.archive-version"`

  oldvsnstr=`printf "%s\\n" "$oldvsn" | sed -e "s/[0-9][0-9]*//" -e "/[0-9]/d"`
  oldvsnnum=`printf "%s\\n" "$oldvsn" | sed -e "s/[^0-9]*//g" `

  thisvsnstr=`printf "%s\\n" "$fmtstr" | sed -e "s/[0-9][0-9]*//" -e "/[0-9]/d"`
  thisvsnnum=$fmtvsn

  if test "x$oldvsnstr" != "x$thisvsnstr" ; then
    printf "Can not update old archive -- unrecognized format string\\n" 1>&2
    printf "  (old format: %s)\\n" "$oldvsn" 1>&2
    printf "  (current format: %s)\\n" "$fmtstr" 1>&2
    printf "  see %s/.archive-version" "$dir" 1>&2
    exit 1
  fi

  if test "x$oldvsnnum" = x ; then
    printf "Can not update old archive -- illegal format string (no version number)\\n" 1>&2
    printf "  (format string: %s)\\n" "$fmtstr" 1>&2
    printf "  see %s/.archive-version" "$dir" 1>&2
    exit 1
  fi

  if test $oldvsnnum -gt $thisvsnnum ; then
    printf "Can not update old archive -- format version too new!\\n" 1>&2
    printf "  Format version in archive: %d\\n" $oldvsnnum 1>&2
    printf "  Format version of this version of arch: %d\\n" $thisvsnnum 1>&2
    printf "  see %s/.archive-version" "$dir" 1>&2
    exit 1
  fi

fi

################################################################
# Create the Directory
# 

if test $update -eq 0 ; then

  mkdir -p "$dir"

fi

################################################################
# Create the Meta-info Directory
# 

if test ! -e "$dir/=meta-info" ; then
  mkdir "$dir/=meta-info"
elif test ! -d "$dir/=meta-info" ; then
  printf "Corrupt archive: =meta-info is not a directory.\\n" 1>&2
  exit 1
fi


if test $update -eq 0 ; then
  printf "%s\\n" "$name" > "$dir/=meta-info/name"
fi

if test "x$readme" != x ; then
  if test -e "$dir/=meta-info/README" ; then
    rm -f "$dir/=meta-info/README~"
    mv "$dir/=meta-info/README" "$dir/=meta-info/README~"
  fi
  cp "$readme" "$dir/=meta-info/README"
fi


################################################################
# Create .archive-version
# 

if test -e "$dir/.archive-version" ; then 
  rm -f  "$dir/.archive-version~"
  mv  "$dir/.archive-version"  "$dir/.archive-version~"
fi

printf "%s\\n" "$fmtstr" > "$dir/.archive-version"

################################################################
# Register the Archive Location
# 

larch register-archive "$name" "$dir"

