// pkg_item.cc
//
// Copyright 1999,2000,2001,2002 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Definitions of stuff in pkg_item.h

#include "aptitude.h"

#include "vscreen/config/keybindings.h"
#include "vscreen/vs_multiplex.h"
#include "vscreen/vs_pager.h"
#include "vscreen/vs_util.h"
#include "vscreen/config/column_definition.h"

#include "pkg_columnizer.h"
#include "download.h"
#include "edit_pkg_hier.h"
#include "pkg_description_treeitem.h"
#include "pkg_item.h"
#include "vs_progress.h"

#include "download_bar.h"
#include "download_manager.h"

#include "ui.h"

#include "generic/pkg_changelog.h"
#include "generic/apt.h"
#include "generic/matchers.h"
#include "generic/config_signal.h"

#include <apt-pkg/configuration.h>
#include <apt-pkg/error.h>

#include <sigc++/bind.h>

#include <iostream>

using namespace std;

static const char *confirm_str=N_("Yes, I am aware this is a very bad idea");

static void try_delete_essential(string s,
				 const pkgCache::PkgIterator pkg,
				 bool purge)
{
  if(s==_(confirm_str))
    {
      undo_group *grp=new apt_undo_group;

      (*apt_cache_file)->mark_delete(pkg, purge, false, grp);

      if(grp->empty())
	delete grp;
      else
	apt_undos->add_item(grp);
    }
}

static void confirm_delete_essential(const pkgCache::PkgIterator &pkg,
				     bool purge)
{
  assert((pkg->Flags&pkgCache::Flag::Essential)==pkgCache::Flag::Essential ||
	 (pkg->Flags&pkgCache::Flag::Important)==pkgCache::Flag::Important);

  char buf[512];
  snprintf(buf, 512, _("%s is an essential package!\n\nAre you sure you want to remove it?\nType '%s' if you are."), pkg.Name(), _(confirm_str));

  vscreen_widget *w=vs_dialog_string(buf,
				     "",
				     bind(SigC::slot(try_delete_essential),
					  pkg, purge),
				     NULL,
				     NULL,
				     NULL);

  w->show_all();

  popup_widget(w);
}

class pkg_changelog_screen:public vs_file_pager
// Needed to make sure the changelog is properly cleaned up.
{
  pkg_changelog *cl;
public:
  pkg_changelog_screen(pkg_changelog *_cl, vscreen_widget *owner=NULL, int x=0, int y=0, int width=0, int height=0):
    vs_file_pager(_cl->get_filename()),cl(_cl)
  {
  }

  virtual ~pkg_changelog_screen() {delete cl;}
};

void pkg_item::highlighted(vs_tree *win)
{
  if(info_signal)
    (*info_signal)(package, visible_version());
}

void pkg_item::unhighlighted(vs_tree *win)
{
  if(info_signal)
    (*info_signal)(pkgCache::PkgIterator(),
		   pkgCache::VerIterator(*apt_cache_file));
}

void pkg_item::do_select(undo_group *undo)
{
	(*apt_cache_file)->mark_install(package, aptcfg->FindB(PACKAGE "::Auto-Install", true), false, undo);
}

void pkg_item::select(undo_group *undo)
{
  if(aptcfg->FindB(PACKAGE "::UI::New-Package-Commands", true))
    do_select(undo);
  else if(!(*apt_cache_file)[package].Delete())
    do_select(undo);
  else
    do_hold(undo);
}

void pkg_item::do_hold(undo_group *undo)
  // Sets an explicit hold state.
{
  (*apt_cache_file)->mark_keep(package, false, true, undo);
}

void pkg_item::hold(undo_group *undo)
  // Sets an /explicit/ hold state.  May be useful for, eg, saying that the
  // current package version (which is the newest) should be kept even if/when
  // a newer version becomes available.
{
  if(aptcfg->FindB(PACKAGE "::UI::New-Package-Commands", true))
    do_hold(undo);
  else
    // Toggle the held state.
    (*apt_cache_file)->mark_keep(package,
				 false,
				 (*apt_cache_file)->get_ext_state(package).selection_state!=pkgCache::State::Hold,
				 undo);
}

void pkg_item::do_remove(undo_group *undo)
{
  if((package->Flags&pkgCache::Flag::Essential)==pkgCache::Flag::Essential ||
     (package->Flags&pkgCache::Flag::Important)==pkgCache::Flag::Important)
    confirm_delete_essential(package, false);
  else
    (*apt_cache_file)->mark_delete(package, false, false, undo);
}

void pkg_item::remove(undo_group *undo)
{
  if(aptcfg->FindB(PACKAGE "::UI::New-Package-Commands", true))
    do_remove(undo);
  else if(!(*apt_cache_file)[package].Install() && !((*apt_cache_file)[package].iFlags&pkgDepCache::ReInstall))
    do_remove(undo);
  else
    {
      if((*apt_cache_file)[package].iFlags&pkgDepCache::ReInstall)
	(*apt_cache_file)->mark_keep(package, false, false, undo);
      else
	(*apt_cache_file)->mark_keep(package, false, (*apt_cache_file)[package].Status==1, undo);
    }
}

// No "do_purge" because purge was always idempotent.
void pkg_item::purge(undo_group *undo)
{
  if((package->Flags&pkgCache::Flag::Essential)==pkgCache::Flag::Essential ||
     (package->Flags&pkgCache::Flag::Important)==pkgCache::Flag::Important)
    confirm_delete_essential(package, false);
  else
    (*apt_cache_file)->mark_delete(package, true, false, undo);
}

void pkg_item::reinstall(undo_group *undo)
{
  if(!package.CurrentVer().end())
    (*apt_cache_file)->mark_install(package,
				    aptcfg->FindB(PACKAGE "::Auto-Install", true),
				    true,
				    undo);
}

void pkg_item::set_auto(bool isauto, undo_group *undo)
{
  (*apt_cache_file)->mark_auto_installed(package, isauto, undo);
}

int pkg_item::get_normal_attr()
{
  if(package.VersionList().end())
    {
      for(pkgCache::PrvIterator i=package.ProvidesList(); !i.end(); i++)
	{
	  if(!i.OwnerPkg().CurrentVer().end())
	    return A_BOLD|vs_treeitem::get_normal_attr();
	}
      return vs_treeitem::get_normal_attr();
    }
  else
    {
      pkgDepCache::StateCache &state=(*apt_cache_file)[package];

      if(!state.InstBroken() &&
	 (state.NewInstall() || (state.iFlags&pkgDepCache::ReInstall)))
	return get_color("PkgToInstall");
      else if(state.Status!=2 && // Not being upgraded
	      (*apt_cache_file)->get_ext_state(package).selection_state==pkgCache::State::Hold // Flagged for hold
	      && !state.InstBroken()) // Not currently broken.
	return get_color("PkgToHold");
      else if(state.Delete())
	return get_color("PkgToRemove");
      else if(state.InstBroken())
	return get_color("PkgBroken");
      else if(state.Upgrade())
	return get_color("PkgToUpgrade");
      else if(package.CurrentVer().end())
	return vs_treeitem::get_normal_attr();
      else
	return A_BOLD|vs_treeitem::get_normal_attr();
    }
}

void pkg_item::paint(vs_tree *win, int y, bool hierarchical)
{
  int basex=hierarchical?2*get_depth():0;
  int width, height;

  win->getmaxyx(height, width);
  pkg_columnizer::setup_columns();

  string disp=pkg_columnizer(package, visible_version(), pkg_columnizer::get_columns(), basex).layout_columns(width);
  win->mvaddnstr(y, 0, disp.c_str(), width);
}

bool pkg_item::dispatch_char(chtype ch, vs_tree *owner)
{
  if(bindings->key_matches(ch, "Versions"))
    {
      vscreen_widget *w=make_ver_screen(package);
      insert_main_widget(w, string(_("Available versions of "))+package.Name(),
			 "");
    }
  else if(bindings->key_matches(ch, "Dependencies"))
    {
      if(!visible_version().end())
	{
	  char buf[512];
	  snprintf(buf, 512, _("Dependencies of %s"), package.Name());

	  vscreen_widget *w=make_dep_screen(package, visible_version());
	  insert_main_widget(w, buf, "");
	  w->show();
	}
    }
  else if(bindings->key_matches(ch, "ReverseDependencies"))
    {
      char buf[512];
      snprintf(buf, 512, _("Packages depending on %s"), package.Name());

      vscreen_widget *w=make_dep_screen(package, visible_version(), true);
      insert_main_widget(w, buf, "");
    }
  else if(bindings->key_matches(ch, "Description"))
    {
      if(!visible_version().end())
	{
	  char buf[512];
	  snprintf(buf, 512, _("Description of %s"), package.Name());

	  vscreen_widget *w=new pkg_description_screen(visible_version());
	  insert_main_widget(w, buf, "");
	}
    }
  else if(bindings->key_matches(ch, "InfoScreen"))
    {
      char buf[512];
      snprintf(buf, 512, _("Information about %s"), package.Name());

      vscreen_widget *w=make_info_screen(package, visible_version());
      insert_main_widget(w, buf, "");
    }
  else if(bindings->key_matches(ch, "Changelog"))
    {
      download_manager *widget=gen_download_progress(true);
      pkg_changelog *cl=get_changelog(package.Name(), widget);
      widget->Complete();
      if(cl)
	{
	  char buf[512];
	  snprintf(buf, 512, _("ChangeLog of %s"), package.Name());

	  pkg_changelog_screen *newscreen=new pkg_changelog_screen(cl);
	  insert_main_widget(newscreen, buf, "");
	}
    }
  else if(bindings->key_matches(ch, "InstallSingle"))
    {
      if((*apt_cache_file)[package].CandidateVerIter(*apt_cache_file).end())
	return true;

      undo_group *grp=new apt_undo_group;
      (*apt_cache_file)->mark_single_install(package, grp);
      if(!grp->empty())
	apt_undos->add_item(grp);
      else
	delete grp;
    }
  else if(bindings->key_matches(ch, "DpkgReconfigure"))
    // Don't bother with my internal su-to-root stuff here, since I don't
    // need to touch the package lists in the subprocess.
    {
      // Try to do *something*.
      char *sucmd=NULL;

      if(getuid()==0)
	sucmd="dpkg-reconfigure '%s'";
      else if(access("/usr/sbin/su-to-root", X_OK)==0)
	sucmd="/usr/sbin/su-to-root -c \"/usr/sbin/dpkg-reconfigure '%s'\"";
      else if(access("/bin/su", X_OK)==0)
	sucmd="/bin/su -c \"/usr/sbin/dpkg-reconfigure '%s'\"";
      else
	popup_widget(vs_dialog_ok(_("You are not root and I cannot find any way\nto become root.  To reconfigure this package,\ninstall the menu package, the login package,\nor run aptitude as root.")));

      if(sucmd)
	{
	  vscreen_suspend();

	  apt_cache_file->ReleaseLock();

	  printf("Reconfiguring %s\n", package.Name());

	  char buf[512];
	  if(sucmd)
	    {
	      snprintf(buf, 512, sucmd,
		       package.Name());

	      system(buf);

	      cerr<<_("Press return to continue.\n");
	      getchar();

	      vscreen_resume();

	      apt_reload_cache(gen_progress_bar(), true);
	    }
	}
    }
  else if(bindings->key_matches(ch, "EditHier"))
    {
      vs_hier_editor *e=new vs_hier_editor;
      e->set_package(package, visible_version());

      // FIXME: better title
      add_main_widget(e, _("Hierarchy editor"), "");

      e->connect_key("Quit", &global_bindings,
		     slot(e, &vscreen_widget::destroy));
    }
  else
    return pkg_tree_node::dispatch_char(ch, owner);

  return true;
}

void pkg_item::dispatch_mouse(short id, int x, mmask_t bstate, vs_tree *owner)
{
  if(bstate & (BUTTON1_DOUBLE_CLICKED | BUTTON2_DOUBLE_CLICKED |
	       BUTTON3_DOUBLE_CLICKED | BUTTON4_DOUBLE_CLICKED |
	       BUTTON1_TRIPLE_CLICKED | BUTTON2_TRIPLE_CLICKED |
	       BUTTON3_TRIPLE_CLICKED | BUTTON4_TRIPLE_CLICKED))
    {
      char buf[512];
      snprintf(buf, 512, _("Information about %s"), package.Name());

      vscreen_widget *w=make_info_screen(package, visible_version());
      insert_main_widget(w, buf, "");
    }
}

bool pkg_item::matches(string s) const
{
  return pkg_matches(s, package, visible_version());
}

pkgCache::VerIterator pkg_item::visible_version(const pkgCache::PkgIterator &pkg)
{
  pkgDepCache::StateCache &state=(*apt_cache_file)[pkg];

  if(pkg.CurrentVer().end())
    return state.CandidateVerIter(*apt_cache_file);
  else if(state.Install())
    return state.InstVerIter(*apt_cache_file);
  // Return the to-be-installed version
  else
    return pkg.CurrentVer();
}
