//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006-2012 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/un.h>
#include <termios.h>
#include <unistd.h>

#ifdef USE_BLUEZ
#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#endif

#ifdef USE_BT_FBSD
#include <bluetooth.h>
#include <sdp.h>
#include <err.h>
#endif

#include <glib.h>

#include "btio.h"
#include "common.h"
#include "parse.h"
#include "utils.h"
#include "conf.h"

extern char tmp[MAXMAXLEN];
extern gboolean stillRun;

static int portfd       = -1;
static int sportfd      = -1;
static int portflags    = 0;
static int oldflags     = 0;
static int rtscts       = 0;

static int ivHeartbeat  = 0;
static int ivInCommand  = 0;


static struct termios oldterm;
static struct termios portterm;

char * uSocket = NULL;

static char * btAddress = NULL;

int openSerialPort(char* port)
{

    /* get a file descriptor */
    if ((portfd = open(port, O_RDWR|O_NOCTTY|O_NDELAY/*|O_CLOEXEC fails to compile ? */)) < 0) {
        sprintf(tmp, "can not open %s", port);
        logger(L_ERR, tmp);
        printf("ERROR: can not open %s\n", port);
        portfd = -1;
        return -1;
    }
    fcntl(portfd,F_SETFD,FD_CLOEXEC);

    /* check to see that the file is a terminal */
    if (!isatty(portfd)) {
        printf("ERROR: %s does not refer to a terminal\n", port);
        return -1;
    }

    /* get port attributes, store in oldterm */
    if (tcgetattr(portfd, &oldterm) < 0) {
        printf("ERROR: Can not get port attributes (%s)\n", port);
        return -1;
    }

    /* get port flags, save in oldflags */
    if ((oldflags = fcntl(portfd, F_GETFL)) < 0) {
        printf("ERROR: Can't get port flags (%s)\n", port);
        return -1;
    }

    portterm = oldterm;
    portflags = oldflags;

    portterm.c_cflag = getBaudrate() | CRTSCTS | CS8 | CLOCAL | CREAD | O_NDELAY;
    if (!rtscts) {
        portterm.c_cflag &= ~CRTSCTS;
    }
    portterm.c_iflag	= IGNPAR;
    portterm.c_oflag	= 0;
    portterm.c_lflag	= 0;
    portterm.c_cc[VTIME]	= 0;
    portterm.c_cc[VMIN]	= 0;

    tcflush(portfd, TCIOFLUSH);

    if (tcsetattr(portfd,TCSANOW,&portterm) < 0) {
        printf("ERROR: Can't set port attributes (%s)\n", port);
        return -1;
    }

    /* set non-blocking */
    if (fcntl(portfd, F_SETFL, (portflags |= O_NONBLOCK)) < 0) {
        printf("ERROR: Can't set port flags (%s)\n", port);
        return -1;
    }
    return 0;
}

// Then port will be closed from a forked child use 0
int closeSerialPort(int final)
{
    int retval = 0;

    logger(L_INF, "closeSerialPort");
    //printf("closeSerialPort (%d) (%d)\n", final, portfd);

    if (portfd < 0) { /* already closed */
        if (final) {
            logger(L_INF, "Already closed ?");
        }
        return -1;
    }

    if (final) {
        /* restore old settings */
        if (tcsetattr(portfd, TCSADRAIN, &oldterm) < 0) {
            retval = -1;
        }

        if (fcntl(portfd, F_SETFL, oldflags) < 0) {
            retval = -1;
        }
    }
    retval = close(portfd);
    portfd = -1;

    return retval;
}

//
// RFCOMM socket handling
//

int rfcommConnect(const char *addr, int channel)
{
    if (portfd >= 0) {
        rfcommClose();
    }
#ifdef USE_BLUEZ

    struct sockaddr_rc bt_addr = { 0 };

    bt_addr.rc_family  = AF_BLUETOOTH;
    bt_addr.rc_channel = (uint8_t) channel;
    str2ba(addr, &(bt_addr.rc_bdaddr));

    if ((portfd = socket(AF_BLUETOOTH, SOCK_STREAM|SOCK_CLOEXEC, BTPROTO_RFCOMM)) < 0) {
        logger(L_ERR, "opening socket");
        printf("ERROR: opening socket\n");
        portfd = -1;
        return -1;
    }

    connect(portfd, (struct sockaddr *) &bt_addr, sizeof(bt_addr));

#endif

    return portfd;
}

void rfcommClose(void)
{
    if (portfd >= 0) {
        shutdown(portfd, SHUT_RDWR);
        portfd = -1;
    }
}


int bt_read(char* buf, int len)
{
    return read(portfd, buf, len);
}

static int bt_read_tmo(char* buf, int len, int timeout)
{
    int ok;
    fd_set rdfds;
    struct timeval tv;

    FD_ZERO(&rdfds);
    FD_SET(portfd, &rdfds);

    /* block until something to read or timeout occurs.  select() is damn cool */
    if (timeout < 0) {
        ok = select(portfd + 1, &rdfds, NULL, NULL, NULL);
    } else {
        tv.tv_sec = timeout / 1000000;
        tv.tv_usec = (timeout % 1000000);
        ok = select(portfd + 1, &rdfds, NULL, NULL, &tv);
    }

    if (ok > 0) {
        ok = read(portfd, buf, len);
        if (ok == 0) {
            logger(L_DBG, "EOF during read()");
            return EOF;
        }

        /////////////////////////////////////
        //if(ok > 0){
        //	char buf2[1024];
        //	strncpy(buf2,buf,ok);
        //	sprintf(tmp, "READ: %s", buf2);
        //	logger(L_DBG, tmp);
        //}
        //////////////////////////////////////
        return ok;

    } else {

        return 0;

        //if (ok < 0) {
        //    return -1;
        //} else {
        //    errno = ETIMEDOUT;
        //    return 0;
        //}
    }
}

int  bt_readchar(char* c, int timeout)
{
    char buf[1];
    *c = 0;

    int rv =  bt_read_tmo(buf, 1, timeout);
    /*if (rv != 0) {
    	sprintf(tmp, "bt_read_tmo returns: %d", rv);
    	logger(L_DBG, tmp);
    }*/

    if (rv == 1) {
        *c =  buf[0];
        return 1;
    }
    if (rv == 0) {
        return EOF - 1;
    }

    //  else - EOF
    closeSerialPort(0);
    return EOF;
}

// max should be >= 100
int bt_put_command(const char* command,
                   char* answer,
                   int   max,
                   int   timeout,
                   char* expect)
{
    int count=0;
    int readcount;
    char tmp2[100];
    int timeoutcounter=0;
    int found=0;

    if (portfd < 0) {
        return 0;
    }

    logger(L_DBG,command);
    if (expect != NULL) {
        logger(L_DBG,expect);
    }

    // send command
    if (command && command[0]) {
        #ifdef __cplusplus
        ssize_t dummy = 
        #endif
            write(portfd,command,strlen(command));
	
        tcdrain(portfd);
    }

    if (max == 0) {
        return 0;
    }

    answer[0]=0;
    do {
        // try to read some bytes.
        usleep(100000);
        //write(1,".",1);
        timeoutcounter++;

        // read data
        readcount=read(portfd,tmp2,sizeof(tmp2)-1);
        if (readcount<0) {
            readcount=0;
        }
        tmp2[readcount]=0;

        // add read bytes to the output buffer
        if (readcount) {
            strcat(answer,tmp2);
            count+=readcount;

            // if we have more time to read, check if we got already the expected string
            if ((timeoutcounter<timeout) && (found==0)) {

                // check if it's the expected answer
                if ((strstr(answer,"OK\r") || strstr(answer,"ERR")) && expect == NULL) {
                    found=1;
                }

                if (expect && expect[0]) {
                    if (strstr(answer,expect)) {
                        sprintf(tmp, "Got expected %s (iteration %d)", answer, timeoutcounter);
                        logger(L_DBG, tmp);
                        found=1;
                    }
                }

                // if found then set timoutcounter to read only 0.1s after that and not more
                if (found) {
                    //timeoutcounter=timeout-1;
                    break;
                }
            }
        }
    }
    // repeat until timout
    while (timeoutcounter<timeout && count < max);

    if (getLog()) {
        char *a2 = answer;

        while (a2[0] == '\r' || a2[0] == '\n')  {
            a2++;
        }
        logger(L_DBG,a2);
    }

    return count;
}

//
// Support for stdin non-blocking IO (actually only Input)
//
// need a non-blocking IO because of another connection to front-end
// at the same time
//
#define NB_ENABLE  1
#define NB_DISABLE 0

static char stdinbuf[1024];
static int  cnt = 0;

static int kbhit()
{
    struct timeval tv;
    fd_set fds;
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    FD_ZERO(&fds);
    FD_SET(STDIN_FILENO, &fds); //STDIN_FILENO is 0
    select(STDIN_FILENO+1, &fds, NULL, NULL, &tv);
    return FD_ISSET(STDIN_FILENO, &fds);
}

static void nonblock(int state)
{
    struct termios ttystate;

    //get the terminal state
    tcgetattr(STDIN_FILENO, &ttystate);

    if (state==NB_ENABLE) {
        //turn off canonical mode
        ttystate.c_lflag &= ~ICANON;
        //minimum of number input read.
        ttystate.c_cc[VMIN] = 1;
    } else if (state==NB_DISABLE) {
        //turn on canonical mode
        ttystate.c_lflag |= ICANON;
    }
    //set the terminal attributes.
    tcsetattr(STDIN_FILENO, TCSANOW, &ttystate);

}

void openStdin()
{
    //printf("openStdin\n");
    nonblock(NB_ENABLE);
}

void closeStdin()
{
    //printf("closeStdin\n");
    nonblock(NB_DISABLE);
}

int readStdin(char* buf, int max)
{
    int ch = '1';

    //nonblock(NB_ENABLE);

    while (kbhit() /*&& cnt < max*/) {
        //printf("readStdin kbhit() OK\n");
        ch = getchar();
        //printf("readStdin got %d %c\n",cnt,ch);
        if (ch == '\n' || ch == EOF) {
            break;
        }

        stdinbuf[cnt] = ch;
        cnt++;
    }

    if (cnt == max || ch == '\n') {
        stdinbuf[cnt] = '\0';
        strcpy(buf,stdinbuf);
        int i = cnt;
        cnt = 0;
        return i;
    }

    buf[0] = '\0';
    return 0;
}

//
// Support for local server mode sockets
//
int unix_open_port(char* port)
{
    sprintf(tmp, "unix_open_port >%s<", port);
    logger(L_INF, tmp);

    struct sockaddr_un serveraddr;

    portfd = socket(AF_UNIX, SOCK_STREAM|SOCK_CLOEXEC, 0);
    if (portfd < 0) {
        sprintf(tmp, "can not open UNIX %s", port);
        logger(L_ERR, tmp);
        printf("ERROR: can not open UNIX %s\n", port);
        portfd = -1;
        return -1;
    }
    memset(&serveraddr, 0, sizeof(serveraddr));
    serveraddr.sun_family = AF_UNIX;
    strncpy(serveraddr.sun_path, port, sizeof serveraddr.sun_path - 1);

    int ret = connect(portfd, (struct sockaddr *)&serveraddr, SUN_LEN(&serveraddr));
    if (ret < 0) {
        sprintf(tmp, "can not open UNIX socket %s", port);
        logger(L_ERR, tmp);
        printf("ERROR: can not open UNIX socket %s\n", port);
        portfd = -1;
        return -1;
    }

    return 0;
}

int unix_close_port()
{
    int retval = 0;
    //printf("INFO: Close port\n");

    if (portfd < 0) { /* already closed */
        //printf("INFO: Already closed ?\n");
        return -1;
    }

    retval = close(portfd);
    portfd = -1;

    return retval;
}
/*
int unix_read_port(char* buf, int l)
{
	int ret = 0;
	int bytesReceived = 0;
	printf("SOCKET: read fd %d\n", portfd);

	while (bytesReceived < l-1)
	{
		int rc = recv(portfd, & buf[bytesReceived], l - bytesReceived, 0);
		printf("unix_read_port %d\n",rc);
		if (rc < 0) {
	   		//logger(L_ERR,"recv() failed");
			ret = EOF-1;
	   		break;
		} else if (rc == 0) {
	   		//logger(L_INF,"Read 0 bytes");
			ret = EOF;
	   		break;
		}

		bytesReceived += rc;
		ret = bytesReceived;
	}
	buf[bytesReceived] = '\0';

	return ret;
}
*/


//
// Support SDP
//

#ifdef USE_BLUEZ

#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

sdp_session_t *session = NULL;
sdp_record_t  *record  = NULL;

void sdpRegister(int port)
{
    uint8_t svc_uuid_int[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0xab, 0xcd };
    uint8_t rfcomm_channel = port;
    const char *svc_dsc = "Bluetooth remote control";
    const char *service_prov = "anyRemote";

    uuid_t root_uuid, l2cap_uuid, rfcomm_uuid, svc_uuid,
           svc_class_uuid;
    sdp_list_t *l2cap_list = 0,
                *rfcomm_list = 0,
                 *root_list = 0,
                  *proto_list = 0,
                   *access_proto_list = 0,
                    *svc_class_list = 0,
                     *profile_list = 0;
    sdp_data_t *channel = 0;
    sdp_profile_desc_t profile;
    record = sdp_record_alloc();

    // set the general service ID
    sdp_uuid128_create( &svc_uuid, &svc_uuid_int );
    sdp_set_service_id( record, svc_uuid );

    // set the service class
    sdp_uuid16_create(&svc_class_uuid, SERIAL_PORT_SVCLASS_ID);
    svc_class_list = sdp_list_append(0, &svc_class_uuid);
    sdp_set_service_classes(record, svc_class_list);

    // set the Bluetooth profile information
    sdp_uuid16_create(&profile.uuid, SERIAL_PORT_PROFILE_ID);
    profile.version = 0x0100;
    profile_list = sdp_list_append(0, &profile);
    sdp_set_profile_descs(record, profile_list);

    // make the service record publicly browsable
    sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
    root_list = sdp_list_append(0, &root_uuid);
    sdp_set_browse_groups( record, root_list );

    // set l2cap information
    sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
    l2cap_list = sdp_list_append( 0, &l2cap_uuid );
    proto_list = sdp_list_append( 0, l2cap_list );

    // register the RFCOMM channel for RFCOMM sockets
    sdp_uuid16_create(&rfcomm_uuid, RFCOMM_UUID);
    channel = sdp_data_alloc(SDP_UINT8, &rfcomm_channel);
    rfcomm_list = sdp_list_append( 0, &rfcomm_uuid );
    sdp_list_append( rfcomm_list, channel );
    sdp_list_append( proto_list, rfcomm_list );

    access_proto_list = sdp_list_append( 0, proto_list );
    sdp_set_access_protos( record, access_proto_list );

    // set the name, provider, and description
    char *sn = getServiceName();
    sdp_set_info_attr(record, sn, service_prov, svc_dsc);
    free(sn);

    // connect to the local SDP server, register the service record,
    // and disconnect
    session = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, SDP_RETRY_IF_BUSY);
    if ( (!(session && record)) || sdp_record_register(session, record, 0) == -1) {
        logger(L_ERR, "can not register SDP service");
    }

    // cleanup
    sdp_data_free( channel );
    sdp_list_free( l2cap_list, 0 );
    sdp_list_free( proto_list, 0 );
    sdp_list_free( rfcomm_list, 0 );
    sdp_list_free( root_list, 0 );
    sdp_list_free( access_proto_list, 0 );
    sdp_list_free( svc_class_list, 0 );
    sdp_list_free( profile_list, 0 );
}
#endif

#ifdef USE_BT_FBSD
void			*session = NULL;
uint32_t		 record;


void sdpRegister(int port)
{
    int channel,service;
    bdaddr_t		 bt_addr_any;
    sdp_lan_profile_t	 lan;

    channel = port;
    service = SDP_SERVICE_CLASS_SERIAL_PORT;

    session = sdp_open_local(NULL);
    if (session == NULL) {
        errx(1, "Unable to create local SDP session");
    }
    if (sdp_error(session) != 0)
        errx(1, "Unable to open local SDP session. %s (%d)",
             strerror(sdp_error(session)), sdp_error(session));
    memset(&lan, 0, sizeof(lan));
    lan.server_channel = channel;

    memcpy(&bt_addr_any, NG_HCI_BDADDR_ANY, sizeof(bt_addr_any));
    if (sdp_register_service(session, service, &bt_addr_any,
                             (void *)&lan, sizeof(lan), &record) != 0) {
        errx(1, "Unable to register LAN service with "
             "local SDP daemon. %s (%d)",
             strerror(sdp_error(session)), sdp_error(session));
    }
}
#endif

static void sdpDeregister()
{
    logger(L_DBG, "Deregister SDP service");
#ifdef USE_BLUEZ
    if (session != NULL) {
        sdp_record_unregister(session, record);
        sdp_close(session);
        session = NULL;
        //sdp_record_free(record);
    }
#endif
#ifdef USE_BT_FBSD
    if (session != NULL) {
        sdp_unregister_service(session, record);
        sdp_close(session);
        sdp_close(session);
        session = NULL;
    }
#endif
}

//
// Support server mode sockets
//

int openSocketPort(int type, int port, char *path)
{
    struct sockaddr_in tcp_addr;
    struct sockaddr_un un_addr;
#ifdef USE_BLUEZ
    struct sockaddr_rc bt_addr;
#endif
#ifdef USE_BT_FBSD
    struct sockaddr_rfcomm bt_addr;
#endif

    struct sockaddr*   socketaddr = NULL;

    int addFamily = 0;
    int proto     = 0;
    int sz;

    if (sportfd >= 0) {
        logger(L_ERR, "socket was already opened");
        //close(sportfd);
        //sportfd = 0;
        return 1;
    }

    if (type == SERVER_TCP) {
        addFamily = AF_INET;
        proto	  = IPPROTO_TCP;
    } else if (type == SERVER_BT) {
#ifdef USE_BLUEZ
        addFamily = PF_BLUETOOTH;
        proto	  = BTPROTO_RFCOMM;
#endif
#ifdef USE_BT_FBSD
        addFamily = PF_BLUETOOTH;
        proto	  = BLUETOOTH_PROTO_RFCOMM;
#endif
    } else if (type == SERVER_UX) {
        addFamily = AF_UNIX;
        proto	  = 0;
    } else {
        logger(L_ERR, "incorrect input");
        return -1;
    }

    if ((sportfd = socket(addFamily, SOCK_STREAM|SOCK_CLOEXEC, proto)) < 0) {
        logger(L_ERR, "opening socket");
        printf("ERROR: opening socket\n");
        sportfd = -1;
        return -1;
    }

    int optval = 1;
    setsockopt(sportfd, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));

    /*
    // Set non-blocking mode
    if (-1 == (oldflags = fcntl(sportfd, F_GETFL, 0))) {
        oldflags = 0;
    }
    fcntl(sportfd, F_SETFL, oldflags | O_NONBLOCK);
    */


    if (type == SERVER_TCP) {

        memset((void *) &tcp_addr, 0, sizeof(tcp_addr));
        tcp_addr.sin_family = AF_INET;
        tcp_addr.sin_addr.s_addr = INADDR_ANY;
        tcp_addr.sin_port = htons(port);

        socketaddr=(struct sockaddr *)&tcp_addr;
        sz = sizeof(tcp_addr);

    } else if (type == SERVER_BT) {	// SERVER_BT
#ifdef USE_BLUEZ
        memset((void *) &bt_addr, 0, sizeof(bt_addr));
        sz = sizeof(bt_addr);

        // bind socket to the specified port of the first available local bluetooth adapter
        bt_addr.rc_family = AF_BLUETOOTH;
        bt_addr.rc_bdaddr = *BDADDR_ANY;
        bt_addr.rc_channel = (uint8_t) port;

        //char tmpstring[512];
        //sprintf(tmpstring, "sdptool add --channel=%i SP", port);
        //sprintf(tmpstring, "sdptool add --channel=%i SP;sdptool setattr `sdptool search --bdaddr local SP|grep \"Service RecHandle\"|tail -1|cut -f 3 -d \" \"` 0x100 anyRemote", port);
        //sprintf(tmpstring, "bash -c \'A=`sdptool search --bdaddr local SP|grep \"Service Name\"|grep anyRemote|wc -l`; if [ \"x$A\" == \"x0\" ]; then sdptool add --channel=%i SP;sdptool setattr `sdptool search --bdaddr local SP|grep \"Service RecHandle\"|tail -1|cut -f 3 -d \" \"` 0x100 anyRemote; fi\'", port);
        //system(tmpstring);

        sdpRegister(port);
        sprintf(tmp, "registered SP for channel %i", port);
        logger(L_INF, tmp);
        socketaddr=(struct sockaddr *)&bt_addr;
#endif

#ifdef USE_BT_FBSD
        memset(&bt_addr, 0, sizeof(bt_addr));

        bt_addr.rfcomm_len = sizeof(bt_addr);
        bt_addr.rfcomm_family = AF_BLUETOOTH;
        bt_addr.rfcomm_channel =  (uint8_t) port;;
        sdpRegister(port);
        sprintf(tmp, "registered SP for channel %i", port);
        logger(L_INF, tmp);

        socketaddr=(struct sockaddr *)&bt_addr;
        sz = sizeof(bt_addr);
#endif
    } else if (type == SERVER_UX && path != NULL) {

        memset(&un_addr, 0, sizeof(un_addr));
        un_addr.sun_family = AF_UNIX;
        strncpy(un_addr.sun_path, path, sizeof un_addr.sun_path - 1);
        printf("ERROR: SOCKET %s\n", path);
        socketaddr=(struct sockaddr *)&un_addr;
        sz = sizeof(un_addr);

    } else {
        logger(L_ERR, "incorrect input 2");
        return -1;
    }

    if (bind(sportfd, (struct sockaddr *) socketaddr, sz) < 0) {
        logger(L_ERR, "on binding");
        printf("ERROR: on binding %d->%s\n", errno, strerror(errno));
        return -1;
    }

    if (type == SERVER_UX) {
        uSocket = strdup(path);
    }
    return 1;
}

int closeSocketPort(int type, int final)
{
    logger(L_INF, "closeSocketPort");

    if (portfd >= 0) {
        logger(L_INF, "closeSocketPort close socket");
        close(portfd);
        portfd = -1;
    }
    if (sportfd >= 0) {
        logger(L_INF, "closeSocketPort close server socket");
        close(sportfd);
        sportfd = -1;
    }
    if (type == SERVER_BT && final) {
        sdpDeregister();
        freeBtAddress();
    } else if (type == SERVER_UX) {
        unlink(uSocket);
        free(uSocket);
    }

    return 1;
}

int listenAndAcceptSocketConn(int type)
{
    int cnt, sz;

    struct sockaddr* socketaddr = NULL;
    struct sockaddr_in ip_addr;

#ifdef USE_BLUEZ
    struct sockaddr_rc bt_addr;
    bdaddr_t ba;
#endif

#ifdef USE_BT_FBSD
    struct sockaddr_rfcomm bt_addr;
#endif

    logger(L_INF, "listenAndAcceptSocketConn");
    cnt = 0;

    if (type == SERVER_BT) {
#if defined(USE_BLUEZ) || defined(USE_BT_FBSD)
        socketaddr=(struct sockaddr *)&bt_addr;
        sz = sizeof(bt_addr);
#endif
    } else if (type == SERVER_TCP) {
        socketaddr=(struct sockaddr *)&ip_addr;
        sz = sizeof(ip_addr);
    }

    while (stillRun) {

        logger(L_INF, "listenAndAcceptSocketConn: accept");

        listen(sportfd,5);

        portfd = accept(sportfd, (struct sockaddr *) socketaddr, (socklen_t *)&sz);

        if (portfd < 0 && errno == EAGAIN) {

            if (cnt >= 60) {    // Print to log every minute
                logger(L_INF, "listenAndAcceptSocketConn: waiting for connection");
                //printf(".");
                cnt = 0;
            }
            fflush(stdout);

            sleep(1);
            cnt++;

            continue;
        }

        if (portfd < 0) {
            logger(L_ERR, "on accept");
            printf("ERROR: on accept %d\n", errno);
            return -1;
            /*} else {
            	// Set non-blocking mode
            	if (-1 == (flags = fcntl(portfd, F_GETFL, 0))) {
            	    flags = 0;
            	}
            	fcntl(portfd, F_SETFL, flags | O_NONBLOCK);
            */
        }
        logger(L_INF, "listenAndAcceptSocketConn: accepted");

#ifdef USE_BLUEZ
        if (type == SERVER_BT) {
            baswap(&ba, &bt_addr.rc_bdaddr);
            if (btAddress) {
                free(btAddress);
            }
            btAddress = batostr(&ba);

            sprintf(tmp, "listenAndAcceptSocketConn: remote BT address is %s", btAddress);
            logger(L_INF, tmp);
        }
#endif
#ifdef USE_BT_FBSD
        if (type == SERVER_BT) {
            if (btAddress) {
                free(btAddress);
            }
            btAddress = strdup(bt_ntoa(&bt_addr.rfcomm_bdaddr, NULL));

            sprintf(tmp, "listenAndAcceptSocketConn: remote BT address is %s", btAddress);
            logger(L_INF, tmp);
        }
#endif

        break;
    }
    return 1;
}

int writeSocketConn(const char* command, int count)
{
    //logger(L_DBG, "writeSocketConn");
    if (!command) {
        return EXIT_OK;
    }

    // send command
    if (portfd >= 0 && count > 0) {

        memset(tmp, 0, MAXMAXLEN);
        strcat(tmp, "writeSocketConn ");

        ivInCommand = 1;
        int logSz = (count > 256 ? 255 : count);

        // it is possible to get binary data here
        memcpy(tmp, command, logSz); // Do not dump long commands
        tmp[logSz] = '\0';
        logger(L_DBG, tmp);

        sprintf(tmp, "writeSocketConn %d bytes", count);
        logger(L_INF, tmp);

        int n = write(portfd,command,count);
        if (n < 0) {
            logger(L_ERR, "error writing to socket");
            return EXIT_NOK;
        }
        //tcdrain(portfd);
        return EXIT_OK;
    } else {
        logger(L_ERR, "error writing to socket: already closed");
    }
    return EXIT_NOK;
}

static int writeByteSocketConn(int byte)
{
    unsigned char byte2write[2];
    byte2write[0] = (unsigned char) byte;
    byte2write[1] = '\0';

    if (write(portfd, byte2write, 1) < 0) {
        logger(L_ERR, "error writing byte to socket");
        return EXIT_NOK;
    }
    return EXIT_OK;
}

static void writeIViewerHeartbeat()
{
    // reply message is h=1\03
    writeByteSocketConn(104); // h
    writeByteSocketConn(61 ); // =
    writeByteSocketConn(49 ); // 1
    writeByteSocketConn(3  ); // \03
}

int writeBytesSocketConn(char* command)
{
    //logger(L_DBG, "writeBytesSocketConn");

    // send command
    if (portfd >= 0 && command && command[0]) {

        char byteStr[MAXCKPDLEN];
        memset(byteStr,0,MAXCKPDLEN);

        strncpy(byteStr,command,MAXCKPDLEN-1);

        sprintf(tmp, "writeBytesSocketConn >%s<", byteStr);
        logger(L_DBG, tmp);

        char* bStr = strtok(byteStr,",");
        while (bStr != NULL) {

            //sprintf(tmp, "Next byte is >%s<", bStr);
            //logger(L_DBG, tmp);

            char bStripped[4];

            while (*bStr == ' ') {
                bStr++;
            }
            int i = 0;
            while (*bStr != ' ' && i < 3) {  // 0 < ... < 256
                bStripped[i] = *bStr;
                bStr++;
                i++;
            }
            bStripped[i] = '\0';

            sprintf(tmp, "Next byte is >%s<", bStripped);
            logger(L_DBG, tmp);

            if (writeByteSocketConn(atoi(bStripped)) != EXIT_OK) {
                return EXIT_NOK;
            }

            // send iViewer heartbeat message
            if (strcmp(bStripped,"3") == 0) {
                ivInCommand = 0;

                if (ivHeartbeat) {
                    logger(L_DBG, "Send buffered heartbeat message");
                    ivHeartbeat = 0;
                    writeIViewerHeartbeat();
                }
            } else {
                ivInCommand = 1;
            }

            bStr = strtok(NULL,",");
        }
    }
    //logger(L_DBG, "writeBytesSocketConn EXIT");
    return EXIT_OK;
}

char * getBtAddress()
{
    return btAddress;
}

void freeBtAddress()
{
    if (btAddress) {
        free(btAddress);
    }
    btAddress = NULL;
}

void sendIViewerHeartbeat()
{
    if (ivInCommand) {
        ivHeartbeat = 1;
    } else {
        writeIViewerHeartbeat();
    }
}
