/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <cassert>
#include <errno.h>
#include <fcntl.h>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <kglobal.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <string.h>
#include <qobject.h>
#include <qtextcodec.h>
#include <sstream>
#include <sys/types.h>
#include <sys/stat.h>
#include "unitMap.hpp"
#include "utils.hpp"

using namespace std;

string replaceAll( const string &text,
                   const string &toReplace,
                   const string &replaceBy )
{
  string retVal;

  assert( toReplace.length() > 0 );

  int
    toReplaceLength = toReplace.length(),
    previous =0;

  while ( true ) {

    int position = text.find( toReplace, previous );

    if ( position < 0 ) {
      retVal += text.substr( previous, text.length() - previous );
      break;
    };
    
    retVal += text.substr( previous, position - previous );
    retVal += replaceBy;

    previous = position + toReplaceLength;
  };

  return retVal;
}

string xmlText( const string &text )
{
  string retVal =
    string( "<![CDATA[" ) +
    replaceAll( text, "]]>", "]]]]><![CDATA[>" ) + "]]>";
  return retVal;
}

string versionText( int version )
{
  int
    mainVersion = version / 1000,
    subVersion = version % 1000;
  ostringstream text;
  text << mainVersion << '.' << subVersion;
  return text.str();
}

string dictionaryFile(void) throw (Error)
{
  char *fileName = strdup( "/tmp/anymeal-dictionary-xsl.XXXXXX" );
  int fd = mkstemp( fileName );
  string retVal = fileName;
  free( fileName );
  ERRORMACRO( fd != -1, Error, ,
              "Error creating temporary file." );
  ofstream f( retVal.c_str() );
  close( fd );
  f << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    << "<dictionary>\n"
    << "<entry text=\"Amount\">" << i18n( "Amount" ) << "</entry>"
    << "<entry text=\"Cookbook\">" << i18n( "Cookbook" ) << "</entry>"
    << "<entry text=\"Categories\">" << i18n( "Categories" )
    << "</entry>"
    << "<entry text=\"generated by AnyMeal\">"
    << i18n( "generated by AnyMeal" ) << "</entry>"
    << "<entry text=\"Ingredients\">" << i18n( "Ingredients" )
    << "</entry>"
    << "<entry text=\"Instructions\">" << i18n( "Instructions" )
    << "</entry>"
    << "<entry text=\"Name\">" << i18n( "Name" ) << "</entry>"
    << "<entry text=\"Preparation\">" << i18n( "Preparation" )
    << "</entry>"
    << "<entry text=\"Unit\">" << i18n( "Unit" ) << "</entry>"
    << "<entry text=\"Yield\">" << i18n( "Yield" ) << "</entry>"
    << "</dictionary>";
  return retVal;
}

string mealmasterMapFile(void) throw (Error)
{
  map< string, string > unitMap( createUnitMap() );
  char *fileName = strdup( "/tmp/anymeal-mealmastermap-xsl.XXXXXX" );
  int fd = mkstemp( fileName );
  string retVal = fileName;
  free( fileName );
  ERRORMACRO( fd != -1, Error, ,
              "Error creating temporary file." );
  ofstream f( retVal.c_str() );
  close( fd );
  f << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl
    << "<mealmastermap>" << endl;
  for ( map< string, string >::const_iterator i = unitMap.begin();
        i != unitMap.end(); i++ )
    f << "<item text=\"" << i->first << "\">" << i->second
      << "</item>" << endl;
  f << "</mealmastermap>" << endl;
  return retVal;
}

string findAnyMealFile( const char *type, const char *fileName )
{
  string retVal;
  KStandardDirs *dirs = KGlobal::dirs();
  if ( dirs != NULL ) {
    QString result = dirs->findResource( type, fileName );
    if ( result != QString::null )
      retVal = (const char *)result;
    else {
#ifndef NDEBUG
      cerr << "Warning: KStandardDirs::findResource( \"" << type << "\", \""
           << fileName << "\" ) returned QString::null." << endl;
#endif
      retVal = fileName;
    };
  } else {
#ifndef NDEBUG
    cerr << "Warning: KGlobal::dirs() is null." << endl;
#endif
    retVal = fileName;
  };
  return retVal;
}

string anyMealLanguage(void)
{
  QString language, country, charset;
  KLocale::splitLocale( QTextCodec::locale(),
                        language, country, charset );

  return language;
}

int gcd( int a, int b )
{
  assert( a >= 0 );
  assert( b >= 0 );
  int retVal;
  if ( a < b )
    retVal = gcd( b, a );
  else {
    assert( a >= b );
    if ( b == 0 )
      retVal = a;
    else
      retVal = gcd( b, a % b );
  };
  return retVal;
}

int testFile( const string &fileName )
{
  int retVal = 0;
  int desc = open( fileName.c_str(), O_RDWR );
  if ( desc == -1 )
    retVal = errno;
  else
    close( desc );
  return retVal;
}

void createDirRecursively( const string &dirName ) throw (Error)
{
  int result = mkdir( dirName.c_str(), S_IRWXU );
  if ( result != 0 ) {
    switch ( errno ) {
    case ENOENT: {
      unsigned int pos = dirName.find_last_of( "/" );
      ERRORMACRO( pos != string::npos, Error, ,
                  "Failed to create directory '" << dirName << "': "
                  << strerror( errno ) );
      assert( pos < dirName.size() );
      createDirRecursively( string( dirName, 0, pos ) );
      createDirRecursively( dirName );
      break;}
    case EEXIST:
      ERRORMACRO( testFile( dirName ) != EISDIR, Error, ,
                  "Failed to create directory '" << dirName << "': "
                  << strerror( errno ) );
      break;
    default:
#ifndef NDEBUG
      cerr << "mkdir resulted in error number " << errno << '.' << endl;
#endif
      ERRORMACRO( false, Error, ,
                  "Failed to create directory '" << dirName << "': "
                  << strerror( errno ) );
    };
  };
}
