/* Copyright (C) 2002, 2003, 2004, 2005 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include "include.hpp"
#ifdef SIMPLEXPATH
#include <XPath/XPathEvaluator.hpp>
#include <DOMSupport/XalanDocumentPrefixResolver.hpp>
#else
#include <XPath/XPath.hpp>
#include <XPath/ElementPrefixResolverProxy.hpp>
#include <XPath/XPathProcessorImpl.hpp>
#endif
#include "xmlDocument.hpp"
#include "xmlReference.hpp"
#include "xmlNodeReferenceList.hpp"

using namespace boost;

template<>
XMLReference< XalanNode > XMLReference< XalanElement >::selectNode
  ( const std::string &xPath ) throw (Error,XMLException)
{
  XMLNodeReferenceList nodeList( selectNodes( xPath ) );
  ERRORMACRO( nodeList.getLength() >= 1, Error, ,
              "Could not find XML node with x-path \"" << xPath  << "\"." );
  ERRORMACRO( nodeList.getLength() <= 1, Error, ,
              "There are multiple nodes with x-path \"" << xPath << "\"." );
  assert( nodeList.getLength() == 1 );
  return nodeList.item( 0 );
}

template<>
XMLNodeReferenceList XMLReference< XalanElement >::selectNodes
  ( const std::string &xPath ) throw (Error,XMLException)
{
  ERRORMACRO( document != NULL, Error, ,
              "Can not select nodes from empty XML document." );
#ifdef SIMPLEXPATH
  XalanDocumentPrefixResolver thePrefixResolver( document );
  XPathEvaluator theEvaluator;
  // Using XalanDOMString-constructor, which is present in Xalan-C version
  // 1.9 and in earlier versions as well!
  NodeRefList obj =
    theEvaluator.selectNodeList( *document->getDOMSupport(), node,
                                 XalanDOMString( xPath.c_str() ).c_str(),
                                 thePrefixResolver );
#else
  XPathProcessorImpl theXPathProcessor;
  XPath * const contextXPath = document->getXPathFactory()->create();
  const ElementPrefixResolverProxy prefixResolver
	  ( node, *document->getEnvSupport(), *document->getDOMSupport() );
  theXPathProcessor.initXPath( *contextXPath,
  // With which version was this introduced exactly?
#if _XALAN_VERSION >= 10500
                               *document->getXPathConstructionContext(),
#endif
                               XalanDOMString( xPath.c_str() ),
			       prefixResolver );

  shared_ptr< XObjectPtr > obj( new XObjectPtr );
  *obj =
    contextXPath->execute( node, prefixResolver,
                           *document->getExecutionContext() );
#endif
  return XMLNodeReferenceList( obj, document );
}

template<>
bool XMLReference< XalanElement >::existNode( const std::string &xPath )
  throw (XMLException)
{
  return selectNodes( xPath ).getLength() >= 1;
}

