/*
 *
 *  (c) COPYRIGHT INRIA 1999-2001.
 *  Please first read the full copyright statement in file COPYRIGHT.
 *
 */

/* Unicode Routines
 * Authors: I. Vatton (W3C/INRIA)
 *          R. Guetari (W3C/INRIA): previous version
 */

#include <stdlib.h>
#include <ctype.h>
#include "thot_sys.h"
#include "uconvert.h"

extern unsigned long offset[6];

unsigned short ISO_8859_2_Map [] = {
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x0104, 0x02D8, 0x0141, 0x00A4, 0x013D, 0x015A, 0x00A7,
    0x00A8, 0x0160, 0x015E, 0x0164, 0x0179, 0x00AD, 0x017D, 0x017B,
    0x00B0, 0x0105, 0x02DB, 0x0142, 0x00B4, 0x013E, 0x015B, 0x02C7,
    0x00B8, 0x0161, 0x015F, 0x0165, 0x017A, 0x02DD, 0x017E, 0x017C,
    0x0154, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x0139, 0x0106, 0x00C7,
    0x010C, 0x00C9, 0x0118, 0x00CB, 0x011A, 0x00CD, 0x00CE, 0x010E,
    0x0110, 0x0143, 0x0147, 0x00D3, 0x00D4, 0x0150, 0x00D6, 0x00D7,
    0x0158, 0x016E, 0x00DA, 0x0170, 0x00DC, 0x00DD, 0x0162, 0x00DF,
    0x0155, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x013A, 0x0107, 0x00E7,
    0x010D, 0x00E9, 0x0119, 0x00EB, 0x011B, 0x00ED, 0x00EE, 0x010F,
    0x0111, 0x0144, 0x0148, 0x00F3, 0x00F4, 0x0151, 0x00F6, 0x00F7,
    0x0159, 0x016F, 0x00FA, 0x0171, 0x00FC, 0x00FD, 0x0163, 0x02D9
};
#define ISO_8859_2_length sizeof(ISO_8859_2_Map) / sizeof(unsigned short);

unsigned short ISO_8859_3_Map [] = {
    0x007F,
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x0126, 0x02D8, 0x00A3, 0x00A4, 0x00A5, 0x0124, 0x00A7,
    0x00A8, 0x0130, 0x015E, 0x011E, 0x0134, 0x00AD, 0xFFFE, 0x017B,
    0x00B0, 0x0127, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x0125, 0x00B7,
    0x00B8, 0x0131, 0x015F, 0x011F, 0x0135, 0x00BD, 0xFFFE, 0x017C,
    0x00C0, 0x00C1, 0x00C2, 0xFFFE, 0x00C4, 0x010A, 0x0108, 0x00C7,
    0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
    0xFFFE, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x0120, 0x00D6, 0x00D7,
    0x011C, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x016C, 0x015C, 0x00DF,
    0x00E0, 0x00E1, 0x00E2, 0xFFFE, 0x00E4, 0x010B, 0x0109, 0x00E7, 
    0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
    0xFFFE, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x0121, 0x00F6, 0x00F7,
    0x011D, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x016D, 0x015D, 0x02D9
};
#define ISO_8859_3_length sizeof(ISO_8859_3_Map) / sizeof(unsigned short);

unsigned short ISO_8859_4_Map [] = {
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x0104, 0x0138, 0x0156, 0x00A4, 0x0128, 0x013B, 0x00A7,
    0x00A8, 0x0160, 0x0112, 0x0122, 0x0166, 0x00AD, 0x017D, 0x00AF,
    0x00B0, 0x0105, 0x02DB, 0x0157, 0x00B4, 0x0129, 0x013C, 0x02C7,
    0x00B8, 0x0161, 0x0113, 0x0123, 0x0167, 0x014A, 0x017E, 0x014B,
    0x0100, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x012E,
    0x010C, 0x00C9, 0x0118, 0x00CB, 0x0116, 0x00CD, 0x00CE, 0x012A,
    0x0110, 0x0145, 0x014C, 0x0136, 0x00D4, 0x00D5, 0x00D6, 0x00D7, 
    0x00D8, 0x0172, 0x00DA, 0x00DB, 0x00DC, 0x0168, 0x016A, 0x00DF,
    0x0101, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x012F,
    0x010D, 0x00E9, 0x0119, 0x00EB, 0x0117, 0x00ED, 0x00EE, 0x012B,
    0x0111, 0x0146, 0x014D, 0x0137, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
    0x00F8, 0x0173, 0x00FA, 0x00FB, 0x00FC, 0x0169, 0x016B, 0x02D9
};
#define ISO_8859_4_length sizeof(ISO_8859_4_Map) / sizeof(unsigned short);

unsigned short ISO_8859_5_Map [] = {
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x0401, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407,
    0x0408, 0x0409, 0x040A, 0x040B, 0x040C, 0x00AD, 0x040E, 0x040F,
    0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
    0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F,
    0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
    0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F,
    0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
    0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F,
    0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
    0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F,
    0x2116, 0x0451, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457,
    0x0458, 0x0459, 0x045A, 0x045B, 0x045C, 0x00A7, 0x045E, 0x045F
};
#define ISO_8859_5_length sizeof(ISO_8859_5_Map) / sizeof(unsigned short);

/* Arabic ISO Latin encoding */
unsigned short ISO_8859_6_Map [] = {
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x060C, 0x00AD, 0x00AE, 0x00AF,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x00B9, 0x00BA, 0x061B, 0x00BC, 0x00BD, 0x00BE, 0x061F,
    0x00C0, 0x0621, 0x0622, 0x0623, 0x0624, 0x0625, 0x0626, 0x0627,
    0x0628, 0x0629, 0x062A, 0x062B, 0x062C, 0x062D, 0x062E, 0x062F,
    0x0630, 0x0631, 0x0632, 0x0633, 0x0634, 0x0635, 0x0636, 0x0637,
    0x0638, 0x0639, 0x063A, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF,
    0x0640, 0x0641, 0x0642, 0x0643, 0x0644, 0x0645, 0x0646, 0x0647,
    0x0648, 0x0649, 0x064A, 0x064B, 0x064C, 0x064D, 0x064E, 0x064F,
    0x0650, 0x0651, 0x0652, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
    0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF
};
#define ISO_8859_6_length sizeof(ISO_8859_6_Map) / sizeof(unsigned short);

/* Greek */
unsigned short ISO_8859_7_Map [] = {
    0x007F,
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087, 
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F, 
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097, 
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F, 
    0x00A0, 0x02BD, 0x02BC, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, 
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x2015, 
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x0384, 0x0385, 0x0386, 0x00B7, 
    0x0388, 0x0389, 0x038A, 0x00BB, 0x038C, 0x00BD, 0x038E, 0x038F, 
    0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 
    0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F, 
    0x03A0, 0x03A1, 0x00D2, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, 
    0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x03AC, 0x03AD, 0x03AE, 0x03AF, 
    0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, 
    0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF, 
    0x03C0, 0x03C1, 0x03C2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 
    0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE, 0x00FF
};
#define ISO_8859_7_length sizeof(ISO_8859_7_Map) / sizeof(unsigned short);

/* Hebrew */
unsigned short ISO_8859_8_Map [] = {
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087, 
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F, 
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097, 
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F, 
    0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x00D7, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x203E,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x00B9, 0x00F7, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
    0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, 
    0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
    0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,
    0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x2017, 
    0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7,
    0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF,
    0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7,
    0x05E8, 0x05E9, 0x05EA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF
};
#define ISO_8859_8_length sizeof(ISO_8859_8_Map) / sizeof(unsigned short);

/*  */
unsigned short ISO_8859_9_Map [] = {
    0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087,
    0x0088, 0x0089, 0x008A, 0x008B, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009A, 0x009B, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
    0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7,
    0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
    0x011E, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,
    0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x0130, 0x015E, 0x00DF,
    0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
    0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
    0x011F, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
    0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x0131, 0x015F, 0x00FF
};
#define ISO_8859_9_length sizeof(ISO_8859_9_Map) / sizeof(unsigned short);

/* Windows Latin 2 (Central Europe) Code Page */
unsigned short WIN1250CP [] = {
    0xFFFE, 0xFFFE, 0x201A, 0xFFFE, 0x201E, 0x2026, 0x2020, 0x2021,
    0xFFFE, 0x2030, 0x0160, 0x2039, 0x015A, 0x0164, 0x017D, 0x0179,
    0xFFFE, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0xFFFE, 0x2122, 0x0161, 0x203A, 0x015B, 0x0165, 0x017E, 0x017A,
    0x00A0, 0x02C7, 0x02D8, 0x0141, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x015E, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x017B,
    0x00B0, 0x00B1, 0x02DB, 0x0142, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x0105, 0x015F, 0x00BB, 0x013D, 0x02DD, 0x013E, 0x017C,
    0x0154, 0x00C1, 0x00C2, 0x0102, 0x00C4, 0x0139, 0x0106, 0x00C7,
    0x010C, 0x00C9, 0x0118, 0x00CB, 0x011A, 0x00CD, 0x00CE, 0x010E,
    0x0110, 0x0143, 0x0147, 0x00D3, 0x00D4, 0x0150, 0x00D6, 0x00D7,
    0x0158, 0x016E, 0x00DA, 0x0170, 0x00DC, 0x00DD, 0x0162, 0x00DF,
    0x0155, 0x00E1, 0x00E2, 0x0103, 0x00E4, 0x013A, 0x0107, 0x00E7,
    0x010D, 0x00E9, 0x0119, 0x00EB, 0x011B, 0x00ED, 0x00EE, 0x010F,
    0x0111, 0x0144, 0x0148, 0x00F3, 0x00F4, 0x0151, 0x00F6, 0x00F7,
    0x0159, 0x016F, 0x00FA, 0x0171, 0x00FC, 0x00FD, 0x0163, 0x02D9
};
#define WIN1250CP_length sizeof(WIN1250CP) / sizeof(unsigned short);

/* Windows Cyrillic (Slavic) Code Page */
unsigned short WIN1251CP [] = {
    0x0402, 0x0403, 0x201A, 0x0453, 0x201E, 0x2026, 0x2020, 0x2021,
    0x0088, 0x2030, 0x0409, 0x2039, 0x040A, 0x040C, 0x040B, 0x040F,
    0x0452, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0x0098, 0x2122, 0x0459, 0x203A, 0x045A, 0x045C, 0x045B, 0x045F,
    0x00A0, 0x040E, 0x045E, 0x0408, 0x00A4, 0x0490, 0x00A6, 0x00A7,
    0x0401, 0x00A9, 0x0404, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x0407,
    0x00B0, 0x00B1, 0x0406, 0x0456, 0x0491, 0x00B5, 0x00B6, 0x00B7,
    0x0451, 0x2116, 0x0454, 0x00BB, 0x0458, 0x0405, 0x0455, 0x0457,
    0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417,
    0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, 0x041E, 0x041F,
    0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427,
    0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 0x042F,
    0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
    0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F,
    0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447,
    0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F	
};
#define WIN1251CP_length sizeof(WIN1251CP) / sizeof(unsigned short);

/* Windows Latin 1 Code Page */
unsigned short WIN1252CP [] = {
    0x0080, 0x0081, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
    0x02C6, 0x2030, 0x0160, 0x2039, 0x0152, 0x008D, 0x008E, 0x008F,
    0x0090, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0x02DC, 0x2122, 0x0161, 0x203A, 0x0153, 0x009D, 0x009E, 0x0178, 
    0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
    0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7,
    0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
    0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,
    0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF,
    0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
    0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
    0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
    0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF
};
#define WIN1252CP_length sizeof(WIN1252CP) / sizeof(unsigned short);

/* Windows Greek Code Page */
unsigned short WIN1253CP [] = {
    0x0080, 0x0081, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
    0x0088, 0x2030, 0x008A, 0x2039, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0x0098, 0x2122, 0x009A, 0x203A, 0x009C, 0x009D, 0x009E, 0x009F,
    0x00A0, 0x0385, 0x0386, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x2015,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x0384, 0x00B5, 0x00B6, 0x00B7,
    0x0388, 0x0389, 0x038A, 0x00BB, 0x038C, 0x00BD, 0x038E, 0x038F,
    0x0390, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 
    0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F,
    0x03A0, 0x03A1, 0x00D2, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7,
    0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x03AC, 0x03AD, 0x03AE, 0x03AF,
    0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7,
    0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
    0x03C0, 0x03C1, 0x03C2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 
    0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE, 0x00FF
};
#define WIN1253CP_length sizeof(WIN1253CP) / sizeof(unsigned short);

/* Windows Latin 5 (Turkish) Code Page */
unsigned short WIN1254CP [] = {
    0x0080, 0x0081, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, 
    0x02C6, 0x2030, 0x0160, 0x2039, 0x0152, 0x008D, 0x008E, 0x008F, 
    0x0090, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014, 
    0x02DC, 0x2122, 0x0161, 0x203A, 0x0153, 0x009D, 0x009E, 0x0178, 
    0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7, 
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
    0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4, 0x00C5, 0x00C6, 0x00C7, 
    0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC, 0x00CD, 0x00CE, 0x00CF,
    0x011E, 0x00D1, 0x00D2, 0x00D3, 0x00D4, 0x00D5, 0x00D6, 0x00D7,
    0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x0130, 0x015E, 0x00DF,
    0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
    0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
    0x011F, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
    0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x0131, 0x015F, 0x00FF
};
#define WIN1254CP_length sizeof(WIN1254CP) / sizeof(unsigned short);

/* Windows Hebrew Code Page */
unsigned short WIN1255CP [] = {
    0x0080, 0x0081, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021, 
    0x02C6, 0x2030, 0x008A, 0x2039, 0x008C, 0x008D, 0x008E, 0x008F,
    0x0090, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0x02DC, 0x2122, 0x009A, 0x203A, 0x009C, 0x009D, 0x009E, 0x009F, 
    0x00A0, 0x00A1, 0x00A2, 0x00A3, 0x20AA, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0x00AA, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7, 
    0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00BF,
    0x05B0, 0x05B1, 0x05B2, 0x05B3, 0x05B4, 0x05B5, 0x05B6, 0x05B7,
    0x05B8, 0x05B9, 0x05BA, 0x05BB, 0x05BC, 0x05BD, 0x05BE, 0x05BF,
    0x05C0, 0x05C1, 0x05C2, 0x05C3, 0x05F0, 0x05F1, 0x05F2, 0x00D7,
    0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC, 0x00DD, 0x00DE, 0x00DF,
    0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7,
    0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF,
    0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7,
    0x05E8, 0x05E9, 0x05EA, 0x00FB, 0x00FC, 0x200E, 0x200F, 0x00FF
};
#define WIN1255CP_length sizeof(WIN1255CP) / sizeof(unsigned short);

/* Windows Arabic Code Page */
unsigned short WIN1256CP [] = {
    0xFFFE, 0x067E, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
    0x02C6, 0x2030, 0xFFFE, 0x2039, 0x0152, 0x0686, 0x0698, 0xFFFE,
    0x06AF, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0xFFFE, 0x2122, 0xFFFE, 0x203A, 0x0153, 0x200C, 0x200D, 0xFFFE,
    0x00A0, 0x060C, 0x00A2, 0x00A3, 0x00A4, 0x00A5, 0x00A6, 0x00A7,
    0x00A8, 0x00A9, 0xFFFE, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00AF,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00B8, 0x00B9, 0x061B, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x061F,
    0xFFFE, 0x0621, 0x0622, 0x0623, 0x0624, 0x0625, 0x0626, 0x0627,
    0x0628, 0x0629, 0x062A, 0x062B, 0x062C, 0x062D, 0x062E, 0x062F,
    0x0630, 0x0631, 0x0632, 0x0633, 0x0634, 0x0635, 0x0636, 0x00D7,
    0x0637, 0x0638, 0x0639, 0x063A, 0x0640, 0x0641, 0x0642, 0x0643,
    0x00E0, 0x0644, 0x00E2, 0x0645, 0x0646, 0x0647, 0x0648, 0x00E7,
    0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x0649, 0x064A, 0x00EE, 0x00EF,
    0x064B, 0x064C, 0x064D, 0x064E, 0x00F4, 0x064F, 0x0650, 0x00F7,
    0x0651, 0x00F9, 0x0652, 0x00FB, 0x00FC, 0x200E, 0x200F, 0xFFFE
};
#define WIN1256CP_length sizeof(WIN1256CP) / sizeof(unsigned short);

/* Windows Baltic RIM Code Page */
unsigned short WIN1257CP [] = {
    0xFFFE, 0xFFFE, 0x201A, 0xFFFE, 0x201E, 0x2026, 0x2020, 0x2021,
    0xFFFE, 0x2030, 0xFFFE, 0x2039, 0xFFFE, 0x00A8, 0x02C7, 0x00B8,
    0xFFFE, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
    0xFFFE, 0x2122, 0xFFFE, 0x203A, 0xFFFE, 0x00AF, 0x02DB, 0xFFFE, 
    0x00A0, 0xFFFE, 0x00A2, 0x00A3, 0x00A4, 0xFFFE, 0x00A6, 0x00A7,
    0x00D8, 0x00A9, 0x0156, 0x00AB, 0x00AC, 0x00AD, 0x00AE, 0x00C6,
    0x00B0, 0x00B1, 0x00B2, 0x00B3, 0x00B4, 0x00B5, 0x00B6, 0x00B7,
    0x00F8, 0x00B9, 0x0157, 0x00BB, 0x00BC, 0x00BD, 0x00BE, 0x00E6,
    0x0104, 0x012E, 0x0100, 0x0106, 0x00C4, 0x00C5, 0x0118, 0x0112,
    0x010C, 0x00C9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012A, 0x013B,
    0x0160, 0x0143, 0x0145, 0x00D3, 0x014C, 0x00D5, 0x00D6, 0x00D7, 
    0x0172, 0x0141, 0x015A, 0x016A, 0x00DC, 0x017B, 0x017D, 0x00DF,
    0x0105, 0x012F, 0x0101, 0x0107, 0x00E4, 0x00E5, 0x0119, 0x0113,
    0x010D, 0x00E9, 0x017A, 0x0117, 0x0123, 0x0137, 0x012B, 0x013C,
    0x0161, 0x0144, 0x0146, 0x00F3, 0x014D, 0x00F5, 0x00F6, 0x00F7,
    0x0173, 0x0142, 0x015B, 0x016B, 0x00FC, 0x017C, 0x017E, 0x02D9
};
#define WIN1257CP_length sizeof(WIN1257CP) / sizeof(unsigned short);


/*----------------------------------------------------------------------
  uctio
  ----------------------------------------------------------------------*/
int uatoi (const STRING string)
{
#ifdef _I18N_
#ifdef _WINDOWS
  /* 
     Windows provides a routine that allows to convert
     from a wide character string to an integer value 
  */
  return _wtoi (string);
#else  /* _WINDOWS */
  /* 
     For the moment Unix platforms do not provide a routine 
     allowing to convert from a wide character string to an 
     interger.  We have to  convert string  into  multibyte 
     character string and use atoi.
  */
  char str[MAX_LENGTH];

  wcstombs (str, string, MAX_LENGTH);
  return atoi (str);
#endif /* _WINDOWS */
#else  /* _I18N_ */
  return atoi (string);
#endif /* _I18N_ */
}

/*----------------------------------------------------------------------
  uatol converts a strint to a long
  ----------------------------------------------------------------------*/
long uatol (const STRING string)
{
#ifdef _I18N_
#ifdef _WINDOWS
  return _wtol (string);
#else  /* _WINDOWS */
  char str[MAX_LENGTH];

  wcstombs (str, string, MAX_LENGTH);
  return atol (str);
#endif /* _WINDOWS */
#else  /* _I18N_ */
  return atol (string);
#endif /* _I18N_ */
}

/*----------------------------------------------------------------------
 utolower converts uppercases to lowercases
  ----------------------------------------------------------------------*/
CHAR_T utolower (CHAR_T c)
{
#ifdef _I18N_
  return (CHAR_T) towlower((wint_t) c);
#else  /* _I18N_ */
  return ((CHAR_T) tolower ((int) c));
#endif /* _I18N_ */
}


/*----------------------------------------------------------------------
  TtaGetCharFromUnicode: returns the char code in the corresponding encoding
  of  the Unicode value wc.
  ----------------------------------------------------------------------*/
unsigned char TtaGetCharFromUnicode (const wchar_t wc, CHARSET encoding)
{
  unsigned int  c, max;
  unsigned short *table;

  if (wc < 127)
    /* ASCII character */
    return (unsigned char) wc;
  else
    {
      /* look for the right table */
      switch (encoding)
	{
	case ISO_8859_2:
	  table = ISO_8859_2_Map;
	  max = ISO_8859_2_length;
	  break;
	case ISO_8859_3:
	  table = ISO_8859_3_Map;
	  max = ISO_8859_3_length;
	  break;
	case ISO_8859_4:
	  table = ISO_8859_4_Map;
	  max = ISO_8859_4_length;
	  break;
	case ISO_8859_5:
	  table = ISO_8859_5_Map;
	  max = ISO_8859_5_length;
	  break;
	case ISO_8859_6:
	  table = ISO_8859_6_Map;
	  max = ISO_8859_6_length;
	  break;
	case ISO_8859_7:
	  table = ISO_8859_7_Map;
	  max = ISO_8859_7_length;
	  break;
	case ISO_8859_8:
	  table = ISO_8859_8_Map;
	  max = ISO_8859_8_length;
	  break;
	case ISO_8859_9:
	  table = ISO_8859_9_Map;
	  max = ISO_8859_9_length;
	  break;
	case WINDOWS_1250:
	  table = WIN1250CP;
	  max = WIN1250CP_length;
	  break;
	case WINDOWS_1251:
	  table = WIN1251CP;
	  max = WIN1251CP_length;
	  break;
	case WINDOWS_1252:
	  table = WIN1252CP;
	  max = WIN1252CP_length;
	  break;
	case WINDOWS_1253:
	  table = WIN1253CP;
	  max = WIN1253CP_length;
	  break;
	case WINDOWS_1254:
	  table = WIN1254CP;
	  max = WIN1254CP_length;
	  break;
	case WINDOWS_1255:
	  table = WIN1255CP;
	  max = WIN1255CP_length;
	  break;
	case WINDOWS_1256:
	  table = WIN1256CP;
	  max = WIN1256CP_length;
	  break;
	case WINDOWS_1257:
	  table = WIN1257CP;
	  max = WIN1257CP_length;
	  break;
	default:
	  return (unsigned char) wc;
	  break;
	}

      c = 0;
      while (table[c] != wc && c < max)
	c++;
      if (c < max)
	return (unsigned char) (c + 127);
      else
	return (unsigned char) wc;
    }
}


/*----------------------------------------------------------------------
  TtaGetUnicodeFromChar: return the Unicode val corresponding
  to the ISO Latin 2 code c.
  ----------------------------------------------------------------------*/
wchar_t TtaGetUnicodeFromChar (const unsigned char c, CHARSET encoding)
{
  unsigned int  val, max;
  unsigned short *table;

  if (c < 127)
    /* ASCII character */
    return (wchar_t) c;
  else
    {
      val = c - 127;
      /* look for the right table */
      switch (encoding)
	{
	case ISO_8859_2:
	  table = ISO_8859_2_Map;
	  max = ISO_8859_2_length;
	  break;
	case ISO_8859_3:
	  table = ISO_8859_3_Map;
	  max = ISO_8859_3_length;
	  break;
	case ISO_8859_4:
	  table = ISO_8859_4_Map;
	  max = ISO_8859_4_length;
	  break;
	case ISO_8859_5:
	  table = ISO_8859_5_Map;
	  max = ISO_8859_5_length;
	  break;
	case ISO_8859_6:
	  table = ISO_8859_6_Map;
	  max = ISO_8859_6_length;
	  break;
	case ISO_8859_7:
	  table = ISO_8859_7_Map;
	  max = ISO_8859_7_length;
	  break;
	case ISO_8859_8:
	  table = ISO_8859_8_Map;
	  max = ISO_8859_8_length;
	  break;
	case ISO_8859_9:
	  table = ISO_8859_9_Map;
	  max = ISO_8859_9_length;
	  break;
	case WINDOWS_1250:
	  table = WIN1250CP;
	  max = WIN1250CP_length;
	  break;
	case WINDOWS_1251:
	  table = WIN1251CP;
	  max = WIN1251CP_length;
	  break;
	case WINDOWS_1252:
	  table = WIN1252CP;
	  max = WIN1252CP_length;
	  break;
	case WINDOWS_1253:
	  table = WIN1253CP;
	  max = WIN1253CP_length;
	  break;
	case WINDOWS_1254:
	  table = WIN1254CP;
	  max = WIN1254CP_length;
	  break;
	case WINDOWS_1255:
	  table = WIN1255CP;
	  max = WIN1255CP_length;
	  break;
	case WINDOWS_1256:
	  table = WIN1256CP;
	  max = WIN1256CP_length;
	  break;
	case WINDOWS_1257:
	  table = WIN1257CP;
	  max = WIN1257CP_length;
	  break;
	default:
	  return (wchar_t) c;
	  break;
	}
      if (val < max)
	return table[val];
      else
	return (wchar_t) c;
    }
}


/*----------------------------------------------------------------------
  TtaWC2MB converts a wide character into a multibyte character.
  Returns the number of bytes in the multibyte character or -1
  ----------------------------------------------------------------------*/
int TtaWC2MB (wchar_t src, char *dest, CHARSET encoding)
{
  unsigned char   leadByteMark;
  unsigned char  *mbcptr = dest;
  int             nbBytes = 1;

  if (encoding ==  UTF_8)
    {
      if (src < 0x80)
	{
	  nbBytes  = 1;
	  leadByteMark = 0x00;
	}
      else if (src < 0x800)
	{
	  nbBytes      = 2;
	  leadByteMark = 0xC0;
	}
      else if (src < 0x10000)
	{
	  nbBytes      = 3;
	  leadByteMark = 0xE0;
	}
      else if (src < 0x200000)
	{
	  nbBytes      = 4;
	  leadByteMark = 0xF0;
	}
      else if (src < 0x4000000)
	{
	  nbBytes      = 5;
	  leadByteMark = 0xF8;
	}
      else if (src < 0x7FFFFFFF)
	{
	  nbBytes      = 6;
	  leadByteMark = 0xFC;
	}
      else
	{
	  *dest = '?';
	  return -1;
	}

      mbcptr += nbBytes;
      switch (nbBytes)
	{
	case 6:
	  *--mbcptr = (src | 0x80) & 0xBF;
	  src >>= 6;
	case 5:
	  *--mbcptr = (src | 0x80) & 0xBF;
	  src >>= 6;
	case 4:
	  *--mbcptr = (src | 0x80) & 0xBF;
	  src >>= 6;
	case 3:
	  *--mbcptr = (src | 0x80) & 0xBF;
	  src >>= 6;
	case 2:
	  *--mbcptr = (src | 0x80) & 0xBF;
	  src >>= 6;
	case 1:
	  *--mbcptr = src | leadByteMark;
	}
    }
  else
    {
      *dest++ = TtaGetCharFromUnicode (src, encoding);
      *dest   = 0;
    }
  return nbBytes;
}

/*----------------------------------------------------------------------
  TtaWCS2MBs converts a wide character into a multibyte string according to
  the charset.                                                                
  Return the number of bytes in the multibyte character or -1
  The pointer to the source multibyte string is updated.
  ----------------------------------------------------------------------*/
int TtaWC2MBs (wchar_t *src, unsigned char **dest, CHARSET encoding)
{
  wchar_t         wc = src[0];
  int             nbBytes;

  nbBytes = TtaWC2MB (wc, *dest, encoding);
  if (nbBytes > 0)
    {
      *dest += nbBytes;
      return nbBytes;
    }
  else
    return -1;
}



/*----------------------------------------------------------------------
  TtaMBs2WCS converts a multibyte string into a wide character according
  to the charset.
  Returns the number of bytes in the multibyte character or -1
  The pointer to the source multibyte string is updated.
  ----------------------------------------------------------------------*/
int TtaMBs2WC (unsigned char **src, wchar_t *dest, CHARSET encoding)
{
  unsigned char *ptrSrc = *src;
  wchar_t        res;
  int            nbBytesConverted = 0;
  int            nbBytesToConvert;

  if (encoding ==  UTF_8)
    {
      if (*ptrSrc < 0xC0)
	nbBytesToConvert = 1;
      else if (*ptrSrc < 0xE0)
	nbBytesToConvert = 2;
      else if (*ptrSrc < 0xF0)
	nbBytesToConvert = 3;
      else if (*ptrSrc < 0xF8)
	nbBytesToConvert = 4;
      else if (*ptrSrc < 0xFC)
	nbBytesToConvert = 5;
      else if (*ptrSrc <= 0xFF)
	nbBytesToConvert = 6;
                 
      nbBytesConverted += nbBytesToConvert;
      res = 0;
      switch (nbBytesToConvert)
	{
	case 6:
	  res += *ptrSrc++;
	  res <<= 6;
	case 5:
	  res += *ptrSrc++;
	  res <<= 6;
	case 4:
	  res += *ptrSrc++;
	  res <<= 6;
	case 3:
	  res += *ptrSrc++;
	  res <<= 6;
	case 2:
	  res += *ptrSrc++;
	  res <<= 6;
	case 1:
	  res += *ptrSrc++;
	}

      res -= offset[nbBytesToConvert - 1];
      if (res <= 0xFFFF)
	*dest = res;
      else 
	*dest = '?';
    }
  else
    {
      *dest = TtaGetUnicodeFromChar (*ptrSrc, encoding);
      nbBytesConverted++;
      ptrSrc++;
    }
  *src = ptrSrc;
  if (nbBytesConverted > 0)
    return nbBytesConverted;
  else
    return -1;
}


/*----------------------------------------------------------------------
  TtaGetNextWideCharFromMultibyteString: Looks for the next Wide character 
  value in a multibyte character string.
  ----------------------------------------------------------------------*/
int TtaGetNextWideCharFromMultibyteString (wchar_t *car, unsigned char **txt, CHARSET encoding)
{
  int            nbBytesToRead;
  unsigned char *start = *txt;
  wchar_t        res;

  if (encoding ==  UTF_8)
    {
      if (*start < 0xC0)
	nbBytesToRead = 1;
      else if (*start < 0xE0)
	nbBytesToRead = 2;
      else if (*start < 0xF0)
	nbBytesToRead = 3;
      else if (*start < 0xF8)
	nbBytesToRead = 4;
      else if (*start < 0xFC)
	nbBytesToRead = 5;
      else if (*start <= 0xFF)
	nbBytesToRead = 6;
      
      res = 0;
      /* See how many bytes to read to build a wide character */
      switch (nbBytesToRead)
	{
	  /** WARNING: There is not break statement between cases */
	case 6:
	  res += *start++;
	  res <<= 6;
	case 5:
	  res += *start++;
	  res <<= 6;
	case 4:
	  res += *start++;
	  res <<= 6;
	case 3:
	  res += *start++;
	  res <<= 6;
	case 2:
	  res += *start++;
	  res <<= 6;
	case 1:
	  res += *start++;
	}
      res -= offset[nbBytesToRead - 1];      
      *car = res;
    }
  else
    {
      nbBytesToRead = 1;
      *car = TtaGetUnicodeFromChar (*start, encoding);
      start++;
    }
  return nbBytesToRead;
}

/*----------------------------------------------------------------------
  TtaGetNumberOfBytesToRead: 
  Returns the number of bytes to read
  ----------------------------------------------------------------------*/
int  TtaGetNumberOfBytesToRead (unsigned char **txt, CHARSET encoding)
{
  int            nbBytesToRead = 1;
  unsigned char *start = *txt;

  if (encoding ==  UTF_8)
    {
      if (*start < 0xC0)
	nbBytesToRead = 1;
      else if (*start < 0xE0)
	nbBytesToRead = 2;
      else if (*start < 0xF0)
	nbBytesToRead = 3;
      else if (*start < 0xF8)
	nbBytesToRead = 4;
      else if (*start < 0xFC)
	nbBytesToRead = 5;
      else if (*start <= 0xFF)
	nbBytesToRead = 6;
    }

  return nbBytesToRead;
}

/*-------------------------------------------------------------
  TtaCopyWC2Iso copies src (16-bit) into dest (8-bit). This 
  function suposes that enough memory has been already allocated.
  Return the encoding detected.
  -------------------------------------------------------------*/
void TtaCopyWC2Iso (unsigned char *dest, CHAR_T *src, CHARSET encoding)
{
#ifdef _I18N_
  int               i;

  i = 0;
  while (src[i] != WC_EOS)
    {
      dest[i] = TtaGetCharFromUnicode (src[i], encoding);
      i++;
    }
  dest[i] = EOS;
#else  /* _I18N_ */
  strcpy (dest, src);
#endif /* _I18N_ */
}


/*-------------------------------------------------------------
  TtaCopyIso2WC copies src (8-bit) into dest (16-bits). This 
  function suposes that enough memory has been already allocated.
  -------------------------------------------------------------*/
void TtaCopyIso2WC (CHAR_T *dest, unsigned char *src, CHARSET encoding)
{
#ifdef _I18N_ 
  int               i;

  i = 0;
  while (src[i] != EOS)
    {
      dest[i] = TtaGetUnicodeFromChar (src[i], encoding);
      i++;
    }
  dest[i] = WC_EOS;
#else  /* _I18N_ */
  strcpy (dest, src);
#endif /* _I18N_ */
}
