/****************************************************************************************
 * Copyright (c) 2008-2010 Soren Harward <stharward@gmail.com>                          *
 *                                                                                      *
 * This program is free software; you can redistribute it and/or modify it under        *
 * the terms of the GNU General Public License as published by the Free Software        *
 * Foundation; either version 2 of the License, or (at your option) any later           *
 * version.                                                                             *
 *                                                                                      *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY      *
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A      *
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.             *
 *                                                                                      *
 * You should have received a copy of the GNU General Public License along with         *
 * this program.  If not, see <http://www.gnu.org/licenses/>.                           *
 ****************************************************************************************/

#ifndef APG_CONSTRAINTSOLVER
#define APG_CONSTRAINTSOLVER

#include "core/meta/Meta.h"

#include <QMutex>
#include <QString>
#include <threadweaver/Job.h>

class ConstraintNode;

namespace Collections {
    class MetaQueryMaker;
}

namespace APG {
    class ConstraintSolver : public ThreadWeaver::Job {
        Q_OBJECT

        public:
            static const int QUALITY_RANGE; // allows used to adjust speed/quality tradeoff

            ConstraintSolver( ConstraintNode*, int );
            ~ConstraintSolver();

            Meta::TrackList getSolution() const;
            double finalSatisfaction() const;
            int serial() const { return m_serialNumber; }
            int iterationCount() const { return m_maxCoolingIterations; }

            // overloaded ThreadWeaver::Job functions
            bool canBeExecuted();
            bool success() const;

        public slots:
            void requestAbort();

        signals:
            void readyToRun();
            void incrementProgress();

        protected:
            void run(); // from ThreadWeaver::Job

        private slots:
            void receiveQueryMakerData( QString, Meta::TrackList );
            void receiveQueryMakerDone();

        private:
            int m_serialNumber;                 // a randomly-generated serial number to help with debugging
            int m_minPlaylistSize;
            int m_maxPlaylistSize;

            ConstraintNode* const m_constraintTreeRoot;
            Collections::MetaQueryMaker* m_qm;

            Meta::TrackList m_domain;           // tracks available to solver
            QMutex m_domainMutex;
            bool m_domainReductionFailed;
            Meta::TrackList m_solvedPlaylist;   // playlist generated by solver

            bool m_readyToRun;                  // job execution depends on QueryMaker finishing
            bool m_abortRequested;

            // internal mathematical functions
            Meta::TrackPtr randomTrackFromDomain() const;
            double mutateRandom( const double );
            double mutateByVote( const double );
            double improveBySwapping();

            // internal mathematical state data
            double m_satisfactionThreshold;
            double m_qualityFactor;
            int    m_maxCoolingIterations;
            int    m_maxMutationIterations;
            int    m_maxSwapIterations;

            double m_playlistEntropy;
            double m_finalSatisfaction;
    };
} // namespace APG

#endif
