/*
 *  Routines for the universal switch interface
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"
#include "../include/switch.h"

void snd_switch_prepare(snd_kswitch_list_t *list)
{
	list->count = 0;
	list->switches = NULL;
	init_MUTEX(&list->lock);
}

snd_kswitch_t *snd_switch_new(snd_kswitch_t * ksw)
{
	snd_kswitch_t *nsw;

	nsw = (snd_kswitch_t *) snd_kmalloc(sizeof(snd_kswitch_t), GFP_KERNEL);
	if (!nsw)
		return NULL;
	memcpy(nsw, ksw, sizeof(snd_kswitch_t));
	return nsw;
}

void snd_switch_free_one(snd_kswitch_t * ksw)
{
	if (ksw) {
		if (ksw->private_free)
			ksw->private_free(ksw->private_data);
		snd_kfree(ksw);
	}
}

int snd_switch_add(snd_kswitch_list_t *list, snd_kswitch_t *ksw)
{
	snd_kswitch_t **x;

	if (!list || !ksw)
		return -EINVAL;
	down(&list->lock);
	x = (snd_kswitch_t **)
			snd_kmalloc((list->count + 1) * sizeof(void *), GFP_KERNEL);
	if (!x) {
		up(&list->lock);
		return -ENOMEM;
	}
	if (list->switches) {
		memcpy(x, list->switches, (list->count + 1) * sizeof(void *));
		snd_kfree(list->switches);
	}
	list->switches = x;
	list->switches[list->count++] = ksw;
	up(&list->lock);
	return 0;
}

int snd_switch_remove(snd_kswitch_list_t * list, snd_kswitch_t * ksw)
{
	if (!list || !ksw)
		return -EINVAL;
	return -EINVAL;	/* TODO */
}

void snd_switch_free(snd_kswitch_list_t * list)
{
	snd_kswitch_t **x;
	int idx, count;

	down(&list->lock);
	if ((x = list->switches) != NULL) {
		count = list->count;
		list->switches = NULL;
		list->count = 0;
		up(&list->lock);
		for (idx = 0; idx < count; idx++)
			snd_switch_free_one(x[idx]);
		snd_kfree(x);
	} else {
		up(&list->lock);
	}
}

void snd_switch_lock(snd_kswitch_list_t * list, int xup)
{
	if (!xup) {
		down(&list->lock);
	} else {
		up(&list->lock);
	}
}

int snd_switch_list(snd_kswitch_list_t * list, snd_switch_list_t *_list)
{
	int idx;
	snd_switch_list_t xlist;
	snd_switch_list_item_t item;
	snd_kswitch_t *ksw;

	if (copy_from_user(&xlist, _list, sizeof(xlist)))
		return -EFAULT;
	if (xlist.switches_size > 0) {
		if (!xlist.pswitches)
			return -EINVAL;
	}
	snd_switch_lock(list, 0);
	for (idx = 0; idx < list->count && idx < xlist.switches_size; idx++) {
		ksw = list->switches[idx];
		memcpy(item.name, ksw->name, sizeof(item.name));
		if (copy_to_user(&xlist.pswitches[idx], &item, sizeof(item)))
			return -EFAULT;
	}
	xlist.switches = idx;
	xlist.switches_over = list->count - idx;
	snd_switch_lock(list, 1);
	if (copy_to_user(_list, &xlist, sizeof(*_list)))
		return -EFAULT;
	return 0;
}

int snd_switch_read1(snd_kswitch_list_t * list, void *desc, snd_switch_t *_switch)
{
	snd_switch_t sw, sw1;
	snd_kswitch_t *ksw, **x;
	int idx, err = -ENOENT;

	if (copy_from_user(&sw, _switch, sizeof(sw)))
		return -EFAULT;
	x = list->switches;
	for (idx = 0; idx < list->count; idx++) {
		ksw = x[idx];
		if (!strcmp(ksw->name, sw.name)) {
			memcpy(&sw1, &sw, sizeof(sw1));
			memset(&sw, 0, sizeof(sw));
			memcpy(sw.name, sw1.name, sizeof(sw.name));
			err = ksw->get(desc, ksw, &sw);
			break;
		}
	}
	if (!err)
		if (copy_to_user(_switch, &sw, sizeof(sw)))
			return -EFAULT;
	return err;	
}

int snd_switch_read(snd_kswitch_list_t * list, void *desc, snd_switch_t *_switch)
{
	int result;

	snd_switch_lock(list, 0);
	result = snd_switch_read1(list, desc, _switch);
	snd_switch_lock(list, 1);
	return result;
}

int snd_switch_write1(snd_kswitch_list_t * list, void *desc, snd_switch_t *_switch)
{
	snd_switch_t sw;
	snd_kswitch_t *ksw, **x;
	int idx, err = -ENOENT;

	if (copy_from_user(&sw, _switch, sizeof(sw)))
		return -EFAULT;
	x = list->switches;
	for (idx = 0; idx < list->count; idx++) {
		ksw = x[idx];
		if (!strcmp(ksw->name, sw.name)) {
			err = ksw->set(desc, ksw, &sw);
			break;
		}
	}
	if (!err)
		if (copy_to_user(_switch, &sw, sizeof(sw)))
			return -EFAULT;
	return err;	
}

int snd_switch_write(snd_kswitch_list_t * list, void *desc, snd_switch_t *_switch)
{
	int result;

	snd_switch_lock(list, 0);
	result = snd_switch_write1(list, desc, _switch);
	snd_switch_lock(list, 1);
	return result;
}

int snd_switch_count(snd_kswitch_list_t * list)
{
	return list->count;
}

int snd_switch_size(snd_kswitch_list_t * list)
{
	return sizeof(unsigned int *) + (list->count * sizeof(snd_switch_t));
}

long snd_switch_store(snd_kswitch_list_t * list, void *desc, void *ptr, long size)
{
	int idx;
	long result = 0;
	snd_switch_t uswitch;
	snd_kswitch_t *sw;
	mm_segment_t fs;	
	
	if (size < sizeof(unsigned int))
		return -EINVAL;
	*((unsigned int *)ptr) = list->count;
	ptr += sizeof(unsigned int);
	result += sizeof(unsigned int);
	size -= sizeof(unsigned int);
	for (idx = 0; idx < list->count; idx++) {
		if (size < sizeof(snd_switch_t))
			return -ENOMEM;
		memset(&uswitch, 0, sizeof(uswitch));
		sw = list->switches[idx];
		strncpy(uswitch.name, sw->name, sizeof(uswitch.name));
		fs = snd_enter_user();
		snd_switch_read1(list, desc, &uswitch);
		snd_leave_user(fs);
		memcpy(ptr, &uswitch, sizeof(uswitch));
		ptr += sizeof(snd_switch_t);
		result += sizeof(snd_switch_t);
		size -= sizeof(snd_switch_t);
	}
	return result;
}

long snd_switch_restore(snd_kswitch_list_t * list, void *desc, void *ptr, long size)
{
	int idx, count;
	long result = 0;
	snd_switch_t uswitch;
	mm_segment_t fs;

	if (size < sizeof(unsigned int))
		return -EINVAL;
	count = *((unsigned int *)ptr);
	ptr += sizeof(unsigned int);
	size -= sizeof(unsigned int);
	result += sizeof(unsigned int);
	if (count * sizeof(snd_switch_t) < size)
		return -EINVAL;
	for (idx = 0; idx < count; idx++) {
		memcpy(&uswitch, ptr, sizeof(uswitch));
		fs = snd_enter_user();
		snd_switch_write1(list, desc, &uswitch);
		snd_leave_user(fs);
		ptr += sizeof(snd_switch_t);
		result += sizeof(snd_switch_t);
	}
	return result;
}
