# Copyright (C) 2005 Laurent Sansonetti
#
# Alexandria is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# Alexandria is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with Alexandria; see the file COPYING.  If not,
# write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

module Alexandria
module UI
    class LibrariesDataSource < OSX::NSObject
        include OSX

        include GetText
        GetText.bindtextdomain(Alexandria::TEXTDOMAIN, nil, nil, "UTF-8")

        attr_reader :libraries
        
        def awakeFromNib
            @libraries = Library.loadall
        end
        
        def addLibraryWithAutogeneratedName
            i = 1
            while true do
                name = _("Untitled %d") % i
                break unless @libraries.find { |x| x.name == name }
                i += 1
            end
            library = Library.load(name)
            self.addLibrary(library)
        end
        
        def addLibrary(library)
            @libraries << library
            return library
        end
        
        def removeLibraryAtIndex(row)
            library = @libraries[row]
            library.delete
            @libraries.delete(library)
        end
        
        def numberOfRowsInTableView(tableView)
            _librariesCount
        end
        
        def numberOfItemsInComboBox(comboBox)
            _librariesCount
        end
        
        def tableView_objectValueForTableColumn_row(tableView, col, row)
            library = _libraryAtIndex(row)
            return [library.name, Icons::LIBRARY_SMALL]
        end
        
        def comboBox_objectValueForItemAtIndex(comboBox, index)
            _libraryAtIndex(index).name
        end
        
        def tableView_validateDrop_proposedRow_proposedDropOperation(tableView, draggingInfo, row, operation)
            unless operation == NSTableViewDropOn
                return NSDragOperationNone  
            end
            
            pasteboard = draggingInfo.draggingPasteboard
            unless pasteboard.types.containsObject?(BooksDataSource::PASTEBOARD_TYPE)
                return NSDragOperationNone
            end
            
            bookIdents = pasteboard.propertyListForType(BooksDataSource::PASTEBOARD_TYPE).to_a
            library = _libraryAtIndex(row)
            if bookIdents.all? { |x| library.find { |book| book.ident == x.to_s } != nil }
                return NSDragOperationNone
            end

            return NSDragOperationMove
        end

        def tableView_acceptDrop_row_dropOperation(tableView, draggingInfo, row, operation)
            pasteboard = draggingInfo.draggingPasteboard
            unless pasteboard.types.containsObject?(BooksDataSource::PASTEBOARD_TYPE)
                return false
            end
            
            bookIdents = pasteboard.propertyListForType(BooksDataSource::PASTEBOARD_TYPE).to_a
            sourceLibrary = _libraryAtIndex(tableView.selectedRow)
            books = bookIdents.map { |x| sourceLibrary.find { |book| book.ident == x.to_s } }
            return false if books.include?(nil)
            destinationLibrary = _libraryAtIndex(row)
            books = books.select { |x| destinationLibrary.find { |book| book.ident == x } == nil }
            return false if sourceLibrary == destinationLibrary
            
            Library.move(sourceLibrary, destinationLibrary, *books)
            
            if tableView.delegate.respondsToSelector?('acceptedDropOnTableView:')
                tableView.delegate.acceptedDropOnTableView(tableView)
            end
            
            return true
        end

        def tableView_setObjectValue_forTableColumn_row(tableView, objectValue, col, row)
            newName = objectValue.objectAtIndex(0).to_s
            library = _libraryAtIndex(row)
            if library.name != newName and 
               newName !~ /([^\w\s'"()?!:;.\-])/ and
               newName.length > 0 and
               @libraries.find { |x| x.name == newName } == nil

                library.name = newName
            end
        end
        
        #######
        private
        #######
        
        def _librariesCount
            @libraries != nil ? @libraries.length : 0
        end
        
        def _libraryAtIndex(index)
            @libraries[index]
        end        
    end
end
end