// ---------------------------------------------------------------------------
// - Vector.cpp                                                              -
// - standard object library - dynamic vector class implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Real.hpp"
#include "Output.hpp"
#include "Vector.hpp"
#include "Method.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // the vector supported quarks
  static const long QUARK_GET    = String::intern ("get");
  static const long QUARK_SET    = String::intern ("set");
  static const long QUARK_FIND   = String::intern ("find");
  static const long QUARK_RESET  = String::intern ("reset");
  static const long QUARK_GETIT  = String::intern ("get-iterator");
  static const long QUARK_LENGTH = String::intern ("length");
  static const long QUARK_APPEND = String::intern ("append");
  static const long QUARK_EXISTS = String::intern ("exists");
  static const long QUARK_REMOVE = String::intern ("remove");

  // -------------------------------------------------------------------------
  // - vector class section                                                  -
  // -------------------------------------------------------------------------

  // create an empty vector

  Vector::Vector (void) {
    d_size   = 0;
    d_length = 0;
    p_vector = nilp;
  }

  // create a vector with an original size
  
  Vector::Vector (const long size) {
    if (size < 0) throw Exception ("size-error","in vector constructor");
    d_size   = size;
    d_length = 0;
    p_vector = new (Object*)[d_size];
    for (long i = 0; i < d_size; i++) p_vector[i] = nilp;
  }
  
  // copy constructor for this vector

  Vector::Vector (const Vector& that) {
    d_size   = that.d_size;
    d_length = that.d_length;
    p_vector = new (Object*)[d_size];
    for (long i = 0; i < d_length; i++)
      p_vector[i] = Object::iref (that.p_vector[i]);
  }

  // destroy this vector

  Vector::~Vector (void) {
    for (long i = 0; i < d_length; i++)
      Object::dref (p_vector[i]);
    delete [] p_vector;
  }

  // return the class name

  String Vector::repr (void) const {
    return "Vector";
  }

  // assignment operator for this class

  Vector& Vector::operator = (const Vector& that) {
    // protect against this = that
    if (this == &that) return *this;
    
    // clean vector first
    if (d_length != 0) {
      for (long i = 0; i < d_length; i++)
      Object::dref (p_vector[i]);
      delete [] p_vector;
    }

    // copy old to new
    d_size   = that.d_size;
    d_length = that.d_length;
    p_vector = new (Object*)[d_size];
    for (long i = 0; i < d_length; i++)
    p_vector[i] = Object::iref (that.p_vector[i]);
    return *this;
  }

  // make this vector a shared object

  void Vector::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    for (long i = 0; i < d_length; i++) {
      Object* obj = p_vector[i];
      if (obj != nilp) obj->mksho ();
    }
  }

  // return the vector serial code

  t_byte Vector::serialid (void) const {
    return SERIAL_VECT_ID;
  }

  // serialize this vector

  void Vector::wrstream (Output& os) const {
    rdlock ();
    // write the vector length
    Integer vlen (d_length);
    vlen.wrstream (os);
    // write the objects
    for (long i = 0; i < d_length; i++) {
      Object* obj = get (i);
      if (obj == nilp) {
	os.write (Serial::SERIAL_NILP_ID);
      } else {
	Serial* sobj = dynamic_cast <Serial*> (obj);
	if (sobj == nilp) {
	  unlock ();
	  throw Exception ("serial-error", "cannot serialize object", 
			   obj->repr ());
	}
	sobj->serialize (os);
      }
    }
    unlock ();
  }

  // deserialize this vector

  void Vector::rdstream (Input& is) {
    wrlock ();
    reset ();
    // get the vector length
    Integer vlen;
    vlen.rdstream (is);
    long len = vlen.tointeger ();
    // read in each object
    for (long i = 0; i < len; i++) append (Serial::deserialize (is));
    unlock ();
  }

  // add a new element in this vector

  void Vector::append (Object* object) {
    wrlock ();
    try {
      // check if we have to resize the vector
      if (d_length >= d_size) {
	long size = (d_size <= 0) ? 1 : d_size * 2;
	Object** vector = new (Object*)[size];
	for (long i = 0; i < d_length; i++)
	  vector[i] = p_vector[i];
	delete [] p_vector;
	d_size   = size;
	p_vector = vector;
      }
      // check for shared
      if ((p_shared != nilp) && (object != nilp)) object->mksho ();
      // set the object in this vector
      p_vector[d_length++] = Object::iref (object);
      // unlock the vector
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // set an object at a certain position in this vector. The old object is
  // destroyed.
  
  void Vector::set (const long index, Object* object) {
    wrlock ();
    try {
      // check that we are bounded
      if (index >= d_length) 
	throw Exception ("index-error","in vector set");
      // check for shared
      if ((p_shared != nilp) && (object != nilp)) object->mksho ();    
      // set the object
      Object::dref (p_vector[index]);
      p_vector[index] = Object::iref (object);
      // unlock the vector
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get an object at a certain position
  
  Object* Vector::get (const long index) const {
    rdlock ();
    try {
      // check that we are bounded
      if ((index < 0) || (index >= d_length))
	throw Exception ("index-error","index is out of range");
      // get the object and unlock
      Object* result = p_vector[index];
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // backtrack one position in this vector
  void Vector::back (void) {
    wrlock ();
    if (d_length == 0) {
      unlock ();
      return;
    }
    Object::dref (p_vector[--d_length]);
    unlock ();
  }

  // get the number of element in this vector

  long Vector::length (void) const {
    rdlock ();
    long result = d_length;
    unlock ();
    return result;
  }

  // return true if the object exists in this vector

  bool Vector::exists (Object* object) const {
    rdlock ();
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == object) {
	unlock ();
	return true;
      }
    }
    unlock ();
    return false;
  }

  // find an object in this vector

  long Vector::find (Object* object) {
    if (object == nilp) return -1;
    rdlock ();
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == object) {
	unlock ();
	return i;
      }
    }
    unlock ();
    return -1;
  }

  // remove an object by index and repack the vector

  void Vector::remove (const long index) {
    if ((index < 0) || (index >= d_length))
      throw Exception ("index-error","index is out of range");
    wrlock ();
    try {
      // remove the object
      Object::dref (p_vector[index]);
      // repack the vector
      long mark = d_length - 1;
      for (long i = index; i < mark; i++)
	p_vector[i] = p_vector[i+1];
      d_length = mark;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // remove and object and repack the vector

  void Vector::remove (Object* object) {
    if (object == nilp) return;
    wrlock ();
    try {
      long index = find (object);
      if (index == -1) {
	unlock ();
	return;
      }
      remove (index);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // merge a vector into this one

  void Vector::merge (const Vector& v) {
    wrlock ();
    long len = v.length ();
    for (long i = 0; i < len; i++) append (v.get (i));
    unlock ();
  }
      
  // reset this vector

  void Vector::reset (void) {
    wrlock ();
    if (p_vector != nilp) {
      for (long i = 0; i < d_length; i++) Object::dref (p_vector[i]);
    }
    d_length = 0;
    unlock ();
  }

  // return a new vector iterator

  Iterator* Vector::makeit (void) {
    return new Vectorit (this);
  }

  // get an integer value from an index

  t_long Vector::getint (const long index) const {
    Object*  obj  = get (index);
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj == nilp) 
      throw Exception ("type-error", "looking for integer but got",
		       Object::repr (obj));
    return iobj->tointeger ();
  }

  // get a real value from an index

  t_real Vector::getreal (const long index) const {
    Object* obj  = get (index);
    Real*   robj = dynamic_cast <Real*> (obj);
    if (robj == nilp) 
      throw Exception ("type-error", "looking for real but got",
		       Object::repr (obj));
    return robj->toreal ();
  }

  // get a real value from an index (either from an integer)

  t_real Vector::getireal (const long index) const {
    Object*   obj = get (index);
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj != nilp) return iobj->tointeger ();
    Real* robj = dynamic_cast <Real*> (obj);
    if (robj != nilp) return robj->toreal ();
    // not found
    throw Exception ("type-error", "looking for real but got", 
		     Object::repr (obj));
  }

  // get a boolean value from an index

  bool Vector::getbool (const long index) const {
    Object*   obj = get (index);
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    if (bobj == nilp) 
      throw Exception ("type-error", "looking for boolean but got",
		       Object::repr (obj));
    return bobj->toboolean ();
  }

  // get a character value from an index

  char Vector::getchar (const long index) const {
    Object*     obj = get (index);
    Character* cobj = dynamic_cast <Character*> (obj);
    if (cobj == nilp) 
      throw Exception ("type-error", "looking for character but got",
		       Object::repr (obj));
    return cobj->tochar ();
  }

  // get a string value from an index

  String Vector::getstring (const long index) const {
    Object*  obj = get (index);
    String* sobj = dynamic_cast <String*> (obj);
    if (sobj == nilp) 
      throw Exception ("type-error", "looking for string but got",
		       Object::repr (obj));
    return *sobj;
  }

  // generate a vector of arguments

  Vector* Vector::eval (Runnable* robj, Nameset* nset, Cons* args) {
    long len = 0;
    if ((args == nilp) || ((len = args->length ()) == 0)) return nilp;
    Vector* result = new Vector (len);
  
    // loop in the cons cell and accumulate arguments
    try {
      while (args != nilp) {
	Object* car = args->getcar ();
	if (car == nilp) 
	  result->append ((Object*) nilp);
	else
	  result->append (car->eval (robj,nset));
	args = args->getcdr ();
      }
    } catch (...) {
      delete result;
      throw;
    }
    return result;
  }

  // create a new vector in a generic way

  Object* Vector::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Vector;
    return new Vector (*argv);
  }

  // apply a vector method with a set of arguments and a quark

  Object* Vector::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_RESET) {
	reset  ();
	return nilp;
      }
      if (quark == QUARK_GETIT)  return makeit ();
      if (quark == QUARK_LENGTH) return new Integer (length ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GET) {
	rdlock ();
	try {
	  Object* result = get (argv->getint (0));
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_APPEND) {
	Object* result = argv->get (0);
	append (result);
	robj->post (result);
	return result;
      }
      if (quark == QUARK_EXISTS) {
	Object* obj = argv->get (0);
	bool result = exists (obj);
	return new Boolean (result);
      }
      if (quark == QUARK_FIND) {
	Object* obj = argv->get (0);
	long index = find (obj);
	if (index == -1) return nilp;
	return new Integer (index);
      }
      if (quark == QUARK_REMOVE) {
	Object* obj = argv->get (0);
	remove (obj);
	return nilp;
      }
    }

    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SET) {
	t_long val = argv->getint (0);
	Object* result = argv->get (1);
	set (val, result);
	robj->post (result);
	return result;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

  // -------------------------------------------------------------------------
  // - vector iterator class section                                         -
  // -------------------------------------------------------------------------

  // create a new vector iterator

  Vectorit::Vectorit (Vector* vec) {
    p_vector = vec;
    Object::iref (vec);
    begin ();
  }

  // destroy this vector iterator

  Vectorit::~Vectorit (void) {
    Object::dref (p_vector);
  }

  // return the class name

  String Vectorit::repr (void) const {
    return "Vectorit";
  }

  // make this vector iterator a shared object

  void Vectorit::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_vector != nilp) p_vector->mksho ();
  }

  // reset the iterator to the begining

  void Vectorit::begin (void) {
    d_index = 0;
  }

  // reset the iterator to the end

  void Vectorit::end (void) {
    long len = p_vector->d_length;
    d_index  = (len == 0) ? 0 : len - 1;
  }

  // go to the next object

  void Vectorit::next (void) {
    if (++d_index >= p_vector->d_length) 
      d_index = p_vector->d_length;
  }

  // go to the previous object

  void Vectorit::prev (void) {
    if (--d_index < 0) d_index = 0;
  }

  // get the object at the current position

  Object* Vectorit::getobj (void) {
    if (d_index >= p_vector->d_length) return nilp;
    return p_vector->get (d_index);
  }

  // return true if the iterator is at the end

  bool Vectorit::isend (void) {
    if (d_index >= p_vector->d_length) return true;
    return false;
  }
}
