// ---------------------------------------------------------------------------
// - OutputTerm.cpp                                                          -
// - standard object library - terminal output stream class implementation   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"
#include "OutputTerm.hpp"
#include "csio.hpp"
#include "ctrm.hpp"
#include "cstr.hpp"
#include "cerr.hpp"

namespace aleph {

  // create a new default output stream. By default the output stream is
  // mapped to the default standard output

  OutputTerm::OutputTerm (void)  {
    d_sid    = c_stdout ();
    p_tinfo  = c_rtinfo (false);
    d_insert = true;
  }

  // create a new default output stream according to the mode.
  
  OutputTerm::OutputTerm (t_mode mode)  {
    switch (mode) {
    case OutputTerm::OUTPUT:
      d_sid    = c_stdout ();
      break;
    case OutputTerm::ERROR:
      d_sid    = c_stderr ();
      break;
    }
    p_tinfo  = c_rtinfo (false);
    d_insert = true;
  }

  // destroy this output terminal

  OutputTerm::~OutputTerm (void) {
    if (p_tinfo != nilp)
      for (int i = 0; i < OTERM_MAX; i++)
	delete [] p_tinfo[i];
    delete [] p_tinfo;
  }

  // return the class name

  String OutputTerm::repr (void) const {
    return "OutputTerm";
  }

  // return the stream descriptor

  int OutputTerm::getsid (void) const {
    return d_sid;
  }

  // return true if we have a tty

  bool OutputTerm::istty (void) const {
    rdlock ();
    bool result = c_istty (d_sid);
    unlock ();
    return result;
  }

  // set or reset the terminal error mode

  void OutputTerm::temode (const bool mode) {
    wrlock ();
    c_temode (d_sid, p_tinfo, mode);
    unlock ();
  }

  // write one character to the output stream
  
  void OutputTerm::write (const char value) {
    wrlock ();
    long count = c_write (d_sid, &value,1);
    if (count < 0) {
      unlock ();
      throw Exception ("write-error", c_errmsg (count));
    }
    unlock ();
  }

  // write a character string to the output stream
  
  void OutputTerm::write (const char* value) {
    wrlock ();
    long size = c_strlen (value);
    if (size == 0) {
      unlock ();
      return;
    }
    // write the string
    long count = c_write (d_sid, value,size);
    if (count < 0) {
      unlock ();
      throw Exception ("write-error", c_errmsg (count));
    }
    unlock ();
  }

  // write a string with the terminal in error mode

  void OutputTerm::error (const String& value) {
    wrlock ();
    try {
      c_temode      (d_sid, p_tinfo,true);
      Output::write (value);
      c_temode      (d_sid, p_tinfo, false);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // insert a character based on the mode

  void OutputTerm::insert (const char c) {
    wrlock ();
    try {
      // check that we have terminal capabilities
      if (p_tinfo == nilp) {
	write (c);
	unlock ();
	return;
      }
      // check for mode and perform insert
      if (d_insert == true) {
	if (p_tinfo[OTERM_INSERT_CHAR] != nilp) {
	  c_tparm (d_sid,p_tinfo, OTERM_INSERT_CHAR);
	  write (c);
	} else {
	  c_tparm (d_sid,p_tinfo, OTERM_IMODE_START);
	  write (c);
	  c_tparm (d_sid,p_tinfo, OTERM_IMODE_END);
	}
      } else
	write (c);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // erase on the left a certain amount of characters
  
  void OutputTerm::erasel (const long num) {
    if (num <= 0) return;
    wrlock ();
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_LEFT);
      c_tparm (d_sid,p_tinfo, OTERM_DELETE_CHAR);
    }
    unlock ();
  }
  
  // delete one character at the cursor position
  
  void OutputTerm::del (void) {
    wrlock ();
    c_tparm (d_sid,p_tinfo, OTERM_DELETE_CHAR);
    unlock ();
  }
  
  // move to the left by a certain number of characters
  
  void OutputTerm::movel (const long num) {
    if (num <= 0) return;
    wrlock ();
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_LEFT);
    }
    unlock ();
  }
  
  // move to the right by a certain amount
  
  void OutputTerm::mover (const long num) {
    if (num <= 0) return;
    wrlock ();
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_RIGHT);
    }
    unlock ();
  }

  // move up by a certain amount
  
  void OutputTerm::moveu (const long num) {
    if (num <= 0) return;
    wrlock ();
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_UP);
    }
    unlock ();
  }
  
  // move down by a certain amount
  
  void OutputTerm::moved (const long num) {
    if (num <= 0) return;
    wrlock ();
    for (long i = 0; i < num; i++) {
      c_tparm (d_sid,p_tinfo, OTERM_MOVE_DOWN);
    }
    unlock ();
  }

  // clear the screen and refresh with value
  
  bool OutputTerm::refresh (const String& value) {
    wrlock ();
    try {
      if (p_tinfo[OTERM_CLEAR_SCREEN] != nilp) {
	c_tparm (d_sid,p_tinfo, OTERM_CLEAR_SCREEN);
	Output::write (value);
	unlock ();
	return true;
      }
      unlock ();
      return false;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // create a new output term in a generic way

  Object* OutputTerm::mkout (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", 
		       "invalid arguments with with output term"); 
    return new OutputTerm (OutputTerm::OUTPUT);
  }

  // create a new error term in a generic way

  Object* OutputTerm::mkerr (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", 
		       "invalid arguments with with error term"); 
    return new OutputTerm (OutputTerm::ERROR);
  }

  // apply this output term with a set of arguments and a quark

  Object* OutputTerm::apply (Runnable* robj, Nameset* nset, const long quark,
			     Vector* argv) {
    return Output::apply (robj, nset, quark, argv);
  }
}
