// ---------------------------------------------------------------------------
// - System.cpp                                                              -
// - standard object library - system function class implementation          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "System.hpp"
#include "Integer.hpp"
#include "Exception.hpp"
#include "csio.hpp"
#include "csys.hpp"
#include "cdir.hpp"
#include "cclk.hpp"
#include "cstr.hpp"
#include "ccnv.hpp"

namespace aleph {

  // register the program name

  void System::setpgm (const String& name) {
    char* data = name.tochar ();
    c_setpgm (data);
    delete [] data;
  }

  // get the program name

  String System::getpgm (void) {
    char* name = c_getpgm ();
    String result = name;
    delete [] name;
    return result;
  }

  // return the host name
  
  String System::hostname (void) {
    char* name = c_hostname ();
    String result = name;
    delete [] name;
    return result;
  }

  // return the user name
  
  String System::username (void) {
    char* name = c_username ();
    String result = name;
    delete [] name;
    return result;
  }

  // return the aleph url
  
  String System::geturl (void) {
    return "http://www.aleph-lang.org";
  }

  // return the aleph major version
  
  String System::major (void) {
    Integer val (ALEPH_VERSION_MAJOR);
    return val.tostring ();
  }

  // return the aleph minor version
  
  String System::minor (void) {
    Integer val (ALEPH_VERSION_MINOR);
    return val.tostring ();
  }

  // return the aleph patch version
  
  String System::patch (void) {
    Integer val (ALEPH_VERSION_PATCH);
    return val.tostring ();
  }

  // return the os name
  
  String System::osname (void) {
    return c_osname ();
  }

  // return the os type

  String System::ostype (void) {
    return c_ostype ();
  }

  // return the version of the aleph system
  
  String System::version (void) {
    return ALEPH_VERSION_TOTAL;
  }

  // return the option separator

  char System::optsep (void) {
    return c_optsep ();
  }

  // return an environment variable value

  String System::getenv (const String& name) {
    const char* buf = name.tochar ();
    String result = c_getenv (buf);
    delete [] buf;
    return result;
  }

  // exit unconditionally with a status code
  
  void System::exit (int status) {
    c_exit (status);
  }

  // return the process id

  long System::getpid (void) {
    return c_getpid ();
  }

  // pause for a certain time

  void System::sleep (const long time) {
    c_sleep (time);
  }

  // swap a word between host and network order

  t_word System::wswap (const t_word value) {
    return c_wswap (value);
  }

  // swap a quad between host and network order

  t_quad System::qswap (const t_quad value) {
    return c_qswap (value);
  }

  // swap a word between host and network order

  t_octa System::oswap (const t_octa value) {
    return c_oswap (value);
  }

  // remove the file name extension
  
  String System::rmext (const String& name) {
    char* fname = name.tochar ();
    char* data  = c_rmext (fname);
    String result = data;
    delete [] fname;
    delete [] data;
    return result;
  }
    
  // join two names to form a new file name
  
  String System::join (const String& dir, const String& name) {
    String result = dir + c_dirsep () + name;
    return result;
  }

  // extract a file name from a path

  String System::xname (const String& path) {
    char* data = path.tochar ();
    char* name = c_xname (data);
    String result = name;
    delete [] data;
    delete [] name;
    return result;
  }

  // extract a file name extension

  String System::xext (const String& name) {
    char* data = name.tochar ();
    char* fext = c_xext (data);
    String result = fext;
    delete [] data;
    delete [] fext;
    return result;
  }

  // return true if the name corresponds to a regular file name
  
  bool System::isfile (const String& name) {
    char* fname = name.tochar ();
    bool status = c_isfile (fname);
    delete [] fname;
    return status;
  }
  
  // return true if the name corresponds to a directory name
  
  bool System::isdir (const String& name) {
    char* dname = name.tochar ();
    bool status = c_isdir (dname);
    delete [] dname;
    return status;
  }

  // remove a file by name

  bool System::rmfile (const String& name) {
    char* fname = name.tochar ();
    bool status = c_rm (fname);
    delete [] fname;
    return status;
  }

  // remove a directory by name

  bool System::rmdir (const String& name) {
    char* dname = name.tochar ();
    bool status = c_rmdir (dname);
    delete [] dname;
    return status;
  }
}
