// ---------------------------------------------------------------------------
// - Buffer.cpp                                                              -
// - standard object library - character buffer class implementation         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Buffer.hpp"
#include "Integer.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "ccnv.hpp"

namespace aleph {

  // default buffer size
  const long BUFFER_SIZE = 1024;

  // the buffer supported quarks
  static const long QUARK_ADD      = String::intern ("add");
  static const long QUARK_GET      = String::intern ("get");
  static const long QUARK_READ     = String::intern ("read");
  static const long QUARK_RESET    = String::intern ("reset");
  static const long QUARK_WRITE    = String::intern ("write");
  static const long QUARK_LENGTH   = String::intern ("length");
  static const long QUARK_GETWORD  = String::intern ("get-word");
  static const long QUARK_GETQUAD  = String::intern ("get-quad");
  static const long QUARK_GETOCTA  = String::intern ("get-octa");
  static const long QUARK_TOSTRING = String::intern ("to-string");
  static const long QUARK_PUSHBACK = String::intern ("pushback");

  // create a new buffer class with a default size of 1024 characters

  Buffer::Buffer (void) {
    p_buffer = new char [BUFFER_SIZE];
    d_size   = BUFFER_SIZE;
    d_length = 0;
  }

  // create a new buffer with a predefined size

  Buffer::Buffer (const long size) {
    d_size   = (size <= 0) ? BUFFER_SIZE : size;
    p_buffer = new char[d_size];
    d_length = 0;
  }

  // create a new buffer and initialize it with a c string

  Buffer::Buffer (const char* value) {
    d_size   = BUFFER_SIZE;
    p_buffer = new char[d_size];
    d_length = 0;
    add (String (value));
  }

  // create a new buffer and initialize it with a string

  Buffer::Buffer (const String& value) {
    d_size   = BUFFER_SIZE;
    p_buffer = new char[d_size];
    d_length = 0;
    add (value);
  }

  // copy construct a buffer

  Buffer::Buffer (const Buffer& that) {
    that.rdlock ();
    d_size   = that.d_size;
    d_length = that.d_length;
    p_buffer = new char[d_size];
    for (long i = 0; i < d_length; i++) p_buffer[i] = that.p_buffer[i];
    that.unlock ();
  }

  // destroy this buffer
  
  Buffer::~Buffer (void) {
    delete [] p_buffer;
  }
  
  // return the class name

  String Buffer::repr (void) const {
    return "Buffer";
  }

  // add a character in this buffer
  
  void Buffer::add (const char value) {
    wrlock ();
    // first check if we are at the buffer end
    if (d_length >= d_size) {
      long size = d_size * 2;
      char* buf = new char[size];
      for (long i = 0; i < d_length; i++) buf[i] = p_buffer[i];
      delete [] p_buffer;
      d_size   = size;
      p_buffer = buf;
    }
    p_buffer[d_length++] = value;
    unlock ();
  }

  // add a buffer in this buffer
  
  void Buffer::add (const char* buffer, const long size) {
    if ((buffer == nilp) || (size == 0)) return;
    wrlock ();
    try {
      for (int i = 0; i < size; i++) add (buffer[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // add a string in this buffer
  
  void Buffer::add (const String& value) {
    wrlock ();
    try {
      int len = value.length ();
      for (int i = 0; i < len; i++) add (value[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // add a buffer object to this buffer

  void Buffer::add (const Buffer& buffer) {
    wrlock ();
    buffer.rdlock ();
    try {
      add (buffer.p_buffer, buffer.d_length);
    } catch (...) {
      buffer.unlock ();
      unlock ();
      throw;
    }
    buffer.unlock ();
    unlock ();
  }

  // get the next character but do not remove it

  char Buffer::get (void) const {
    rdlock ();
    char result = (d_length == 0) ? nilc : p_buffer[0];
    unlock ();
    return result;
  }
  
  // read a character in this buffer
  
  char Buffer::read (void) {
    wrlock ();
    // check for no character
    if (d_length == 0) {
      unlock ();
      return nilc;
    }
    // get value and shift
    char value = p_buffer[0];
    for (long i = 0; i < d_length; i++) p_buffer[i] = p_buffer[i+1];
    d_length--;
    unlock ();
    return value;
  }
  
  // pushback a character in this buffer
  
  void Buffer::pushback (const char value) {
    wrlock ();
    // check if we are full
    if (d_length == d_size) {
      long size = d_size * 2;
      char* buf = new char[size];
      for (long i = 0; i < d_length; i++) buf[i] = p_buffer[i];
      d_size = size;
      delete [] p_buffer;
      p_buffer = buf;
    }
    // shift the buffer by one
    for (long i = d_length; i > 0; i--) p_buffer[i] = p_buffer[i-1];
    p_buffer[0] = value;
    d_length++;
    unlock ();
  }

  // pushback a buffer in this buffer
  
  void Buffer::pushback (const char* buffer, const long size) {
    if ((buffer == nilp) || (size == 0)) return;
    wrlock ();
    long len = size - 1;
    for (long i = len; i >= 0; i--) pushback (buffer[i]);
    unlock ();
  }
  
  // pushback a string in this buffer
  
  void Buffer::pushback (const String& value) {
    wrlock ();
    long len = value.length () - 1;
    for (long i = len; i >= 0; i--) pushback (value[i]);
    unlock ();
  }
  
  // return the length of this buffer
  
  long Buffer::length (void) const {
    rdlock ();
    long result = d_length;
    unlock ();
    return result;
  }

  // return the corresponding string accumulated in this buffer
  
  String Buffer::tostring (void) const {
    rdlock ();
    if (d_length == 0) {
      unlock ();
      return String ();
    }
    // create a temporary buffer and set
    char * buf = new char[d_length+1];
    for (long i = 0; i < d_length; i++) buf[i] = p_buffer[i];
    buf[d_length] = nilc;
    // make the result string
    String result = buf;
    delete [] buf;
    // readjust index
    unlock ();
    return result;
  }

  // reset this buffer but do not change the size
  
  void Buffer::reset (void) {
    wrlock ();
    d_length = 0;
    unlock ();
  }

  // map this buffer to an anonymous data structure

  long Buffer::map (void* data, const long size) {
    rdlock ();
    long result = (size < d_length) ? size : d_length;
    char* ptr = (char*) data;
    for (long i = 0; i < result; i++) ptr[i] = p_buffer[i];
    unlock ();
    return result;
  }

  // get a signed word in big endian format from this buffer

  t_word Buffer::getword (void) {
    wrlock ();
    if (length () < 2) {
      unlock ();
      throw Exception ("buffer-error", "not enough character for getword");
    }
    // extract the char buffer
    t_byte buf[2];
    for (long i = 0; i < 2; i++) buf[i] = read ();
    // convert it in host format
    t_word result = c_wntoh (buf);
    unlock ();
    return result;
  }

  // get a signed quad in big endian format from this buffer

  t_quad Buffer::getquad (void) {
    wrlock ();
    if (length () < 4) {
      unlock ();
      throw Exception ("buffer-error", "not enough character for getquad");
    }
    // extract the char buffer
    t_byte buf[4];
    for (long i = 0; i < 4; i++) buf[i] = read ();
    // convert it in host format
    t_quad result = c_qntoh (buf);
    unlock ();
    return result;
  }

  // get an octa in big endian format from this buffer

  t_octa Buffer::getocta (void) {
    wrlock ();
    if (length () < 8) {
      unlock ();
      throw Exception ("buffer-error", "not enough character for getquad");
    }
    // extract the char buffer
    t_byte buf[8];
    for (long i = 0; i < 8; i++) buf[i] = read ();
    // convert it in host format
    t_octa result = c_ontoh (buf);
    unlock ();
    return result;
  }

  // write the buffer content to an output stream
  
  void Buffer::write (Output& os) const {
    wrlock ();
    os.write (p_buffer, d_length);
    unlock ();
  }
  
  // create a new buffer in a generic way

  Object* Buffer::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create an empty buffer with 0 arguments
    Buffer* result = new Buffer;
    // loop with literal objects
    for (long i = 0; i < argc; i++) {
      Object*   obj = argv->get (i);
      Literal* lobj = dynamic_cast <Literal*> (obj);
      if (lobj == nilp)
	throw Exception ("type-error", "invalid object with buffer",
			 Object::repr (obj));
      result->add (lobj->tostring ());
    }
    return result;
  }

  // apply this buffer with a set of arguments and a quark

  Object* Buffer::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GET)      return new Character (get ());
      if (quark == QUARK_READ)     return new Character (read ());
      if (quark == QUARK_LENGTH)   return new Integer (length   ());
      if (quark == QUARK_GETWORD)  return new Integer (getword  ());
      if (quark == QUARK_GETQUAD)  return new Integer (getquad  ());
      if (quark == QUARK_GETOCTA)  return new Integer (getocta  ());
      if (quark == QUARK_TOSTRING) return new String  (tostring ());
      if (quark == QUARK_RESET) {
	reset ();
	return nilp;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	Object* obj = argv->get (0);
	// check for a character
	Character* cobj = dynamic_cast<Character*> (obj);
	if (cobj != nilp) {
	  add (cobj->tochar ());
	  return nilp;
	}
	// check for a literal
	Literal* lobj = dynamic_cast<Literal*> (obj);
	if (lobj != nilp) {
	  add (lobj->tostring ());
	  return nilp;
	}
	// check for a buffer
	Buffer* bobj = dynamic_cast<Buffer*> (obj);
	if (bobj != nilp) {
	  add (*bobj);
	  return nilp;
	}
      }
      if (quark == QUARK_PUSHBACK) {
	Object* obj = argv->get (0);
	// check for a character
	Character* cobj = dynamic_cast<Character*> (obj);
	if (cobj != nilp) {
	  pushback (cobj->tochar ());
	  return nilp;
	}
	// check for a literal
	Literal* lobj = dynamic_cast<Literal*> (obj);
	if (lobj != nilp) {
	  pushback (lobj->tostring ());
	  return nilp;
	}
      }
      if (quark == QUARK_WRITE) {
	Output* os = dynamic_cast <Output*> (argv->get (0));
	if (os == nilp) 
	  throw Exception ("type-error", "output object expected with write");
	write (*os);
	return nilp;
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
