// ---------------------------------------------------------------------------
// - Time.cpp                                                                -
// - aleph:sys library - time class implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Time.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Exception.hpp"
#include "cclk.hpp"

namespace aleph {
  // the supported time quarks
  static const long QUARK_ADD     = String::intern ("add");
  static const long QUARK_GETTIME = String::intern ("get-time");
  static const long QUARK_GETSECS = String::intern ("seconds");
  static const long QUARK_GETMINS = String::intern ("minutes");
  static const long QUARK_GETHOUR = String::intern ("hours");
  static const long QUARK_GETMDAY = String::intern ("day-of-month");
  static const long QUARK_GETYMON = String::intern ("month-of-year");
  static const long QUARK_GETYEAR = String::intern ("year");
  static const long QUARK_GETWDAY = String::intern ("day-of-week");
  static const long QUARK_GETYDAY = String::intern ("day-of-year");
  static const long QUARK_FMTDATE = String::intern ("format-date");
  static const long QUARK_FMTTIME = String::intern ("format-time");
  static const long QUARK_UTCSECS = String::intern ("utc-seconds");
  static const long QUARK_UTCMINS = String::intern ("utc-minutes");
  static const long QUARK_UTCHOUR = String::intern ("utc-hours");
  static const long QUARK_UTCMDAY = String::intern ("utc-day-of-month");
  static const long QUARK_UTCYMON = String::intern ("utc-month-of-year");
  static const long QUARK_UTCYEAR = String::intern ("utc-year");
  static const long QUARK_UTCWDAY = String::intern ("utc-day-of-week");
  static const long QUARK_UTCYDAY = String::intern ("utc-day-of-year");
  static const long QUARK_UTCDATE = String::intern ("utc-format-date");
  static const long QUARK_UTCTIME = String::intern ("utc-format-time");
  static const long QUARK_RFCDATE = String::intern ("utc-format-rfc");
  static const long QUARK_COKDATE = String::intern ("utc-format-cookie");

  // week day string mapping
  const char* DAY_NAMES[7]  = {"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
  const char* MON_NAMES[12] = {"Jan","Feb","Mar","Apr","May","Jun","Jul",
			       "Aug","Sep","Oct","Nov","Dec"};

  // create a new time instance

  Time::Time (void) {
    d_tclk  = c_time     ();
    p_uinfo = c_getutc   (d_tclk);
    p_linfo = c_getlocal (d_tclk);
  }

  // create a new time from counter timer

  Time::Time (const long tval) {
    d_tclk  = tval;
    p_uinfo = c_getutc   (d_tclk);
    p_linfo = c_getlocal (d_tclk);
  }

  // destroy this time instance

  Time::~Time (void) {
    delete p_uinfo;
    delete p_linfo;
  }

  // return the class name

  String Time::repr (void) const {
    return "Time";
  }

  // add a certain time in second to this time

  void Time::add (const long tval) {
    wrlock ();
    d_tclk += tval;
    p_uinfo = c_getutc   (d_tclk);
    p_linfo = c_getlocal (d_tclk);
    unlock ();
  }

  // return the time since the epoch

  long Time::gettime (void) const {
    return d_tclk;
  }

  // return a particular time info

  long Time::getinfo (const bool utc, const t_tdata type) const {
    rdlock ();
    s_tinfo* tinfo = utc ? p_uinfo : p_linfo;
    if (tinfo == nilp) {
      unlock ();
      throw Exception ("time-error", "invalid time access request");
    }
    long result = 0;
    switch (type) {
    case Time::SECS:
      result = tinfo->d_secs;
      break;
    case Time::MINS:
      result = tinfo->d_mins;
      break;
    case Time::HOUR:
      result = tinfo->d_hour;
      break;
    case Time::MDAY:
      result = tinfo->d_mday;
      break;
    case Time::YMON:
      result = tinfo->d_ymon;
      break;
    case Time::YEAR:
      result = tinfo->d_year;
      break;
    case Time::WDAY:
      result = tinfo->d_wday;
      break;
    case Time::YDAY:
      result = tinfo->d_yday;
      break;
    }
    unlock ();
    return result;
  }

  // format the date

  String Time::formatdate (const bool utc) const {
    String result;
    rdlock ();
    result = result + getinfo (utc, Time::YMON) + '/';
    result = result + getinfo (utc, Time::MDAY) + '/';
    result = result + getinfo (utc, Time::YEAR);
    unlock ();
    return result;
  }

  // format the time

  String Time::formattime (const bool utc) const {
    String result;
    rdlock ();
    long hour = getinfo (utc, Time::HOUR);
    if (hour < 10) {
      result = result + '0' + hour + ':';
    } else {
      result = result + hour + ':';
    }
    long mins = getinfo (utc, Time::MINS);
    if (mins < 10) {
      result = result + '0' + mins + ':';
    } else {
      result = result + mins + ':';
    }
    long secs = getinfo (utc, Time::SECS);
    if (secs < 10) {
      result = result + '0' + secs;
    } else {
      result = result + secs;
    }
    unlock ();
    return result;
  }

  // format the date for a cookie expire time

  String Time::fmtcookie (void) const {
    String result;
    rdlock ();
    // get the day of week
    long wday = getinfo (true, WDAY);
    if ((wday < 0) || (wday > 6)) wday = 0;
    result = result + DAY_NAMES[wday] + ", ";
    // get the day in month
    long mday = getinfo (true, MDAY);
    if (mday < 10) {
      result = result + '0' + mday + '-';
    } else {
      result = result + mday + '-';
    }
    // get the year month
    long ymon = getinfo (true, YMON) - 1;
    if ((ymon < 0) || (ymon > 11)) ymon = 0;
    result = result + MON_NAMES[ymon] + '-';
    // get the year
    long year = getinfo (true, YEAR);
    result = result + year + ' ';
    // get the time
    result = result + formattime (true) + " GMT";
    unlock ();
    return result;
  }

  // format the date according to rfc 822

  String Time::formatrfc (void) const {
    String result;
    rdlock ();
    // get the day of week
    long wday = getinfo (true, WDAY);
    if ((wday < 0) || (wday > 6)) wday = 0;
    result = result + DAY_NAMES[wday] + ", ";
    // get the day in month
    long mday = getinfo (true, MDAY);
    if (mday < 10) {
      result = result + '0' + mday + ' ';
    } else {
      result = result + mday + ' ';
    }
    // get the year month
    long ymon = getinfo (true, YMON) - 1;
    if ((ymon < 0) || (ymon > 11)) ymon = 0;
    result = result + MON_NAMES[ymon] + ' ';
    // get the year
    long year = getinfo (true, YEAR);
    result = result + year + ' ';
    // get the time
    result = result + formattime (true) + " GMT";
    unlock ();
    return result;
  }
  // create a new time instance in a generic way
 
  Object* Time::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default time object
    if (argc == 0) return new Time;
    if (argc == 1) {
      long tval = argv->getint (0);
      return new Time (tval);
    }
    throw Exception ("argument-error",
                     "too many argument with time constructor");
  }
 
  // apply a time member with a set of arguments and a quark
 
  Object* Time::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 arguments
    if (argc == 0) {
      if (quark == QUARK_GETTIME) return new Integer (gettime   ());
      if (quark == QUARK_COKDATE) return new String  (fmtcookie ());
      if (quark == QUARK_RFCDATE) return new String  (formatrfc ());
      if (quark == QUARK_FMTDATE) return new String (formatdate (false));
      if (quark == QUARK_UTCDATE) return new String (formatdate (true));
      if (quark == QUARK_FMTTIME) return new String (formattime (false));
      if (quark == QUARK_UTCTIME) return new String (formattime (true));
      if (quark == QUARK_GETSECS)
	return new Integer (getinfo (false,Time::SECS));
      if (quark == QUARK_GETMINS)
	return new Integer (getinfo (false,Time::MINS));
      if (quark == QUARK_GETHOUR)
	return new Integer (getinfo (false,Time::HOUR));
      if (quark == QUARK_GETMDAY)
	return new Integer (getinfo (false,Time::MDAY));
      if (quark == QUARK_GETYMON)
	return new Integer (getinfo (false,Time::YMON));
      if (quark == QUARK_GETYEAR)
	return new Integer (getinfo (false,Time::YEAR));
      if (quark == QUARK_GETWDAY)
	return new Integer (getinfo (false,Time::WDAY));
      if (quark == QUARK_GETYDAY)
	return new Integer (getinfo (false,Time::YDAY));
      if (quark == QUARK_UTCSECS)
	return new Integer (getinfo (true,Time::SECS));
      if (quark == QUARK_UTCMINS)
	return new Integer (getinfo (true,Time::MINS));
      if (quark == QUARK_UTCHOUR)
	return new Integer (getinfo (true,Time::HOUR));
      if (quark == QUARK_UTCMDAY)
	return new Integer (getinfo (true,Time::MDAY));
      if (quark == QUARK_UTCYMON)
	return new Integer (getinfo (true,Time::YMON));
      if (quark == QUARK_UTCYEAR)
	return new Integer (getinfo (true,Time::YEAR));
      if (quark == QUARK_UTCWDAY)
	return new Integer (getinfo (true,Time::WDAY));
      if (quark == QUARK_UTCYDAY)
	return new Integer (getinfo (true,Time::YDAY));
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	long tval = argv->getint (0);
	add (tval);
	return nilp;
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
