// ---------------------------------------------------------------------------
// - HashTable.cpp                                                           -
// - standard object library - hash table class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "HashTable.hpp"
#include "Exception.hpp"
#include "ccnv.hpp"

namespace aleph {
  
  // the hash table bucket
  struct s_bucket {
    // the object name
    String d_name;
    // the hash id value
    long   d_hvl;
    // the object 
    Object* p_object;
    // next record in the list
    s_bucket* p_next;
    // simple constructor
    s_bucket (void) {
      d_hvl    = 0;
      p_object = nilp;
      p_next   = nilp;
    }
    // simple destructor
    ~s_bucket (void) {
      Object::dref (p_object);
      delete p_next;
    }
  };
  
  // find a bucket by name given its root bucket
  static inline s_bucket* getbucket (s_bucket* bucket, const String& name) {
    // simple check as fast as we can
    if (bucket == nilp) return nilp;
    // loop until we have a match
    while (bucket != nilp) {
      if (bucket->d_name == name) return bucket;
      bucket = bucket->p_next;
    }
    // no bucket found
    return nilp;
  }
  
  // extract a bucket by name given its root bucket . This procedure remove the
  // bucket if it is found and maintain the link list.
  static inline s_bucket* rmbucket (s_bucket** root, const String& name) {
    s_bucket* bucket = *root;
    // simple check as fast as we can
    if (bucket == nilp) return nilp;
    // first case for the root bucket
    if (bucket->d_name == name) {
      *root = bucket->p_next;
      bucket->p_next = nilp;
      return bucket;
    }
    // loop until we have a match
    while (bucket->p_next != nilp) {
      if (bucket->p_next->d_name == name) {
	s_bucket* result = bucket->p_next;
	bucket->p_next = result->p_next;
	result->p_next = nilp;
	return result;
      }
      bucket = bucket->p_next;
    } 
    // no node found
    return nilp;
  }
  
  // create a new hash table
  
  HashTable::HashTable (void) {
    // build the array
    d_size   = c_prime (0);
    d_thrs   = (d_size * 7) / 10;
    d_count  = 0;
    p_table  = new (s_bucket*)[d_size];
    
    // initialize the table with null pointers
    for (long i = 0; i < d_size; i++)
      p_table[i] = nilp;
  }
  
  // create a new hash table with a predefined size
  
  HashTable::HashTable (const long size) {
    // build the array - threshold at 70%
    d_size   = c_prime (size);
    d_thrs   = (size * 7) / 10;
    d_count  = 0;
    p_table  = new (s_bucket*)[d_size];
    
    // initialize the table with null pointers
    for (long i = 0; i < d_size; i++)
      p_table[i] = nilp;
  }
  
  // delete this hash table 
  
  HashTable::~HashTable (void) {
    if (p_table != nilp) {
      for (long i = 0; i < d_size; i++) delete p_table[i];
      delete [] p_table;
    }
  }

  // return the class name

  String HashTable::repr (void) const {
    return "HashTable";
  }

  // make this hash table a shared object

  void HashTable::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    for (long i = 0; i < d_size; i++) {
      s_bucket* bucket = p_table[i];
      while (bucket != nilp) {
	Object* obj = bucket->p_object;
	if (obj != nilp) obj->mksho ();
	bucket = bucket->p_next;
      }
    }
  }

  // set or create an object in this table
  
  void HashTable::add (const String& name, Object* object) {
    // protect the object
    Object::iref (object);
    // compute the hash value
    long hvl = name.hashid ();
    long hid = hvl % d_size;
    // look for existing symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) {
      Object::dref (bucket->p_object);
      bucket->p_object = object;
      return;
    }
    // the bucket does not exist, create it 
    bucket           = new s_bucket;
    bucket->d_name   = name;
    bucket->d_hvl    = hvl;
    bucket->p_object = object;
    bucket->p_next   = p_table[hid];
    p_table[hid]     = bucket;
    if (++d_count > d_thrs) resize (c_prime (d_size + 1));
  }
  
  // get an object by name. If the name is not found, nilp is returned
  
  Object* HashTable::get (const String& name) const {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) return bucket->p_object;;
    return nilp;
  }

  // get an object by name. If the name is not found an exception is raised

  Object* HashTable::lookup (const String& name) const {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) return bucket->p_object;;
    throw Exception ("name-error", "name not found", name);
  }
  
  // return true if a name exists in this table

  bool HashTable::exists (const String& name) const {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // look for the node and find symbol
    s_bucket* bucket = getbucket (p_table[hid],name);
    if (bucket != nilp) return true;
    return false;
  }
  
  // remove an object by name. 
  
  void HashTable::remove (const String& name) {
    // compute hash id
    long hid = name.hashid () % d_size;
    
    // extract the bucket
    s_bucket* bucket = rmbucket (&p_table[hid],name);
    delete bucket;
    d_count--;
  }

  // clear this hash table
  
  void HashTable::clear (void) {
    if (p_table != nilp) {
      for (long i = 0; i < d_size; i++) {
	delete p_table[i];
	p_table[i] = nilp;
      }
    }
  }  

  // return a vector of objects in this hash table

  Vector* HashTable::getvector (void) const {
    Vector* result = new Vector;
    rdlock ();
    for (long i = 0; i < d_size; i++) {
      s_bucket* bucket = p_table[i];
      while (bucket != nilp) {
	Object* obj = bucket->p_object;
	if (obj != nilp) result->append (obj);
	bucket = bucket->p_next;
      }
    }
    unlock ();
    return result;
  }  
  // resize the hash table by creating a new one
  
  void HashTable::resize (const long size) {
    // check for the size
    if (size < d_size) return;

    // initialize the new table
    s_bucket** table = new (s_bucket*)[size];
    for (long i = 0; i < size; i++) table[i] = nilp;

    // rebuild the table
    for (long i = 0; i < d_size; i++) {
      s_bucket* bucket = p_table[i];
      while (bucket != nilp) {
	s_bucket* next = bucket->p_next;
	bucket->p_next = nilp;
	long hid = bucket->d_hvl  % size;
	bucket->p_next = table[hid];
	table[hid]     = bucket;
	bucket = next;
      }
    }
    // clean the old table
    delete [] p_table;
    // restore the new table
    d_size  = size;
    d_thrs  = (d_size * 7) / 10;
    p_table = table;
  }
}
