// ---------------------------------------------------------------------------
// - Cursor.cpp                                                              -
// - standard object library - character buffer class implementation         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cursor.hpp"

namespace aleph {

  // default buffer size
  const long BUFFER_SIZE = 80;

  // create a new buffer class with a default size of 1024 characters

  Cursor::Cursor (void) {
    p_buffer = new char [BUFFER_SIZE];
    d_size   = BUFFER_SIZE;
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    d_insert = true;
  }

  // create a new buffer with a predefined size

  Cursor::Cursor (const long size) {
    d_size   = (size <= 0) ? BUFFER_SIZE : size;
    p_buffer = new char[d_size];
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    d_insert = true;
  }

  // create a new buffer and initialize it with a c string

  Cursor::Cursor (const char* value) {
    d_size   = BUFFER_SIZE;
    p_buffer = new char[d_size];
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    d_insert = true;
    // initialize with the string
    String data = value;
    add (data);
  }

  // create a new buffer and initialize it with a string

  Cursor::Cursor (const String& value) {
    d_size   = BUFFER_SIZE;
    p_buffer = new char[d_size];
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    d_insert = true;
    // initialize with the string
    add (value);
  }
  
  // destroy this buffer
  
  Cursor::~Cursor (void) {
    delete [] p_buffer;
  }
  
  // return the class name

  String Cursor::repr (void) const {
    return "Cursor";
  }

  // add a character in this buffer
  
  void Cursor::add (const char value) {
    wrlock ();
    // check for overflow
    if (length () == (d_size - 1)) resize (d_size * 2);
    // if we are in insert mode - we need to shift the buffer if
    // the cursor is not at the end - hence the while loop below
    if (d_insert == true) {
      if (d_cursor == d_end) {
	p_buffer[d_end] = value;
	d_end    = (d_end + 1) % d_size;
	d_cursor = d_end;
      } else {
	long di = d_end;
	while (di != d_cursor) {
	  long si = (di == 0) ? d_size - 1 : di - 1;
	  p_buffer[di] = p_buffer[si];
	  di = si;
	}
	p_buffer[d_cursor] = value;
	d_cursor = (d_cursor + 1) % d_size;
	d_end    = (d_end + 1) % d_size;
      }
    } else {
      if (d_cursor == d_end) {
	p_buffer[d_end] = value;
	d_end    = (d_end + 1) % d_size;
	d_cursor = d_end;
      } else {
	p_buffer[d_cursor] = value;
	d_cursor = (d_cursor + 1) % d_size;
      }
    }
    unlock ();
  }

  // add a buffer in this buffer
  
  void Cursor::add (const char* buffer, const long size) {
    if ((buffer == nilp) || (size == 0)) return;
    wrlock ();
    for (int i = 0; i < size; i++) add (buffer[i]);
    unlock ();
  }

  // add a string in this buffer
  
  void Cursor::add (const String& value) {
    wrlock ();
    try {
      int len = value.length ();
      for (int i = 0; i < len; i++) add (value[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }
  
  // get the next character but do not remove it

  char Cursor::get (void) const {
    rdlock ();
    char result = (d_start == d_end) ? nilc : p_buffer[d_start];
    unlock ();
    return result;
  }
  
  // read a character in this buffer
  
  char Cursor::read (void) {
    wrlock ();
    // check for no character
    if (d_start == d_end) {
      unlock ();
      return nilc;
    }
    // get value and adjust start
    char value = p_buffer[d_start];
    d_start = (d_start + 1) % d_size;
    unlock ();
    return value;
  }
  
  // pushback a character in this buffer
  
  void Cursor::pushback (const char value) {
    wrlock ();
    // check for overflow
    if (length () == (d_size - 1)) resize (d_size * 2);
    long index = (d_start == 0) ? d_size - 1 : d_start - 1;
    p_buffer[index] = value;
    if (d_cursor == d_start) d_cursor = index;
    d_start = index;
    unlock ();
  }

  // pushback a buffer in this buffer
  
  void Cursor::pushback (const char* buffer, const long size) {
    if ((buffer == nilp) || (size == 0)) return;
    wrlock ();
    try {
      int len = size - 1;
      for (int i = len; i >= 0; i--)
	pushback (buffer[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }
    
  // pushback a string in this buffer
  
  void Cursor::pushback (const String& value) {
    wrlock ();
    try {
      int len = value.length () - 1;
      for (int i = len; i >= 0; i--)
	pushback (value[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }
  
  // remove a character before the cursor (backspace mode)
  
  bool Cursor::erase (void) {
    wrlock ();
    // do nothing if the cursor is at start
    if (d_cursor == d_start) {
      unlock ();
      return false;
    }
    // loop and move characters
    long si = d_cursor;
    while (si != d_end) {
      long di = (si == 0) ? d_size - 1 : si - 1;
      p_buffer[di] = p_buffer[si];
      si = (si + 1) % d_size;
    }
    d_cursor = (d_cursor == 0) ? d_size - 1 : d_cursor - 1;
    d_end    = (d_end    == 0) ? d_size - 1 : d_end    - 1;
    unlock ();
    return true;
  }
  
  // remove a character at the cursor (delete mode)
  
  bool Cursor::del (void) {
    wrlock ();
    // do nothing if empty or cursor is at end
    if ((d_start == d_end) || (d_cursor == d_end)) {
      unlock ();
      return false;
    }
    // loop and move characters
    long di = d_cursor;
    while (di != d_end) {
      long si = (di + 1) % d_size;
      p_buffer[di] = p_buffer[si];
      di = (di + 1) % d_size;
    }
    d_end = (d_end == 0) ? d_size - 1 : d_end - 1;
    unlock ();
    return true;
  }
  
  // kill one or several characters in the buffer
  
  void Cursor::kill (const long num) {
    wrlock ();
    for (long i = 0; i < num; i++)
      if (erase () == false) break;
    unlock ();
  }
  
  // return true if the cursor is moved one character left
  
  bool Cursor::movel (void) {
    wrlock ();
    if (d_cursor == d_start) {
      unlock ();
      return false;
    }
    d_cursor = (d_cursor == 0) ? d_size - 1 : d_cursor - 1;
    unlock ();
    return true;
  }
  
  // return true if the cursor is moved one character right
  
  bool Cursor::mover (void) {
    wrlock ();
    if (d_cursor == d_end) {
      unlock ();
      return false;
    }
    d_cursor = (d_cursor + 1) % d_size;
    unlock ();
    return true;
  }
  
  // move the cursor to the beginning of the buffer
  
  long Cursor::setcb (void) {
    wrlock ();
    long result = (d_cursor > d_start) ? d_cursor - d_start : 
                                         d_start - d_cursor;
    d_cursor = d_start;
    unlock ();
    return result;
  }

  // move the cursor to the end of the buffer

  long Cursor::setce (void) {
    wrlock ();
    long result = (d_cursor < d_end) ? d_end - d_cursor: d_cursor - d_end;
    d_cursor = d_end;
    unlock ();
    return result;
  }

  // set the insert mode flag
  
  void Cursor::setimode (const bool mode) {
    wrlock ();
    d_insert = mode;
    unlock ();
  }
  
  // return the length of this buffer
  
  long Cursor::length (void) const {
    rdlock ();
    long result = (d_end > d_start) ? d_end - d_start : d_start - d_end;
    unlock ();
    return result;
  }

  // return the cursor position in character position
  
  long Cursor::getcursor (void) const {
    rdlock ();
    long result = (d_cursor > d_start) ? d_cursor - d_start : 
                                         d_start - d_cursor;
    unlock ();
    return result;
  }
  
  // return the corresponding string accumulated in this buffer
  
  String Cursor::tostring (void) const {
    rdlock ();
    long len = length ();
    if (len == 0) {
      unlock ();
      return String ();
    }
    // create a temporary buffer to hold the characters
    char* buf = new char[len+1];
    long index = d_start;
    long pos   = 0;
    while (index != d_end) {
      buf[pos++] = p_buffer[index];
      index = (index + 1) % d_size;
    }
    buf[pos] = nilc;
    String result = buf;
    delete [] buf;
    unlock ();
    return result;
  }

  // extract a substring from a previous cursor position
  
  String Cursor::substr (const long start) const {
    rdlock ();
    long len = length ();
    // create a temporary buffer to hold the characters
    char* buf = new char[len+1];
    long index = start;
    long pos   = 0;
    while (index != d_end) {
      buf[pos++] = p_buffer[index];
      index = (index + 1) % d_size;
    }
    buf[pos] = nilc;
    String result = buf;
    delete [] buf;
    unlock ();
    return result;
  }
  
  // reset this buffer but do not change the size
  
  void Cursor::reset (void) {
    wrlock ();
    d_start  = 0;
    d_end    = 0;
    d_cursor = 0;
    unlock ();
  }

  // resize this cursor

  void Cursor::resize (const long size) {
    wrlock ();
    if ((size < 0) || (size <= d_size)) {
      unlock ();
      return;
    }
    // get cursor data
    String data = tostring    ();
    long   dlen = data.length ();
    long   cpos = getcursor ();
    // delete and replace old buffer
    delete [] p_buffer;
    p_buffer = new char[size];
    for (long i = 0; i < dlen; i++) p_buffer[i] = data[i];
    // restor buffer context
    d_size   = size;
    d_start  = 0;
    d_end    = dlen;
    d_cursor = cpos;
    unlock ();
  }
}
