//---------------------------------------------------------------------
//  ACOVEA -- Analysis of Compiler Options Via Evolution Algorithm
//
//  runacovea.cpp
//
//  A driver program from the ACOVEA genetic algorithm.
//---------------------------------------------------------------------
//
//  COPYRIGHT NOTICE, DISCLAIMER, and LICENSE:
//
//  This notice applies *only* to this specific expression of this
//  algorithm, and does not imply ownership or invention of the
//  implemented algorithm.
//  
//  If you modify this file, you may insert additional notices
//  immediately following this sentence.
//  
//  Copyright 2003, 2004 Scott Robert Ladd. All rights reserved.
//  All rights reserved, except as noted herein.
//
//  This computer program source file is supplied "AS IS". Scott Robert
//  Ladd (hereinafter referred to as "Author") disclaims all warranties,
//  expressed or implied, including, without limitation, the warranties
//  of merchantability and of fitness for any purpose. The Author
//  assumes no liability for direct, indirect, incidental, special,
//  exemplary, or consequential damages, which may result from the use
//  of this software, even if advised of the possibility of such damage.
//  
//  The Author hereby grants anyone permission to use, copy, modify, and
//  distribute this source code, or portions hereof, for any purpose,
//  without fee, subject to the following restrictions:
//  
//      1. The origin of this source code must not be misrepresented.
//  
//      2. Altered versions must be plainly marked as such and must not
//         be misrepresented as being the original source.
//  
//      3. This Copyright notice may not be removed or altered from any
//         source or altered source distribution.
//  
//  The Author specifically permits (without fee) and encourages the use
//  of this source code for entertainment, education, or decoration. If
//  you use this source code in a product, acknowledgment is not required
//  but would be appreciated.
//  
//  Acknowledgement:
//      This license is based on the wonderful simple license that
//      accompanies libpng.
//
//-----------------------------------------------------------------------
//
//  For more information on this software package, please visit
//  Scott's web site, Coyote Gulch Productions, at:
//
//      http://www.coyotegulch.com
//  
//-----------------------------------------------------------------------

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <iostream>
#include <iomanip>
#include <string>
#include <vector>
using namespace std;

#include "libcoyotl/command_line.h"
using namespace libcoyotl;

#include "acovea.h"
using namespace acovea;

//----------------------------------------------------------
// display program options and command line
void show_usage()
{
    cout << "usage: runacovea -config {name} -bench {name} [options]\n\n"
         << "essential options:\n"
         << "   -config {config file name}              (i.e., -config gcc34_opteron.acovea\n"
         << "   -bench {source base name}               (i.e., -bench almabench.c)\n"
         << "\noptions for tuning the evolutionary algorithm:\n"
         << "   -n {number of populations to create}\n"
         << "   -p {size of each population}\n"
         << "   -g {number of generations to run}\n"
         << "   -sr {survival rate}                     (between 0.0 and 1.0)\n"
         << "   -mr {mutation rate}                     (between 0.0 and 1.0)\n"
         << "   -cr {crossover rate}                    (between 0.0 and 1.0)\n"
         << "   -ir {immigration rate}                  (between 0.0 and 1.0)\n"
         << "   -fn                                     (no fitness scaling)\n"
         << "   -fl                                     (linear fitness scaling)\n"
         << "   -fe                                     (exponential fitness scaling)\n"
         << "   -fq                                     (quadratic fitness scaling)\n"
         << "   -fw                                     (windowed fitness scaling - default)\n"
         << "   -seed {random number seed}\n\n"
         << "example:\n"
         << "   runacovea -config gcc34_opteron.acovea -bench huffbench.c\n\n";
}
 
//----------------------------------------------------------
//  main program
int main(int argc, char * argv[])
{
    // display header
    cout << "\nrunacovea -- a command-line driver program for the Acovea framework\n\n";
            
    // parse command line
    command_line args(argc,argv);
    
    size_t number_of_pops  =   5;
    size_t population_size =  40;
    double survival_rate   =   0.10;
    double migration_rate  =   0.05;
    double mutation_rate   =   0.01;
    double crossover_rate  =   1.00;
    size_t generations     =  20;
    string bench_name;
    string config_name;
    scaling_mode scaling   = SCALING_WINDOWED;
    
    string id;
    
    for (vector<command_line::option>::const_iterator opt = args.get_options().begin(); opt != args.get_options().end(); ++opt)
    {
        if (opt->m_name == "n")
        {
            number_of_pops = atol(opt->m_value.c_str());
            
            if (number_of_pops < 1)
                number_of_pops = 1;
        }
        else if (opt->m_name == "p")
        {
            population_size = atol(opt->m_value.c_str());
            
            if (population_size < 2)
                population_size = 2;
        }
        else if (opt->m_name == "sr") // survival rate
        {
            survival_rate = atof(opt->m_value.c_str());
            
            if (survival_rate < 0.0)
                survival_rate = 0.0;
                
            if (survival_rate > 1.0)
                survival_rate = 1.0;
        }
        else if (opt->m_name == "ir") // survival rate
        {
            migration_rate = atof(opt->m_value.c_str());
            
            if (migration_rate < 0.0)
                migration_rate = 0.0;
                
            if (migration_rate > 0.9)
                migration_rate = 0.9;
        }
        else if (opt->m_name == "mr")
        {
            mutation_rate = atof(opt->m_value.c_str());
            
            if (mutation_rate < 0.0)
                mutation_rate = 0.0;
                
            if (mutation_rate > 0.95)
                mutation_rate = 0.95;
        }
        else if (opt->m_name == "cr")
        {
            crossover_rate = atof(opt->m_value.c_str());
            
            if (crossover_rate < 0.0)
                crossover_rate = 0.0;
                
            if (crossover_rate > 1.0)
                crossover_rate = 1.0;
        }
        else if (opt->m_name == "g")
        {
            generations = atol(opt->m_value.c_str());
            
            if (generations < 1)
                generations = 1;
        }
        else if (opt->m_name == "fn")
        {
            scaling = SCALING_NONE;
        }
        else if (opt->m_name == "fl")
        {
            scaling = SCALING_LINEAR;
        }
        else if (opt->m_name == "fe")
        {
            scaling = SCALING_EXPONENTIAL;
        }
        else if (opt->m_name == "fq")
        {
            scaling = SCALING_QUADRATIC;
        }
        else if (opt->m_name == "fw")
        {
            scaling = SCALING_WINDOWED;
        }
        else if (opt->m_name == "seed")
        {
            libevocosm::globals::set_random_seed(atol(opt->m_value.c_str()));
        }
        else if (opt->m_name == "help")
        {
            show_usage();
            exit(0);
        }
        else if (opt->m_name == "bench")
        {
            bench_name = opt->m_value;
        }
        else if (opt->m_name == "config")
        {
            config_name = opt->m_value;
        }
    }
    
    if ((config_name.length() == 0) || (bench_name.length() == 0))
    {
        cerr << "You didn't specify an benchmark or configuration, so here's some help.\n\n";
        show_usage();
        exit(1);
    }

    // verify existence of benchmark program
    struct stat bench_stat;
    
    if (stat(bench_name.c_str(),&bench_stat))
    {
        string temp_name(ACOVEA_BENCHMARK_DIR);
        temp_name += bench_name;
        
	    if (stat(temp_name.c_str(),&bench_stat))
        {
	        cerr << "Could not find that benchmark, so here's some help.\n\n";
    	    show_usage();
        	exit(1);
        }
        
        bench_name = temp_name;
    }
    
    // create compiler object
    compiler target(config_name);
    
    // create a world
    acovea_world world(bench_name,
                       target,
                       number_of_pops,
                       population_size,
                       survival_rate,
                       migration_rate,
                       mutation_rate,
                       crossover_rate,
                       scaling,
                       generations);

    try
    {    
        // run the world
        world.run();
    }
    catch (std::exception & ex)
    {
        cerr << "runacovea: " << ex.what() << "\n";
    }
    catch (...)
    {
        cerr << "runacovea: unknown exception\n";
    }
    
    // outa here
    return 0;
}
