; Eric Smith, David Russinoff, with contributions and suggestions by Matt Kaufmann
; AMD, June 2001 

(in-package "ACL2")

(include-book "merge")

(in-theory (enable bits-tail))

(in-theory (disable expt)) ;test;test worked. move this up

;;Encoding of floating-point numbers with explicit leading one:
;;bit vectors of length p+q+1, consisting of 1-bit sign field, 
;;q-bit exponent field (bias = 2**(q-1)-1), and p-bit significand field.



; bias of a q bit exponent field is 2^(q-1)-1 
(defun bias (q) (- (expt 2 (- q 1)) 1) )

(in-theory (disable bias))

(defun esgnf  (x p q) (bitn x (+ p q)))
(defun eexpof (x p q) (bits x (1- (+ p q)) p)) 
(defun esigf  (x p)   (bits x (1- p) 0))

;;;**********************************************************************
;;;                       REPRESENTABLE NUMBERS
;;;**********************************************************************

; The next three predicates tell which values for each field of an 
; encoding can be represented.

(defun erep-sgnp (x)  (or (= x 1) (= x -1)) )

(defun erep-expop (x q) 
  (and (integerp x)
       (bvecp (+ x (bias q)) q) ;doc suggested this phrasing
       ))

(defun erep-sigp (x p)
  (and (rationalp x)
       (<= 1 x)
       (< x 2)
       (exactp x p) ))



(defun erepp2 (x p q)
  (and
   (erep-sgnp (sgn x))
   (erep-expop (expo x) q)
   (erep-sigp (sig x) p)))

(defun erepp (x p q)
  (and (rationalp x)
       (not (= x 0))
       (bvecp (+ (expo x) (bias q)) q)
       (exactp x p)))





(defthm erepps-same
  (equal (erepp x p q) 
         (erepp2 x p q))
  :hints (("Goal" 
           :in-theory (enable bvecp exactp sig)
           :use (sig-upper-bound sig-lower-bound))))


;;;**********************************************************************
;;;                      VALID ENCODINGS
;;;**********************************************************************

; these three predicates take extracted fields as args
(defun eencoding-sgnfp  (x)    (or (= x 0) (= x 1)) ) 

(defun eencoding-expofp (x q)  (and (bvecp x q) ) )

(defun eencoding-sigfp  (x p)  (and (bvecp x p)
                                    (= (bitn x (- p 1)) 1) )) ; explicit leading 1

(defun eencodingp2 (x p q)
  (and
   (bvecp x (+ p q 1))
   (eencoding-sgnfp (esgnf x p q))
   (eencoding-expofp (eexpof x p q) q)
   (eencoding-sigfp (esigf x p) p)
   ) 
  )

(defun eencodingp (x p q)
  (and (bvecp x (+ p q 1))
       (= (bitn x (- p 1)) 1)))


(defthm bvecp-forward-to-nat
  (implies (bvecp x k)
           (natp x))
  :rule-classes :forward-chaining)


(defthm eencodingps-same
  (implies (and
            (integerp p)
            (> p 0)
            (integerp q)
            (> q 0))
           (iff (eencodingp x p q) (eencodingp2 x p q)))
  :hints (("Goal" :in-theory (enable bitn-bits) 
           :use ((:instance bitn-0-1 (n (+ p q)))))))
                     







;;;**********************************************************************
;;;                       EENCODE
;;;**********************************************************************



; sig, expo, and sgn are defined in float.lisp

; these three encoders act on the sig, sgn, or expo of a value
(defun eencode-sgn  (x)     (if (= x 1) 0 1) )
(defun eencode-expo (x q)   (+ x (bias q)) )
(defun eencode-sig  (x p)   (* x (expt 2 (- p 1))) )

(defun eencode2 (x p q)
  (cat (cat
        (eencode-sgn (sgn x))
        (eencode-expo (expo x) q)
        q)
       (eencode-sig (sig x) p)
       p) )

(defun eencode (x p q)
  (cat (cat
        (if (= (sgn x) 1) 0 1)
        (+ (expo x) (bias q))
        q)
       (* (sig x) (expt 2 (- p 1)))
       p) )


(defthm eencodes-same
  (equal (eencode x p q) (eencode2 x p q)))






;;;**********************************************************************
;;;                       EDECODE
;;;**********************************************************************

;these three decoders act on the individual fields
(defun edecode-sgn  (x)     (if (= x 0) 1 -1) )
(defun edecode-expo (x q)   (- x (bias q)) )
(defun edecode-sig  (x p)   (/ x (expt 2 (- p 1))) )

(defun edecode2 (x p q)
  (* (edecode-sgn (esgnf x p q))
     (expt 2 (edecode-expo (eexpof x p q) q))
     (edecode-sig (esigf x p) p))
  )

(defun edecode (x p q)
  (* (if (= (esgnf x p q) 0) 1 -1)
     (esigf x p)
     (expt 2 (+ 1 (- p) (eexpof x p q) (- (bias q))))))


(defthm edecodes-same
  (implies (and (eencodingp x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0) )
           (equal (edecode x p q) (edecode2 x p q)))
  :hints (("Goal" 
           :in-theory (set-difference-theories (enable bias)
                                               '(eexpof)))))



;;;**********************************************************************
;;;                      Encoding and Decoding are Inverses
;;;**********************************************************************

(defthm eencode-edecode-sgn
  (implies (eencoding-sgnfp x)
           (equal (eencode-sgn (edecode-sgn x))
                  x )))

(defthm eencode-edecode-expo
  (implies (eencoding-expofp x q)
           (equal (eencode-expo (edecode-expo x q) q)
                  x )))


(defthm eencode-edecode-sig
  (implies (eencoding-sigfp x q)
           (equal (eencode-sig (edecode-sig x q) q)
                  x )))



(defthm edecode-eencode-sgn
  (implies (erep-sgnp x)
           (equal (edecode-sgn (eencode-sgn x))
                  x )))


(defthm edecode-eencode-expo
  (implies (and (erep-expop x q))
           (equal (edecode-expo (eencode-expo x q) q)
                  x )))


(defthm edecode-eencode-sig
  (implies (and (erep-sigp x q))
           (equal (edecode-sig (eencode-sig x q) q)
                  x )))





;forward chaining rules for types


(defthm natp-eencode-sgn-forward
  (implies
   (erep-sgnp x)
   (natp (eencode-sgn x) ) )
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-sgn x))))
  )


(defthm bvecp-eencode-sgn-forward
  (implies
   (erep-sgnp x)
   (bvecp (eencode-sgn x) 1) )
  :rule-classes  ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-sgn x))))
)


(defthm eencode-sgn-forward
  (implies
   (erep-sgnp x)
   (eencoding-sgnfp (eencode-sgn x)) )
  :hints (("Goal" :in-theory (enable erep-sgnp))) ;remove?
  :rule-classes  ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-sgn x))))
)







(defthm natp-eencode-expo-forward
  (implies
   (erep-expop x q)
   (natp (eencode-expo x q) ) )
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-expo x q))))
)

(defthm bvecp-eencode-expo-forward
  (implies
   (erep-expop x q)
   (bvecp (eencode-expo x q) q) )
  :hints (("Goal" :in-theory (enable bvecp eencode-expo erep-expop)))
  :rule-classes  ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-expo x q))))
  )


(defthm eencode-expo-forward
  (implies
   (erep-expop x q)
   (eencoding-expofp (eencode-expo x q) q) )
  :hints (("Goal" :in-theory (enable bvecp eencode-expo erep-expop)))
  :rule-classes  ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-expo x q))))
  )




(defthm erepp2-implies-rationalp-forward
  (implies
   (erepp2 x p q)
   (rationalp x) )
  :hints (("Goal" :in-theory (enable sig)))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((erepp2 x p q))))
  )


;move to somewhere else in library?
(defthm bitn-1
  (implies
   (and (bvecp x p)
        (integerp k)
        (>= k 0)
        (< k p)
        (equal (bitn x k) 1) )
   (>= x (expt 2 k)) )
  :hints (("Goal"
            :use ((:instance bitn-bvecp-0-thm (x x) (n k)))
            :in-theory (enable bvecp))) 
  :rule-classes nil
  )


(local
 (defthm hack-eric-3
  (IMPLIES (AND (<= (EXPT 2 (+ -1 P)) X)
                (INTEGERP P)
                (< 0 P)
                (BVECP X P)
                (> x 0) ; had to add
                (EQUAL (BITN X (+ -1 P)) 1))
           (<= 1 (* X (EXPT 2 (+ 1 (* -1 P))))))))



(defthm natp-eencode-sig
  (implies
   (and (erep-sigp x p)
        (integerp p)
        (> p 0))
   (natp (eencode-sig x p))) 
  :hints (("Goal" :in-theory (enable exactp bvecp)))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-sig x p))))
  )

(defthm bvecp-eencode-sig-forward
  (implies
   (and (erep-sigp x p)
        (integerp p)
        (> p 0))
   (bvecp (eencode-sig x p) p))
  :hints (("Goal" 
           :in-theory (enable expt bvecp eencode-sig erep-sigp exactp)))
  :rule-classes  ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-sig x p))))
  
  )


(defthm eencode-sig-forward
  (implies (and (erep-sigp x p)
                (integerp p)
                (> p 0))
           (eencoding-sigfp (eencode-sig x p) p) )
  :hints (("Goal"
           :in-theory (enable exactp bvecp expt)
           :use ((:instance bitn-force-1
                            (x (* X (EXPT 2 (+ -1 P))))
                            (n  (+ -1 P))
                            ))))
  :rule-classes  ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencode-sig x p)))))




(defthm edecode-sig-forward
  (implies
   (and (eencoding-sigfp x p)
        (integerp p)
        (> p 0) )
 (erep-sigp (edecode-sig x p) p) ) 
  :hints (("Goal"
           :in-theory (enable erep-sigp edecode-sig)
           :use ((:instance bitn-1 (x x) (p p) (k ( + -1 p)))
                 (:instance exactp-shift 
                            (x x)
                            (n (+ 1 (* -1 p)))
                            (m p)))))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sig x p)))))


(defthm rationalp-edecode-sig-forward
  (implies
   (and (eencoding-sigfp x p)
        (integerp p)
        (> p 0))
   (rationalp (edecode-sig x p)) ) 
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sig x p))))
)

(defthm not-0-edecode-sig-forward
  (implies
   (and (eencoding-sigfp x p)
        (integerp p)
        (> p 0))
   (not (equal (edecode-sig x p) 0)))
  :hints (("Goal" :use edecode-sig-forward))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sig x p))))
  )

(defthm lower-bound-edecode-sig-forward
  (implies (and
            (eencoding-sigfp x p)
            (integerp p)
            (> p 0))
   (<= 1 (edecode-sig x p)))
  :hints (("Goal" :use edecode-sig-forward))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sig x p))))
)


(defthm upper-bound-edecode-sig-forward
  (implies
   (and (eencoding-sigfp x p)
        (integerp p)
        (> p 0))
   (<  (edecode-sig x p) 2))
  :hints (("Goal" :use edecode-sig-forward))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sig x p))))
)

(defthm exactp-edecode-sig-forward
  (implies
   (and (eencoding-sigfp x p)
        (integerp p)
        (> p 0))
   (exactp (edecode-sig x p) p))
  :hints (("Goal" :use edecode-sig-forward))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sig x p))))
)

;could perhaps combine some of the above



(defthm edecode-expo-forward
  (implies
   (and (eencoding-expofp x q)
        (integerp q)
        (> q 0))
   (erep-expop (edecode-expo x q) q) )
  :hints (("Goal" :in-theory (enable bias)) )
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-expo x q))))
)


(defthm integerp-edecode-expo-forward
  (implies
   (and (eencoding-expofp x q)
        (integerp q)
        (> q 0))
   (integerp (edecode-expo x q)) )
  :hints (("Goal" :use edecode-expo-forward))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-expo x q))))
)



(defthm edecode-sgn-forward
  (implies
   (eencoding-sgnfp x)
   (erep-sgnp (edecode-sgn x)) )
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode-sgn x))))
  )



(defthm rationalp-edecode-forward
  (implies
   (eencodingp2 x p q)
   (rationalp (edecode2 x p q)) )
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((edecode2 x p q))))
  )


(defthm eencodingp2-implies-bvecp-forward
  (implies
   (eencodingp2 x p q)
   (bvecp x (+ p q 1)) )
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencodingp2 x p q))))
  )

(defthm eencodingp2-general-forward
  (implies
   
   (eencodingp2 x p q)
   (natp x))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencodingp2 x p q))))
)

(defthm eencoding-sigfp-forward
  (implies
   (eencoding-sigfp x p)
   (natp x))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencoding-sigfp x p))))
)

(defthm eencoding-expofp-forward
  (implies
   (eencoding-expofp x q)
   (natp x))
  :rule-classes ((:FORWARD-CHAINING :TRIGGER-TERMS ((eencoding-expofp x q))))
)





(defthm erep-sigp-forward-to-rationalp
  (implies (erep-sigp (sig x) p)
	   (rationalp x))
  :rule-classes :forward-chaining
  :hints (("Goal" :in-theory (enable erep-sigp sig))))

;move some of these disables up?
(in-theory (disable ;cat 
                   
                    expo 
                    sig 
                    sgn 
                   
                    eencode-sgn 
                    eencode-sig 
                    eencode-expo
                   
                    erep-sgnp 
                    erep-sigp 
                    erep-expop))
 




; commuting diagrams

(defthm esigf-eencode
  (implies (and (erepp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal
            (esigf (eencode2 x p q) p)
            (eencode-sig (sig x) p))))



(defthm eexpof-eencode
  (implies (and (erepp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal
            (eexpof (eencode2 x p q) p q)
            (eencode-expo (expo x) q))))



(defthm esgnf-eencode
  (implies (and (erepp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal
            (esgnf (eencode2 x p q) p q)
            (eencode-sgn (sgn x)))))



(in-theory (disable 
            edecode-sgn
            edecode-expo
            edecode-sig
            eencode2))

(in-theory (enable cat edecode2))


(defthm edecode2-eencode2
  (implies (and (erepp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal (edecode2 (eencode2 x p q) p q)
                  x ))
  :hints (("Goal" :use fp-rep)))



(in-theory (disable eexpof esigf esgnf eencoding-sigfp eencoding-expofp))
(in-theory (enable edecode2 erep-sigp erep-expop))

(defthm sig-edecode2
  (implies
   (and (eencodingp2 x p q)
        (integerp p)
        (> p 0)
        (integerp q)
        (> q 0))
   (equal
    (sig (edecode2 x p q))
    (edecode-sig (esigf x p) p)
    ))
  :hints (
          ("goal" 
           :in-theory (enable eencodingp2)
           :use ((:instance sig-expo-shift 
                            (x  (edecode-sig (esigf x p) p) )
                            (n  (edecode-expo (eexpof x p q) q)))))))

;expo of 2^x times a sig is x
;expo-shift?
(defthm expo-power-of-two-*-sig
  (implies (and (erep-sigp x p)
                (integerp p)
                (integerp n) )
           (equal (expo (* (expt 2 n) x))
                  n) )
  :hints (("goal" 
           :use sig-expo-shift
           :in-theory (enable expo)))
  :rule-classes nil)

(in-theory (disable exactp2 exactp2-lemma))

(defthm expo-edecode2
  (implies
   (and (eencodingp2 x p q)
        (integerp p)
        (> p 0)
        (integerp q)
        (> q 0))  
   (equal
    (expo (edecode2 x p q))
    (edecode-expo (eexpof x p q) q)
    ))
  :hints (("goal" :use (eencodingp2-general-forward 
                        (:instance expo-power-of-two-*-sig
                                   (x  (edecode-sig (esigf x p) p))
                                   (n  (edecode-expo (eexpof x p q) q)))))))

(in-theory (enable sgn))

(defthm sgn-edecode2
  (implies
   (and (eencodingp2 x p q)
        (integerp p)
        (> p 0)
        (integerp q)
        (> q 0))
   (equal
    (sgn (edecode2 x p q))
    (edecode-sgn (esgnf x p q)))))

(in-theory (disable cat edecode2 eencoding-sgnfp))
(in-theory (enable eencode2 esgnf eexpof esigf cat))

(defthm bits-plus-bits-eric
    (implies (and (natp x)
		  (natp r)
		  (natp n)
		  (natp m)
		  (> n r)
		  (> m n))
	     (= (+ (bits x (1- n) r)
		   (* (expt 2 (- n r)) (bits x (1- m) n)))
		(bits x (1- m) r)))
    :hints (("goal" :use bits-plus-bits))
  :rule-classes ())



(in-theory (disable bits-n-n-rewrite))


;add ?
;if leading bit is zero, can drop it
(defthm lead-bit-0
  (implies
   (and 
    (bvecp x (+ 1 n))
    (equal (bitn x n) 0)
    (natp n)
    (> n 0)
    )
   (equal (bits x (- n 1) 0)
          x ))
  :hints (("goal" :use ((:instance bits-plus-bitn
                                    (x x)
                                    (m 0)
                                    (n n) )))))


(local
 (defthm hack-eric
  (implies (and (equal (bitn x (+ p q)) 1)
                (bvecp x (+ 1 p q))
                (natp p)
                (< 0 p)
                (natp q)
                (< 0 q)
                (eencoding-expofp (bits x (+ -1 p q) p)
                                  q)
                (eencoding-sigfp (bits x (+ -1 p) 0) p) )
           (equal
            (+ (bits x (+ -1 p) 0)
               (* 2 (expt 2 (+ -1 p q)))
               (* 2 (expt 2 (+ -1 p))
                  (bits x (+ -1 p q) p)))
            (+ (bits x (+ -1 p q) 0)
               (* 2 (expt 2 (+ -1 p q)))) ))
  :hints (("goal" :in-theory (enable expt)
           :use ((:instance bits-plus-bits
                                    (x x)
                                    (m (+ p q)) 
                                    (n p)
                                    (r 0)))))))


(defthm eencode2-edecode2
  (implies (and (eencodingp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal (eencode2 (edecode2 x p q) p q)
                  x ))

  :hints (("Goal" :use (eencodingp2-general-forward
                        (:instance bitn-0-1 (x x) (n (+ p q)))
                        (:instance bits-plus-bits
                                   (x x)
                                   (m (+ p q)) 
                                   (n p)
                                   (r 0))
                        (:instance bits-plus-bitn
                                           (x x)
                                           (n (+ p q))
                                           (m 0))))))


(in-theory (enable bias))
(in-theory (disable esgnf esigf eexpof))


(defthm erepp2-edecode
  (implies (and (eencodingp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (erepp2 (edecode2 x p q) p q))
  :hints (("Goal" :in-theory (enable bvecp edecode-expo eexpof))))


(in-theory (disable erep-expop erep-sigp erep-sgnp cat sgn eencode2))

(defthm bvecp-eencode2
  (IMPLIES (AND (EREP-SGNP (SGN X))
                (EREP-EXPOP (EXPO X) Q)
                (EREP-SIGP (SIG X) P)
                (INTEGERP P)
                (< 0 P)
                (INTEGERP Q)
                (< 0 Q))
           (BVECP (EENCODE2 X P Q) (+ 1 P Q)))
  :hints (("Goal" :in-theory (enable eencode2 cat-bvecp))))

(defthm eencodingp2-eencode
  (implies (and (erepp2 x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (eencodingp2 (eencode2 x p q) p q)))

; for doc:

(in-theory (disable erepp2 eencodingp2 edecode2 eencode2 edecode eencode
                    eencodingp erepp))

 
(defthm erepp-edecode
  (implies (and (eencodingp x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (erepp (edecode x p q) p q)))


(defthm eencodingp-eencode
  (implies (and (erepp x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (eencodingp (eencode x p q) p q) ))


(defthm edecode-eencode
  (implies (and (erepp x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal (edecode (eencode x p q) p q)
                  x )))

(defthm eencode-edecode
  (implies (and (eencodingp x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (equal (eencode (edecode x p q) p q)
                  x )))

(defthm expo-edecode
  (implies
   (and (eencodingp x p q)
        (integerp p)
        (> p 0)
        (integerp q)
        (> q 0))  
   (equal
    (expo (edecode x p q))
    (- (eexpof x p q) (bias q))
    ))
  :hints (("Goal" :in-theory (enable edecode-expo))))




(defthm sgn-edecode
  (implies
   (and (eencodingp x p q)
        (integerp p)
        (> p 0)
        (integerp q)
        (> q 0))
   (equal
    (sgn (edecode  x p q))
    (if (= (esgnf x p q) 0) 1 -1)))
  :hints (("Goal" :in-theory (enable edecode-sgn))))


(defthm sig-edecode
  (implies
   (and (eencodingp  x p q)
        (integerp p)
        (> p 0)
        (integerp q)
        (> q 0))
   (equal
    (sig (edecode  x p q))
    (/ (esigf x p) (expt 2 (- p 1)))))
  :hints (("Goal" :in-theory (enable edecode-sig)))
)


;from Doc

(defthm eencodingp-not-zero
  (implies (and (eencodingp x p q)
                (integerp p)
                (> p 0)
                (integerp q)
                (> q 0))
           (not (equal (edecode x p q) 0)))
  :hints (("Goal" :in-theory (union-theories (disable 
                                              erepps-same 
                                              eencodingps-same 
                                              edecodes-same 
                                              erepp-edecode) 
                                             '(erepp))
           :use (erepp-edecode))))

(defun rebias-expo (expo old new)
  (+ expo (- (bias new) (bias old))))

;;I actually needed all four of the following lemmas, although I would have thought
;;that the two bvecp lemmas would be enough.

(defthm natp-rebias-up
    (implies (and (natp n)
		  (natp m)
		  (< 0 m)
		  (<= m n)
		  (bvecp x m))
	     (natp (rebias-expo x m n)))
  :hints (("goal" :in-theory (enable expt rebias-expo bvecp natp bias)
		  :use (:instance expt-weak-monotone (n m) (m n)))))

(defthm natp-rebias-down
    (implies (and (natp n)
		  (natp m)
		  (< 0 m)
		  (<= m n)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m)))))
	     (natp (rebias-expo x n m)))
    :hints (("goal" :in-theory (enable rebias-expo bvecp natp bias))))

(defthm bvecp-rebias-up
    (implies (and (natp n)
		  (natp m)
		  (< 0 m)
		  (<= m n)
		  (bvecp x m))
	     (bvecp (rebias-expo x m n) n))
  :hints (("goal" :in-theory (enable expt rebias-expo bvecp bias)
		  :use (:instance expt-weak-monotone (n m) (m n)))))

(defthm bvecp-rebias-down
    (implies (and (natp n)
		  (natp m)
		  (< 0 m)
		  (<= m n)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m)))))
	     (bvecp (rebias-expo x n m) m))
  :hints (("goal" :in-theory (enable expt rebias-expo bvecp bias))))

(local-defthm rebias-lemma-1
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 1))
	     (< (bits x (- n 2) 0) (expt 2 (1- m))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bits-plus-bitn (n (1- n)) (m 0))))))

(local-defthm rebias-lemma-2
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n))
	     (>= (bits x (- n 2) 0) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bits-bvecp (i (- n 2)) (j 0) (n (1- n))))
		  :in-theory (union-theories (disable bits-bvecp) '(natp bvecp)))))

(local-defthm rebias-lemma-3
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 1))
	     (bvecp (bits x (- n 2) 0) (1- m)))
  :rule-classes ()
  :hints (("Goal" :use (rebias-lemma-1
			rebias-lemma-2)
		  :in-theory (enable bvecp))))

(local-defthm rebias-lemma-4
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 1))
	     (equal (bits x (- n 2) 0)
		    (bits x (- m 2) 0)))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable natp)
                              '(bits-bits
                                ))
		  :use (rebias-lemma-3
			(:instance bits-bits (i (- n 2)) (j 0) (k (- m 2)) (l 0))))))

(local-defthm rebias-lemma-5
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 1))
	     (equal x (+ (expt 2 (1- n)) (bits x (- m 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories (enable natp) '(rebias-lemma-4))
		  :use (rebias-lemma-4
			(:instance bits-plus-bitn (n (1- n)) (m 0))))))

(local-defthm rebias-lemma-6
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 1))
	     (equal (rebias-expo x n m)
		    (cat (bitn x (1- n))
			 (bits x (- m 2) 0)
			 (1- m))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rebias-expo bias cat)
		  :use (rebias-lemma-5))))

(local-defthm rebias-lemma-7
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (equal (bits x (+ -2 n) 0)
		    x))
  :hints (("Goal" :use ((:instance bits-plus-bitn (n (1- n)) (m 0))))))

(local-defthm rebias-lemma-8
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (equal (bits x (+ -2 n) 0)
		    (+ (* (expt 2 (1- m))
			  (bits x (- n 2) (1- m)))
		       (bits x (- m 2) 0))))
  :hints (("Goal" :use ((:instance bits-plus-bits (m (1- n)) (n (1- m)) (r 0))))))

(local (in-theory (disable rebias-lemma-8)))

(local-defthm rebias-lemma-9
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n))
	     (and (integerp (bits x (- m 2) 0))
		  (< (bits x (- m 2) 0)
		     (expt 2 (1- m)))))
  :hints (("Goal" :use ((:instance bits-bvecp (i (- m 2)) (j 0) (n (1- m))))
		  :in-theory (union-theories (disable bits-bvecp) '(bvecp natp)))))

(local-defthm rebias-lemma-10
    (implies (and (integerp x)
		  (integerp y)
		  (< x y))
	     (<= x (1- y)))
  :rule-classes ())

(local-defthm rebias-lemma-11
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n))
	     (INTEGERP (EXPT 2 (+ -2 M))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp))))


(local-defthm rebias-lemma-12
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n))
	     (<= (bits x (- m 2) 0)
		 (1- (expt 2 (1- m)))))
  :hints (("Goal" :in-theory (union-theories (disable rebias-lemma-9 expt) '(natp))
		  :use (rebias-lemma-9
			rebias-lemma-11
			(:instance rebias-lemma-10 (x (bits x (- m 2) 0)) (y (expt 2 (1- m))))))))

(local-defthm rebias-lemma-13
              (implies (and (natp n)
                            (natp m)
                            (> n m)
                            (> m 1)
                            (bvecp x n)
                            (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
                            (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
                            (= (bitn x (1- n)) 0))
                       (<= (bits x (+ -2 n) 0)
                           (+ (* (expt 2 (1- m))
                                 (bits x (- n 2) (1- m)))
                              (1- (expt 2 (- m 1))))))
              :rule-classes ()
              :hints (("Goal" :in-theory (disable rebias-lemma-12)
                       :use (rebias-lemma-8
                             rebias-lemma-12))))

(local-defthm rebias-lemma-14
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (or (> (bits x (- n 2) (1- m))
		    (- (expt 2 (- n m)) 2))
		 (<= (bits x (- n 2) 0)
		     (+ (* (- (expt 2 (- n m)) 2)
			   (expt 2 (1- m)))
			(1- (expt 2 (1- m)))))))
  :rule-classes ()
  :hints (("Goal" :use (rebias-lemma-13))))

(local-defthm rebias-lemma-15
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (or (> (bits x (- n 2) (1- m))
		    (- (expt 2 (- n m)) 2))
		 (<= (bits x (- n 2) 0)
		     (+ (- (expt 2 (- n 1))
			   (* 2 (expt 2 (1- m))))
			(1- (expt 2 (1- m)))))))
  :rule-classes ()
  :hints (("Goal" :use (rebias-lemma-14
			(:instance expt+ (m (- n m)) (n (1- m)))))))

(local-defthm rebias-lemma-16
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (or (> (bits x (- n 2) (1- m))
		    (- (expt 2 (- n m)) 2))
		 (< (bits x (- n 2) 0)
		    (- (expt 2 (1- n)) (expt 2 (1- m))))))
  :rule-classes ()
  :hints (("goal" :use (rebias-lemma-15))))

(local-defthm rebias-lemma-17
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (iff (< x 
		     (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (< (bits x (+ -2 n) 0)
		     (- (expt 2 (1- n))
			(expt 2 (1- m))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rebias-lemma-7)
           :use (rebias-lemma-7))))

(local-defthm rebias-lemma-18
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (> (bits x (- n 2) (1- m))
		(- (expt 2 (- n m)) 2)))
  :rule-classes ()
  :hints (("Goal" :use (rebias-lemma-16
			rebias-lemma-17))))

(local-defthm rebias-lemma-19
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (and (integerp (bits x (- n 2) (1- m)))
		  (< (bits x (- n 2) (1- m)) (expt 2 (- n m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (union-theories (disable bits-bvecp) '(natp))
		  :use ((:instance bits-bvecp (i (- n 2)) (j (1- m)) (n (- n m)))))))

(local-defthm rebias-lemma-20
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1))
	     (integerp (expt 2 (- n m))))
  :rule-classes ())

(local-defthm rebias-lemma-21
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (= (bits x (- n 2) (1- m))
		(1- (expt 2 (- n m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable bvecp)
                              '(A14 EXPT-2-POSITIVE-INTEGER-TYPE NATP-EXPT  ;; RBK:
                                expt-2-integerp))
		  :use (rebias-lemma-18
			rebias-lemma-20
			rebias-lemma-19))))

(local-defthm rebias-lemma-22
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (= x
		(+ (* (1- (expt 2 (- n m)))
		      (expt 2 (1- m)))
		   (bits x (- m 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rebias-lemma-7)
           :use (rebias-lemma-21
			rebias-lemma-7
			(:instance bits-plus-bits (m (1- n)) (n (1- m)) (r 0))))))

(local-defthm rebias-lemma-23
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (= x
		(+ (- (expt 2 (1- m)))
		   (expt 2 (1- n))
		   (bits x (- m 2) 0))))
  :rule-classes ()
  :hints (("Goal" :use (rebias-lemma-22
			(:instance expt+ (m (- n m)) (n (1- m)))))))

(local-defthm rebias-lemma-24
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m))))
		  (= (bitn x (1- n)) 0))
	     (equal (rebias-expo x n m)
		    (cat (bitn x (1- n))
			 (bits x (- m 2) 0)
			 (1- m))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rebias-expo bias cat)
		  :use (rebias-lemma-23))))

(defthm rebias-down
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x n)
		  (< x (+ (expt 2 (1- n)) (expt 2 (1- m))))
		  (>= x (- (expt 2 (1- n)) (expt 2 (1- m)))))
	     (equal (rebias-expo x n m)
		    (cat (bitn x (1- n))
			 (bits x (- m 2) 0)
			 (1- m))))
  :rule-classes ()
  :hints (("Goal" :use (rebias-lemma-6
			rebias-lemma-24
			(:instance bitn-0-1 (n (1- n)))))))

(local-defthm rebias-up-1
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x m)
		  (= (bitn x (1- m)) 1))
	     (equal (cat (cat 1 0 (- n m))
			 (bits x (- m 2) 0)
			 (1- m))
		    (+ (expt 2 (1- n))
		       (bits x (- m 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable cat)
		  :use ((:instance expt+ (m (- n m)) (n (1- m)))))))

(local-defthm rebias-up-2
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x m)
		  (= (bitn x (1- m)) 1))
	     (equal (cat (cat 1 0 (- n m))
			 (bits x (- m 2) 0)
			 (1- m))
		    (rebias-expo x m n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rebias-expo bias)
		  :use (rebias-up-1
			(:instance bits-plus-bitn (n (1- m)) (m 0))))))

(local (include-book "integerp"))

(local-defthm rebias-up-3
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x m)
		  (= (bitn x (1- m)) 0))
	     (equal (cat (cat 0 (1- (expt 2 (- n m))) (- n m))
			 (bits x (- m 2) 0)
			 (1- m))
		    (rebias-expo x m n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable rebias-expo cat bias)
		  :use ((:instance bits-plus-bitn (n (1- m)) (m 0))
			(:instance expt+ (m (- n m)) (n (1- m)))))))

(defthm rebias-up
    (implies (and (natp n)
		  (natp m)
		  (> n m)
		  (> m 1)
		  (bvecp x m))
	     (equal (rebias-expo x m n)
		    (cat (cat (bitn x (1- m))
			      (mulcat 1 (- n m) (comp1 (bitn x (1- m)) 1))
			      (- n m))
			 (bits x (- m 2) 0)
			 (1- m))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp)
		  :use (rebias-up-2
			rebias-up-3
			(:instance bitn-0-1 (n (1- m)))))))