;;;***************************************************************
;;;an acl2 library of floating point arithmetic

;;;david m. russinoff
;;;advanced micro devices, inc.
;;;february, 1998
;;;***************************************************************

(in-package "ACL2")

(include-book "float")

(defun trunc (x n)
  (* (sgn x) (fl (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))

(defthm trunc-non-rationalp-is-0
  (implies (not (rationalp x))
           (equal (trunc x n)
                  0))
  :hints (("goal" :in-theory (enable trunc sig))))

(defthm trunc-to-0-or-fewer-bits
  (implies (and (integerp n)
                (<= n 0))
           (equal (trunc x n)
                  0))
  :hints (("goal" :in-theory (set-difference-theories
                              (enable trunc)
                              '(sig))
           :use ((:instance fl-unique
                            (x (* 1/2 (sig x) (expt 2 n)))
                            (n 0))
                 (:instance expt-weak-monotone
                            (n n)
                            (m 0))))))

(in-theory (disable trunc-to-0-or-fewer-bits))



(defthm trunc-minus
  (= (trunc (* -1 x) n) (* -1 (trunc x n))))

;(in-theory (disable trunc-minus))



;move up?
(defthm pos*
    (implies (and (rationalp x)
		  (rationalp y)
		  (> x 0)
		  (> y 0))
	     (> (* x y) 0))
  :rule-classes ())


(defthm trunc-pos
  (implies (and (rationalp x)
                (> x 0)
                (integerp n)
                (> n 0))
           (> (trunc x n) 0))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable expo sig fl-weakly-monotonic)
           :use ((:instance sig-lower-bound)
                 (:instance pos* 
                            (x (fl (* (sig x) (expt 2 (1- n))))) 
                            (y (expt 2 (- (1+ (expo x)) n))))
                 (:instance sgn+1)
                 (:instance expo-monotone (x 1) (y (1- n)))
                 (:instance n<=fl-linear (x (sig x)) (n 1))))))

(defthm trunc-0 
  (equal (trunc 0 n) 0)
  :hints (("goal" :in-theory (enable trunc))))


(defthm trunc-non-neg
  (implies (and (rationalp x)
                (>= x 0)
                (integerp n))
           (>= (trunc x n) 0))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable trunc)
           :cases ((> x 0)))
          ("subgoal 1" :use (trunc-pos trunc-to-0-or-fewer-bits))))


(defthm trunc-neg
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0))
	     (< (trunc x n) 0))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable expo sig fl-weakly-monotonic)
		  :use ((:instance sig-lower-bound)
			(:instance pos* 
				   (x (fl (* (sig x) (expt 2 (1- n))))) 
				   (y (expt 2 (- (1+ (expo x)) n))))
			(:instance sgn-1)
			(:instance expo-monotone (x 1) (y (1- n)))
			(:instance n<=fl-linear (x (sig x)) (n 1))))))


(defthm trunc-non-pos
  (implies (and (rationalp x)
                (<= x 0)
                (integerp n)
                )
           (<= (trunc x n) 0))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable trunc)
           :cases ((< x 0)))
          ("subgoal 1" :use (trunc-neg trunc-to-0-or-fewer-bits))))

(defthm sgn-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (trunc x n))
		    (sgn x)))
    :hints (("goal" :in-theory (enable sgn)
             :use ((:instance trunc-pos)
                   (:instance trunc-neg)
                   (:instance trunc-0)))))



(defthm abs-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0)
                  )
	     (equal (abs (trunc x n)) (* (fl (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n)))))
    :hints (("goal" :in-theory (disable expo sig fl-weakly-monotonic)
             :use ((:instance sig-lower-bound)
                   (:instance pos* 
                              (x (fl (* (sig x) (expt 2 (1- n))))) 
                              (y (expt 2 (- (1+ (expo x)) n))))
                   (:instance sgn-1)
                   (:instance sgn+1)
                   (:instance expo-monotone (x 1) (y (1- n)))
                   (:instance n<=fl-linear (x (sig x)) (n 1))))))

(defthm trunc-upper-bound
    (implies (and (rationalp x)
		  (integerp n))
	     (<= (abs (trunc x n)) (abs x)))
    :rule-classes :linear
    :hints (("goal" :in-theory (disable a15 abs expo sig trunc)
             :use (trunc-to-0-or-fewer-bits
                   (:instance fl-def-linear (x (* (expt 2 (1- n)) (sig x))))
                   (:instance sig-lower-bound)
                   (:instance *-weakly-monotonic
                              (x (expt 2 (- (1+ (expo x)) n)))
                              (y (fl (* (sig x) (expt 2 (1- n)))))
                              (y+ (* (sig x) (expt 2 (1- n)))))
                   (:instance fp-abs)
                   (:instance expo+ (m (1- n)) (n (- (1+ (expo x)) n)))))))

  #| subsumed 
(defthm rationalp-trunc
  (implies (and (rationalp x)
                (integerp n)
                (> n 0))
           (rationalp (trunc x n))))
|#

(defthm trunc-0-0
    (implies (and (rationalp x)
                  (integerp n)
                  (> n 0))
             (iff (= (trunc x n) 0)
                  (= x 0)))
    :rule-classes ()
    :hints (("goal" :in-theory (disable trunc)
             :use ((:instance trunc-pos)
                   (:instance trunc-neg)
                   (:instance trunc-0)))))

(defthm trunc-upper-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n))
	     (<= (trunc x n) x))
    :rule-classes :linear
    :hints (("goal" :in-theory (disable abs-trunc trunc)
             :use ((:instance trunc-upper-bound)
                   (:instance trunc-pos)
                   (:instance trunc-0-0)))))

(defthm expo-trunc-upper-bound
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (expo (trunc x n)) (expo x)))
    :rule-classes :linear
    :hints (("goal" :in-theory (disable expo sig trunc)
             :use ((:instance trunc-upper-bound)
                   (:instance trunc-0-0)
                   (:instance expo-monotone (x (trunc x n)) (y x))))))

(defthm expo-trunc-lower-bound
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (expt 2 (expo x))))
    :rule-classes :linear
    :hints (("goal" :in-theory (disable expo sig trunc)
             :use ((:instance sig-lower-bound)
                   (:instance *-weakly-monotonic
                              (y (expt 2 (1- n)))
                              (y+ (fl (* (sig x) (expt 2 (1- n)))))
                              (x (expt 2 (- (1+ (expo x)) n))))
                   (:instance expo+ (m (1- n)) (n (- (1+ (expo x)) n)))
                   (:instance fl-monotone-linear (x (expt 2 (1- n))) (y (* (expt 2 (1- n)) (sig x))))))))

(defthm expo-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (expo (trunc x n)) (expo x)))
    :hints (("goal" :in-theory (disable abs-trunc expo sig trunc)
             :use ((:instance expo-trunc-lower-bound)
                   (:instance expo-trunc-upper-bound)
                   (:instance trunc-0-0)
                   (:instance expo-upper-bound (x (trunc x n)))
                   (:instance expt-strong-monotone (n (expo x)) (m (1+ (expo (trunc x n)))))))))


(defthm trunc-lower-1-1
    (implies (and (rationalp x)
                  (integerp n))
             (= (abs x) (* (* (sig x) (expt 2 (1- n))) (expt 2 (- (1+ (expo x)) n)))))
    :rule-classes ()
    :hints (("goal" :in-theory (disable abs expo sig)
             :use ((:instance fp-abs)))))


(local
   (defthm trunc-lower-1-2
     (implies (and (rationalp u)
                   (rationalp v)
                   (rationalp r)
                   (> r 0)
                   (< u (1+ v)))
              (< (* r u) (* r (1+ v))))
     :rule-classes ()))


(defthm trunc-lower-1-3
    (implies (and (rationalp u)
                  (rationalp v)
                  (rationalp r)
                  (> r 0)
                  (< u (1+ v)))
             (< (* r u) (+ r (* r v))))
    :rule-classes ()
    :hints (("goal" :in-theory (disable *-strongly-monotonic)
             :use ((:instance trunc-lower-1-2)))))


(defthm trunc-lower-1
  (implies (and (rationalp x)
                (integerp n))
           (> (abs (trunc x n)) (- (abs x) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable abs expo sig trunc a15)
           :use ((:instance trunc-lower-1-1)
                 (:instance trunc-lower-1-3
                            (u (* (sig x) (expt 2 (1- n))))
                            (v (fl (* (sig x) (expt 2 (1- n)))))
                            (r (expt 2 (- (1+ (expo x)) n))))
                 ))
          ("subgoal 1" :in-theory (enable trunc))
;		(:instance fl-def-linear (x (* (expt 2 (1- n)) (sig
;                                                               x))))

          ))

(defthm trunc-lower-2-1
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (<= (expt 2 (- (1+ (expo x)) n)) (* (abs x) (expt 2 (- 1 n)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable abs expo sig a15)
		  :use ((:instance expo-lower-bound)
			(:instance expo+ (m (expo x)) (n (- 1 n)))))))

(defthm trunc-lower-2
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (> (abs (trunc x n)) (* (abs x) (- 1 (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable abs expo sig trunc a15)
		  :use ((:instance trunc-lower-1)
			(:instance trunc-lower-2-1)))))

(defthm trunc-lower-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (> (trunc x n) (* x (- 1 (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable abs-trunc abs-pos expo sig trunc a15)
		  :use ((:instance trunc-lower-2)
			(:instance trunc-pos)))))

(defthm trunc-lower-3
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (* (abs x) (- 1 (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable abs expo sig trunc a15)
		  :use ((:instance trunc-lower-1)
			(:instance trunc-0-0)
			(:instance trunc-lower-2-1)))))
(local
 (defthm trunc-lower-4-1
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (- (abs x) (* (abs x) (expt 2 (- 1 n))))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc abs-trunc)
		  :use ((:instance trunc-lower-3))))))

(local
 (defthm trunc-lower-4-2
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0))
	     (>= (trunc x n) x))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc abs-trunc)
		  :use ((:instance trunc-upper-bound)
			(:instance trunc-neg))))))

(defthm trunc-lower-4
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (trunc x n) (- x (* (abs x) (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("goal" :in-theory (disable trunc abs-trunc)
		  :use ((:instance trunc-lower-4-1)
			(:instance trunc-lower-4-2)
			(:instance trunc-pos)
			(:instance trunc-neg)
			(:instance trunc-0-0)))))

(defthm trunc-diff-1
    (implies (and (rationalp x)
		  (rationalp y)
		  (<= (abs y) (abs x))
		  (iff (< x 0) (< y 0)))
	     (= (abs (- x y)) (- (abs x) (abs y))))
  :rule-classes ())

(defthm trunc-diff
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (abs (- x (trunc x n))) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc abs expo abs-trunc)
		  :use ((:instance trunc-diff-1 (y (trunc x n)))
			(:instance trunc-neg)
			(:instance trunc-pos)
			(:instance trunc-0-0)
			(:instance trunc-upper-bound)
			(:instance trunc-lower-1)))))

(defthm trunc-diff-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (< (- x (trunc x n)) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc expo abs-trunc)
		  :use ((:instance trunc-diff)
			(:instance trunc-pos)
			(:instance trunc-upper-bound)))))


(defthm trunc-diff-expo-1
    (implies (and (rationalp x)
		  (not (= x (trunc x n)))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- x (trunc x n))) (- (expo x) n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc abs expo abs-trunc)
		  :use ((:instance trunc-diff)
			(:instance expo-lower-bound (x (- x (trunc x n))))
			(:instance expt-strong-monotone 
				   (n (expo (- x (trunc x n))))
				   (m (- (1+ (expo x)) n)))))))

(defthm trunc-rewrite
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (trunc x n)
		    (* (sgn x) 
		       (fl (* (expt 2 (- (1- n) (expo x))) (abs x))) 
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable trunc))

(local
 (defthm trunc-exactp-1
    (implies (and (rationalp x)
		  (integerp n))
	     (= x (* (sgn x) (* (expt 2 (- (1- n) (expo x))) (abs x)) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo)
		  :use ((:instance expo+ (n (- (1- n) (expo x))) (m (- (1+ (expo x)) n))))))))

(local
 (defthm trunc-exactp-2
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp z)
		  (not (= x 0))
		  (not (= z 0)))
	     (iff (= (* x y z) (* x (fl y) z))
		  (integerp y)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable fl-int fl-int-2 fl)
		  :use ((:instance fl-int-2 (x y))
			(:instance *cancell (x (fl y)) (z (* x z))))))))

(local
 (defthm trunc-exactp-3
    (implies (integerp x) (integerp (- x)))
  :rule-classes ()))

(local
 (defthm trunc-exactp-4
    (implies (rationalp x)
	     (equal (- (- x)) x))))


(defthm trunc-exactp-5
    (implies (rationalp x)
	     (iff (integerp x) (integerp (- x))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable a2)
		  :use ((:instance trunc-exactp-3)
			(:instance trunc-exactp-3 (x (- x)))))))

(local
 (defthm trunc-exactp-6
    (implies (and (rationalp x)
		  (integerp n))
	     (iff (exactp x n)
		  (integerp (* (abs x) (expt 2 (- (1- n) (expo x)))))))
  :rule-classes ()
  :hints (("goal" :in-theory (enable exactp2)
           :use ((:instance trunc-exactp-5 (x (* x (expt 2 (- (1- n) (expo x)))))))))))

(defthm trunc-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (trunc x n))
		  (exactp x n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo
                                      ;; The following was needed after changes
                                      ;; during development of ACL2 2.6 related
                                      ;; to assume-true-false.
                                      (:type-prescription trunc))
		  :use ((:instance trunc-exactp-1)
			(:instance trunc-exactp-6)
			(:instance trunc-exactp-2
				   (x (sgn x))
				   (y (* (expt 2 (- (1- n) (expo x))) (abs x)))
				   (z (expt 2 (- (1+ (expo x)) n))))))))

(defthm trunc-diff-expo
    (implies (and (rationalp x)
		  (not (exactp x n))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- x (trunc x n))) (- (expo x) n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc abs exactp2 expo abs-trunc)
		  :use ((:instance trunc-diff-expo-1)
			(:instance trunc-exactp-a)))))
(local 
 (defthm trunc-exactp-b-1    
    (implies (and (rationalp x)
		  (rationalp y)
                  (integerp n)
                  )
	     (integerp (* (* (sgn x) (fl y) (expt 2 (- (1- n) (expo x)))) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo)
		  :use ((:instance integerp-x-y 
				   (x (sgn x))
				   (y (fl (* (expt 2 (- (1- n) (expo x))) (abs x)))))
			(:instance expo+ (n (- (1- n) (expo x))) (m (- (1+ (expo x)) n))))))))

(local
 (defthm trunc-exactp-b-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0)
                  )
	     (integerp (* (trunc x n) (expt 2 (- (1- n) (expo x))))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo sgn)
		  :use ((:instance trunc-exactp-b-1 (y (* (expt 2 (- (1- n) (expo x))) (abs x)))))))))


(local (in-theory (enable exactp2)))

(defthm trunc-exactp-b
    (implies (and (rationalp x)
		  (integerp n))
	     (exactp (trunc x n) n))
  :hints (("goal" :in-theory (disable expo trunc-rewrite)
		  :use ((:instance trunc-exactp-b-2)
                        (:instance trunc-to-0-or-fewer-bits)))))


(defthm trunc-exactp-c
    (implies (and (rationalp x)
		  (integerp n)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (trunc x n)))
  :hints (("goal" :in-theory (disable expo exactp2 abs-trunc trunc-rewrite trunc-exactp-b)
		  :use ((:instance trunc-exactp-b)
			(:instance trunc-exactp-a)
			(:instance fp+1 (x (trunc x n)) (y a))
			(:instance trunc-lower-1)
                        (:instance trunc-upper-bound)
			(:instance trunc-pos)
                        (:instance only-0-is-0-or-negative-exact (x a))
                        trunc-non-neg))))

(local
 (defthm trunc-monotone-old
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (>= x 0)
		  (> n 0)
		  (<= x y))
	     (<= (trunc x n) (trunc y n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo exactp2 abs-trunc trunc-rewrite
                                      trunc-exactp-b trunc-exactp-c)
		  :use ((:instance trunc-exactp-b)
			(:instance trunc-upper-pos)
			(:instance trunc-exactp-c (x y) (a (trunc x n))))))))

(defthm trunc-monotone
  (implies (and (rationalp x)
                (rationalp y)
                (integerp n)
                (<= x y))
           (<= (trunc x n) (trunc y n)))
  :hints (("Goal" :in-theory (disable trunc trunc-rewrite trunc-upper-pos)
           :cases ((> n 0)))
          ("subgoal 1" 
           :use (trunc-monotone-old 
                 (:instance trunc-monotone-old (x (- y))
                            (y (- x)))))
          ("subgoal 2" :use (trunc-to-0-or-fewer-bits
                             (:instance trunc-to-0-or-fewer-bits (x y)))))

  :rule-classes :linear)


(defthm trunc-pos-rewrite
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (equal (trunc x n)
		    (* (fl (* (expt 2 (- (1- n) (expo x))) x))
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable trunc-rewrite))

(local
 (defthm trunc-trunc-1
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (* (expt 2 (- (1- m) (expo x)))
			  (* (fl (* (expt 2 (- (1- n) (expo x))) x))
			     (expt 2 (- (1+ (expo x)) n)))))
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo expo-trunc)
		  :use ((:instance trunc-pos)
			(:instance expo-trunc)
			(:instance expo-trunc (x (trunc x n)) (n m)))))))

(local
 (defthm trunc-trunc-2
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (* (fl (* (expt 2 (- (1- n) (expo x))) x)) (expt 2 (- m n)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-trunc-1)
			(:instance expo+ (n (- (1- m) (expo x))) (m (- (1+ (expo x)) n))))))))

;make local?
(defthm expt-inverse
    (implies (integerp n)
	     (equal (/ (expt 2 n))
		    (expt 2 (- n)))))

(local
 (defthm trunc-trunc-3
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (/ (fl (* (expt 2 (- (1- n) (expo x))) x)) (expt 2 (- n m)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-trunc-2))))))

(local
 (defthm trunc-trunc-4
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (/ (* (expt 2 (- (1- n) (expo x))) x) (expt 2 (- n m)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable fl/int-rewrite expo trunc-pos-rewrite
                                      trunc-rewrite integerp-expt-type)
           :use ((:instance integerp-expt-type (n (+ n (* -1 m))))
                        (:instance trunc-trunc-3)
			(:instance fl/int-rewrite 
				   (x (* (expt 2 (- (1- n) (expo x))) x))
				   (n (expt 2 (- n m)))))))))

(local
 (defthm trunc-trunc-5
   (implies (and (rationalp x)
                 (>= x 0)
                 (integerp n)
                 (integerp m)
                 (> m 0)
                 (>= n m))
            (= (trunc (trunc x n) m)
               (* (fl (* (expt 2 (- (1- m) (expo x))) x))
                  (expt 2 (- (1+ (expo x)) m)))))
   :rule-classes ()
   :hints (("goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
            :use ((:instance trunc-trunc-4))))))


;(local (in-theory (disable exactp2)))

(local
 (defthm trunc-trunc-old
    (implies (and ;(rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (>= n m))
	     (equal (trunc (trunc x n) m)
		    (trunc x m)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo trunc-non-rationalp-is-0)
		  :use ((:instance trunc-to-0-or-fewer-bits (n m))
                        (:instance trunc-to-0-or-fewer-bits (n m) (x (trunc x n)))
                        (:instance trunc-non-rationalp-is-0 (n m))
                        (:instance trunc-non-rationalp-is-0)
                        (:instance trunc-trunc-5))))))


(defthm trunc-trunc
  (implies (and (integerp n)
                (integerp m)
                (>= n m))
           (equal (trunc (trunc x n) m)
                  (trunc x m)))
  :hints (("goal" :in-theory (disable expo trunc-pos-rewrite)
           :use (trunc-trunc-old
                 (:instance trunc-trunc-old (x (- x)))))))

(in-theory (disable trunc-trunc))


;needed for plus-trunc
(defthm trunc-pos-rewrite-eric
  (implies (and (rationalp x)
                (>= x 0)
                (integerp n))
           (equal (trunc x n)
                  (* (fl (* (expt 2 (- (1- n) (expo x))) x))
                     (expt 2 (- (1+ (expo x)) n)))))
  :hints (("goal" :in-theory (enable trunc sgn)
           :use fp-abs)))


(local
 (defthm plus-trunc-2
  (implies (and (rationalp x)
                (> x 0)
                (rationalp y)
                (> y 0)
                (integerp k)
                (> k 0)
                (= n (+ k (- (expo x) (expo y))))
                (exactp x n))
           (equal (+ x (trunc y k))
                  (* (fl (* (+ x y) (expt 2 (- (1- k) (expo y)))))
                     (expt 2 (- (1+ (expo y)) k)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable fl+int-rewrite expo trunc-pos-rewrite trunc-rewrite)
           :use ((:instance fl+int-rewrite 
                            (x (* y (expt 2 (- (1- k) (expo y)))))
                            (n (* x (expt 2 (- (1- k) (expo y)))))))))))

(defthm plus-trunc
  (implies (and (rationalp x)
                (>= x 0)
                (rationalp y)
                (>= y 0)
                (integerp k)
                (exactp x (+ k (- (expo x) (expo y)))))
           (= (+ x (trunc y k))
              (trunc (+ x y) (+ k (- (expo (+ x y)) (expo y))))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo)
           :use ((:instance plus-trunc-2)
                 (:instance expo-monotone (y (+ x y)))))))

(in-theory (disable trunc-pos-rewrite-eric))

(defthm trunc-plus-1
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e)))
	     (< (expo y) e))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo)
		  :use ((:instance expo-lower-bound (x y))
			(:instance expt-strong-monotone (n (expo y)) (m e))))))

(defthm trunc-plus-2
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e)))
	     (< (+ (expt 2 e) y) (expt 2 (1+ e))))
  :rule-classes ())

(defthm trunc-plus-3
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e)))
	     (= (expo (+ (expt 2 e) y)) e))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo)
		  :use ((:instance expo-lower-bound (x (+ (expt 2 e) y)))
			(:instance expo-upper-bound (x (+ (expt 2 e) y)))
			(:instance trunc-plus-2)
			(:instance expt-strong-monotone (n (expo (+ (expt 2 e) y))) (m (1+ e)))
			(:instance expt-strong-monotone (n e) (m (1+ (expo (+ (expt 2 e) y)))))))))

(defthm trunc-plus-4
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e))
		  (integerp m)
		  (> m 0)
		  (integerp k)
		  (> k 0)
		  (<= m (1+ k)))
	     (= (+ (expt 2 e) (trunc y k))
		(trunc (+ (expt 2 e) y) (- (+ k e) (expo y)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite
                                      integerp-expt-type)
		  :use ((:instance integerp-expt-type (n (+ -1 e k (* -1 (expo y)))))
                        (:instance trunc-plus-1)
			(:instance trunc-plus-3)
			(:instance plus-trunc (x (expt 2 e)) )))))

(defthm trunc-plus
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e))
		  (integerp m)
		  (> m 0)
		  (integerp k)
		  (> k 0)
		  (<= m (1+ k)))
	     (= (trunc (+ (expt 2 e) (trunc y k)) m)
		(trunc (+ (expt 2 e) y) m)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-plus-4)
			(:instance trunc-plus-1)
			(:instance trunc-trunc (x (+ (expt 2 e) y)) (n (- (+ k e) (expo y))))))))

(defthm trunc-n+k-1
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= y (- x (trunc x n))))
	     (< y (expt 2 e)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-diff-pos)))))

(defthm trunc-n+k-2
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (trunc (+ (expt 2 e) y) (1+ k))
		(trunc (+ (expt 2 e) z) (1+ k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-1)
			(:instance trunc-upper-pos)
			(:instance trunc-plus (k n) (m (1+ k)))))))

(defthm trunc-n+k-3
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (and (equal (trunc x n) (* (fl (* x (expt 2 (- e)))) (expt 2 e)))
		  (equal (trunc x (+ n k)) (* (fl (* x (expt 2 (- k e)))) (expt 2 (- e k))))))
  :rule-classes ())

(defthm trunc-n+k-4
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (- (fl (* x (expt 2 (- k e))))
		      (* (expt 2 k) (fl (* x (expt 2 (- e))))))
		   (expt 2 (- e k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo))))

(defthm trunc-n+k-5
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (fl (- (* x (expt 2 (- k e)))
			  (* (expt 2 k) (fl (* x (expt 2 (- e)))))))
		   (expt 2 (- e k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite fl+int-rewrite)
		  :use ((:instance trunc-n+k-4)
			(:instance fl+int-rewrite 
				   (x (* x (expt 2 (- k e)))) 
				   (n (* (expt 2 k) (fl (* x (expt 2 (- e)))))))))))

(defthm trunc-n+k-6
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (fl (* y (expt 2 (- k e))))
		   (expt 2 (- e k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance trunc-n+k-5)))))

(defthm trunc-n+k-7
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (+ (expt 2 k) (fl (* y (expt 2 (- k e)))))
		      (expt 2 (- e k)))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-6)))))

(defthm trunc-n+k-8
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (fl (+ (expt 2 k) (* y (expt 2 (- k e)))))
		      (expt 2 (- e k)))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite fl+int-rewrite)
		  :use ((:instance trunc-n+k-7)
			(:instance fl+int-rewrite (x (* y (expt 2 (- k e)))) (n (expt 2 k)))))))

(defthm trunc-n+k-9
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (fl (* (expt 2 (- k e)) (+ y (expt 2 e))))
		      (expt 2 (- e k)))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-8)
			(:instance expo+ (n e) (m (- k e)))))))

(defthm trunc-n+k-10
    (implies (and (rationalp y)
		  (integerp e)
		  (<= 0 y))
	     (< 0 (+ y (expt 2 e))))
  :rule-classes ())

(defthm trunc-n+k-11
    (implies (and (integerp k)
		  (> k 0)
		  (rationalp y)
		  (> y 0)
		  (integerp e)
		  (= (expo (+ (expt 2 e) y)) e))
	     (= (* (fl (* (expt 2 (- k e)) (+ y (expt 2 e))))
		   (expt 2 (- e k)))
		(trunc (+ (expt 2 e) y) (1+ k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-10)
			(:instance trunc-pos-rewrite (x (+ y (expt 2 e))) (n (1+ k)))))))

(defthm trunc-n+k-12
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (trunc (+ (expt 2 e) y) (1+ k))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-9)
			(:instance trunc-n+k-1)
			(:instance trunc-n+k-11)
			(:instance trunc-plus-3)
			(:instance trunc-diff-pos)
			(:instance trunc-upper-pos)))))

(defthm trunc-n+k-13
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (sig (trunc (+ (expt 2 e) y) (1+ k))) (expt 2 e))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-12)
			(:instance trunc-n+k-1)
			(:instance trunc-n+k-11)
			(:instance trunc-plus-3)
			(:instance trunc-diff-pos)
			(:instance trunc-pos)
			(:instance trunc-upper-pos)))))

(defthm trunc-n+k-14
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
;		  (= y (- x (trunc x n))) ;removed by eric, had to mention y in
;the hints
                  )
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (1- (sig (trunc (+ (expt 2 e) z) (1+ k))))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable sig expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-2 (y (- x (trunc x n))))
			(:instance trunc-n+k-13 (y (- x (trunc x n))))))))

(defthm trunc-n+k
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (exactp x n))  		  ;;this isn't really needed, but it won't hurt me.
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
;		  (= y (- x (trunc x n))) ;removed
                  )
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (1- (sig (trunc (+ (expt 2 e) z) (1+ k))))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable sig expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-14)
			(:instance trunc-exactp-a)))))


;disable a15 first?
(defthm trunc-shift
  (implies (integerp n)
           (= (trunc (* x (expt 2 k)) n)
              (* (trunc x n) (expt 2 k))))
  :instructions
  (:promote (:dv 1)
            :expand :top (:dv 2 1)
            :expand :top (:casesplit (rationalp x))
            (:change-goal nil t)
            :prove (:casesplit (= x 0))
            :prove (:casesplit (integerp k))
            (:change-goal nil t)
            :prove (:dv 1 1)
            (:= (sgn (* x (expt 2 k)))
                (sgn x)
                :hints (("goal" :use sgn-shift)))
            :top (:dv 1 2 1 2)
            (:= (sig (* x (expt 2 k)))
                (sig x)
                :hints
                (("goal" :use (:instance sig-shift (n k)))))
            :top (:dv 1 3 2 1 2)
            (:= (expo (* x (expt 2 k)))
                (+ k (expo x))
                :hints
                (("goal" :use (:instance expo-shift (n k)))))
            :top (:dv 1 3)
            (:= (expt 2 (+ (+ 1 k (expo x)) (- n)))
                (* (expt 2 (+ (+ 1 (expo x)) (- n)))
                   (expt 2 k))
                :hints
                (("goal" :use
                  (:instance a15 (i 2)
                             (j1 k)
                             (j2 (+ (+ 1 (expo x)) (- n)))))))
            :top :prove))

(in-theory (disable trunc-shift))

(local (in-theory (disable sig expo)))

; The two local lemmas in the following encapsulate were added by Daron V. and
; Pete M. for v2-8 ordinals changes.
(encapsulate
 ()
 (local
  (defthm l1
    (implies (and (integerp (* 2 (expt 2 (+ (expo x) (* -1 n)))))
		  (rationalp x)
		  (integerp n)
		  (<= n (expo x))
		  (not (equal x 0))
		  (<= 0 x))
	     (integerp (* 2 (expt 2 (+ (expo x) (* -1 n)))
			  (fl (* (sig x) (expt 2 (+ -1 n)))))))
    :hints (("goal"
	     :in-theory (disable associativity-of-*)
	     :use ((:instance associativity-of-*
			      (x 2)
			      (y (EXPT 2 (+ (EXPO X) (* -1 N))))
			      (z (FL (* (SIG X) (EXPT 2 (+ -1 N)))))))))))
 (local
  (defthm l2
    (implies (and (integerp (* 2 (expt 2 (+ (expo x) (* -1 n)))))
		  (rationalp x)
		  (integerp n)
		  (<= n (expo x))
		  (not (equal x 0))
		  (< x 0))
	     (integerp (* 2 (expt 2 (+ (expo x) (* -1 n)))
			  -1
			  (fl (* (sig x) (expt 2 (+ -1 n)))))))
    :hints (("goal"
	     :in-theory (disable associativity-of-*)
	     :use ((:instance associativity-of-*
			      (x 2)
			      (y (EXPT 2 (+ (EXPO X) (* -1 N))))
			      (z (* -1 (FL (* (SIG X) (EXPT 2 (+ -1 N))))))))))
    :rule-classes :forward-chaining))

 (defthm int-trunc
   (implies (and (rationalp x)
		 (integerp n)
		 (>= (expo x) n))
	    (integerp (trunc x n)))
   :rule-classes :type-prescription
   :hints (("goal" 
	    :in-theory (e/d (trunc)
			    (integerp-expt-type
			     rationalp-expt-type-prescription
			     associativity-of-*))
	    :use ((:instance integerp-expt-type (n (- (1+ (expo x)) n))))))))

(in-theory (disable int-trunc))

(in-theory (disable 
            trunc-minus
            trunc-non-rationalp-is-0))


