/* ------------------------------------------------------------------------
 * $Id: Solid.cc,v 1.10 2001/08/28 13:18:52 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Nobel/SolidVisitor.hh"
#include "Nobel/Primitive.hh"
#include "Nobel/TriangleGeometry.hh"
#include "Nobel/Appearance.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Celsius/Vector3D.hh"
#include "Solid/Face.hh"
#include "Solid/BSP.hh"
#include "Solid/BSPTree.hh"
#include "Solid/BSPTreeBuilder.hh"
#include "Polhem/TransformImpl.hh"
#include "Polhem/Solid.hh"

using namespace Nobel;

// -- Class Definitions

// -- Code Segment

void SolidImpl::Node::receive(const Event &e)
{
    // Pass on the event to any listeners
    emit(e);
}

Solid::Node_ptr SolidImpl::Unary::body()
{
    // Clone the body reference and return it 
    Guard<Mutex> guard(_mutex);
    return Solid::Node::_duplicate(_body);
}

void SolidImpl::Unary::body(Solid::Node_ptr b)
{
    Guard<Mutex> guard(_mutex);

    // Detect nil references
    bool nil = CORBA::is_nil(_body);

    // If the references are equal, then just forget it
    if (nil || !_body->_is_equivalent(b)) {
	
	// Remove old listener, update reference and add listener again
	if (!nil) _body->removeListener(Listener_var(_this()));
	_body = Solid::Node::_duplicate(b);
	_body->addListener(Listener_var(_this()));

	// Signal that we have changed
	emitSubjectChanged();
    }
}

Solid::Node_ptr SolidImpl::Binary::left()
{
    // Clone and return reference
    Guard<Mutex> guard(_mutex);
    return Solid::Node::_duplicate(_left);
}

void SolidImpl::Binary::left(Solid::Node_ptr l)
{
    Guard<Mutex> guard(_mutex);

    // Detect nil references
    bool nil = CORBA::is_nil(_left);

    // If the references are equal, then just forget it
    if (nil || !_left->_is_equivalent(l)) {
	
	// Remove old listener, update reference and add listener again
	if (!nil) _left->removeListener(Listener_var(_this()));
	_left = Solid::Node::_duplicate(l);
	_left->addListener(Listener_var(_this()));

	// Signal that we have changed
	emitSubjectChanged();
    }
}

Solid::Node_ptr SolidImpl::Binary::right()
{
    // Clone and return reference
    Guard<Mutex> guard(_mutex);
    return Solid::Node::_duplicate(_right);
}

void SolidImpl::Binary::right(Solid::Node_ptr r)
{
    Guard<Mutex> guard(_mutex);

    // Detect nil references
    bool nil = CORBA::is_nil(_right);

    // If the references are equal, then just forget it
    if (nil || !_right->_is_equivalent(r)) {
	
	// Remove old listener, update reference and add listener again
	if (!nil) _right->removeListener(Listener_var(_this()));
	_right = Solid::Node::_duplicate(r);
	_right->addListener(Listener_var(_this()));
	
	// Signal that we changed
	emitSubjectChanged();
    }
}

SolidImpl::Geometry::Geometry(TriangleGeometry_ptr g)
    : _geo(TriangleGeometry::_duplicate(g))
{ 
    // empty
}

SolidImpl::Geometry::~Geometry()
{
    // empty
}

void SolidImpl::Geometry::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is a geometry
    v->visitGeometry(_this());
}

TriangleGeometry_ptr SolidImpl::Geometry::geo()
{
    // Clone and return reference
    Guard<Mutex> guard(_mutex);
    return TriangleGeometry::_duplicate(_geo);
}

void SolidImpl::Geometry::geo(TriangleGeometry_ptr g)
{
    Guard<Mutex> guard(_mutex);

    // Detect nil references
    bool nil = CORBA::is_nil(_geo);

    // If the references are equal, then just forget it
    if (nil || !_geo->_is_equivalent(g)) {
	
	// Remove old listener, update reference and add listener again
	if (!nil) _geo->removeListener(Listener_var(_this()));
	_geo = TriangleGeometry::_duplicate(g);
	_geo->addListener(Listener_var(_this()));

	// Signal that we changed
	emitSubjectChanged();
    }
}

void SolidImpl::Geometry::receive(const Event &e, Emitter_ptr o)
{
    // Transform the geometry event to a child changed event
    emitSubjectChanged();
}

SolidImpl::Primitive::Primitive(Primitive_ptr p)
    : _prim(Nobel::Primitive::_duplicate(p))
{ 
    // empty
}

SolidImpl::Primitive::~Primitive()
{
    // empty
}

void SolidImpl::Primitive::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is a primitive
    v->visitPrimitive(_this());
}

Primitive_ptr SolidImpl::Primitive::prim()
{
    // Clone and return reference
    Guard<Mutex> guard(_mutex);
    return Nobel::Primitive::_duplicate(_prim);
}

void SolidImpl::Primitive::prim(Primitive_ptr p)
{
    Guard<Mutex> guard(_mutex);

    // If the references are equal, then just forget it
    if (CORBA::is_nil(_prim) || !_prim->_is_equivalent(p)) {
	
	// Update reference 
	_prim = Nobel::Primitive::_duplicate(p);

	// Signal that we changed
	emitSubjectChanged();
    }
}

SolidImpl::Transform::Transform()
    : _transform(new TransformImpl)
{
    // empty
}

SolidImpl::Transform::~Transform()
{
    // Deactivate and deallocate CORBA entities
    deactivate(_transform);
    delete _transform;
}

void SolidImpl::Transform::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is a transform
    v->visitTransform(Solid::Transform_var(_this()));
}

void SolidImpl::Transform::identity()
{
    // Set the transform to identity
    Guard<Mutex> guard(_mutex);
    _transform->identity();
    emitSubjectChanged();
}

void SolidImpl::Transform::scale(const Vertex3D &s)
{
    // Scale the transform
    Guard<Mutex> guard(_mutex);
    _transform->scale(s);
    emitSubjectChanged();
}

void SolidImpl::Transform::translate(const Vertex3D &t)
{
    // Translate transform
    Guard<Mutex> guard(_mutex);
    _transform->translate(t);
    emitSubjectChanged();
}

void SolidImpl::Transform::rotate(CORBA::Float angle, const Vertex3D &axis)
{
    // Rotate transform
    Guard<Mutex> guard(_mutex);
    _transform->rotate(angle, axis);
    emitSubjectChanged();
}

void SolidImpl::Transform::getTransformationMatrix(Matrix m)
{
    // Store the transformation data into the passed matrix
    Guard<Mutex> guard(_mutex);
    _transform->store(m);
}

SolidImpl::Appearance::Appearance(Nobel::Appearance_ptr a) 
    : _app(Nobel::Appearance::_duplicate(a))
{
    // empty
}

void SolidImpl::Appearance::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is an appearance
    v->visitAppearance(Solid::Appearance_var(_this()));
}

Appearance_ptr SolidImpl::Appearance::app()
{
    // Clone and return appearance reference
    Guard<Mutex> guard(_mutex);
    return Nobel::Appearance::_duplicate(_app);
}

void SolidImpl::Appearance::app(Nobel::Appearance_ptr a)
{
    Guard<Mutex> guard(_mutex);
    
    // If the references are equal, then just forget it
    if (CORBA::is_nil(_app) || !_app->_is_equivalent(a)) {
	
	// Update reference and signal the change
	_app = Nobel::Appearance::_duplicate(a);
	emitSubjectChanged();
    }
}

void SolidImpl::Complement::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is a complement
    v->visitComplement(Solid::Unary_var(_this()));
}

void SolidImpl::Union::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is a union
    v->visitUnion(Solid::Binary_var(_this()));
}

void SolidImpl::Intersection::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is an intersection
    v->visitIntersection(Solid::Binary_var(_this()));
}

void SolidImpl::Subtraction::accept(Solid::Visitor_ptr v)
{
    // Tell the visitor that this is a subtraction
    v->visitSubtraction(Solid::Binary_var(_this()));
}
