/* ------------------------------------------------------------------------
 * $Id: ShapeImpl.cc,v 1.28 2001/08/28 13:18:51 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-08-16 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000, 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <algorithm>

// -- 3Dwm Includes
#include "Nobel/Event.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Polhem/TransformImpl.hh"
#include "Polhem/ImageImpl.hh"
#include "Polhem/VolumeImpl.hh"
#include "Polhem/PickerImpl.hh"
#include "Polhem/AppearanceImpl.hh"
#include "Polhem/ShapeImpl.hh"

using namespace Nobel;

// -- Code Segment

ShapeImpl::ShapeImpl()
    :  _dirty(true),
       _transform(new TransformImpl),
       _appearance(Appearance::_nil()),
       _bounds(new VolumeImpl)
{
    // empty
}

ShapeImpl::~ShapeImpl()
{
    // Deactivate CORBA entities
    deactivate(_transform);
    deactivate(_bounds);

    // Delete entities
    delete _transform;
    delete _bounds;
}
    
Transform_ptr ShapeImpl::transformation()
{
    // @@@ Should we really have a transformation for shapes?
    return _transform->_this();
}

Tag ShapeImpl::addGeometry(Geometry_ptr g)
{
    Guard<Mutex> guard(_mutex);
    static Tag tag_counter = 0;

    // Sanity check
    if (CORBA::is_nil(g) == true)
	return 0;

    // Add the pointer to the geometry vector
    geo_t node(Nobel::Geometry::_duplicate(g), tag_counter++);
    _geolist.push_back(node);

    // Add ourselves as listeners to the geometry
    g->addListener(Node_var(_this()));

    _dirty = true;
    
    return node.tag;
}

void ShapeImpl::removeGeometry(Tag t)
{
    Guard<Mutex> guard(_mutex);

    // Look for the geometry in the shape
    std::list<geo_t>::iterator r = std::find(_geolist.begin(),
					     _geolist.end(), t);

    // Remove the geometry *and* our listening to the geometry
    if (r != _geolist.end()) {
	_geolist.erase(r);
	r->item->removeListener(Node_var(_this()));
	_dirty = true;
    }
}

Appearance_ptr ShapeImpl::getAppearance()
{
    Guard<Mutex> guard(_mutex);
    return Appearance::_duplicate(_appearance);
}

void ShapeImpl::setAppearance(Appearance_ptr a)
{
    Guard<Mutex> guard(_mutex);
    _appearance = Appearance::_duplicate(a);
}

void ShapeImpl::render(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);
    
    // Apply the appearance (if we have one)
    if (!CORBA::is_nil(_appearance))
	_appearance->apply(r);
 
    // Render each of the geometries in the shape
    for (std::list<geo_t>::iterator i = _geolist.begin(); i != _geolist.end();
	 i++)
	i->item->render(r);
}

void ShapeImpl::pick(Picker_ptr r)
{
    Guard<Mutex> guard(_mutex);
    
    // If the shape is dirty, recompute the bounding volume
    if (_dirty)
	recomputeBoundingVolume();

    // Perform a pick with the bounding volume
    r->intersectsVolume(Volume_var(_bounds->_this()), Node_var(_this()));
}

Nobel::Volume_ptr ShapeImpl::getBoundingVolume()
{
    Guard<Mutex> guard(_mutex);
    
    // Recompute the bounding volume if necessary
    if (_dirty)
	recomputeBoundingVolume();
    
    // Return the new volume
    return _bounds->_this();
}

void ShapeImpl::recomputeBoundingVolume()
{
    VolumeImpl volume;

    // Step through all the geometries and merge them together
    for (std::list<geo_t>::iterator i = _geolist.begin(); i != _geolist.end();
	 i++)
	volume.merge(i->item->getBoundingVolume());
    
    Vertex3D min, max;
    volume.getBounds(min, max);

    // Mark that the shape is clean once again
    _dirty = false;

    // Did the bounding volume change at all? 
    if (volume != *_bounds) {
	
	// Update the bounding volume 
	*_bounds = volume;

	// Emit the proper event
	Nobel::Event e;
	e.type = BoundingVolumeChanged;
	//emit(e);
    }
}
