/* ------------------------------------------------------------------------
 * Socket.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

// -- Local Includes
#include "Exception.hh"
#include "Socket.hh"

// -- Code Segment

InetAddress::InetAddress(int port)
    : Address(AF_INET)
{
    // Fill out the socket address struct
    _sin.sin_family      = AF_INET;
    _sin.sin_port        = htons(port);    
    _sin.sin_addr.s_addr = htonl(INADDR_ANY);
}

InetAddress::InetAddress(const char *host, int port)
    : Address(AF_INET)
{
    struct hostent *hentry;
    
    // Lookup the name address
    hentry = gethostbyname(host);
    
    // Fill out the socket address struct
    _sin.sin_family = hentry->h_addrtype;
    _sin.sin_port   = htons(port);
    memcpy(&_sin.sin_addr, hentry->h_addr, hentry->h_length);
}

InetAddress::InetAddress(const sockaddr_in &sin)
    : Address(AF_INET), _sin(sin)
{
    // empty
}    

Socket::Socket(int protocol)
    : _protocol(protocol)
{
    if ((_socket = ::socket(_protocol, SOCK_STREAM, 0)) < 0) 
	throw Exception("Socket::Socket()");
}

Socket::Socket(int fd, int protocol)
    : _socket(fd), _protocol(protocol)
{
    // empty
}

Socket::Socket(const Socket &s)
    : _socket(s._socket), _protocol(s._protocol)
{
    // empty
}

Socket::~Socket()
{
    // empty
}

void Socket::connect(const Address &addr)
{
    if (addr.getType() != getProtocol()) throw Exception("Socket::connect()");
	
    // Connect to the remote host
    if (::connect(_socket, addr.getAddr(), addr.getAddrLen()) < 0)
	throw Exception("Socket::connect()");
}
    
int Socket::send(const void *data, int len)
{
    int bytes_sent;
    
    // Write data to the socket
    if ((bytes_sent = ::write(_socket, data, len)) < 0)
	throw Exception("Socket::send()");

    return bytes_sent;
}

int Socket::receive(void *data, int len)
{
    int bytes_received;
    
    // Read data from the socket
    if ((bytes_received = ::read(_socket, data, len)) < 0)
	throw Exception("Socket::receive()");

    return bytes_received;
}

void Socket::shutdown(bool norecv = true, bool nosend = true)
{
    // Figure out the correct shutdown flag 
    int how = (norecv == true) ? ((nosend == true) ? 2 : 0) : 1;
    
    // Shutdown the socket
    if (::shutdown(_socket, how) < 0) throw Exception("Socket::shutdown()");
}

void Socket::close()
{
    ::close(_socket);
}

void Socket::listen(int backlog)
{
    if (::listen(_socket, backlog) < 0) throw Exception("Socket::listen()");
}

std::pair<Socket *, Address *> Socket::accept()
{
    struct sockaddr_in sin;
    int fd;
    
    // Get length of struct
    socklen_t len = sizeof(sin);

    // Accept any waiting connection (will block if socket is not
    // configured otherwise)
    if ((fd = ::accept(_socket, (struct sockaddr *) &sin, &len)) < 0)
	throw Exception("Socket::accept()");
    
    // Return the new socket and client address
    return std::pair<Socket *, Address *>
	(new Socket(fd), new InetAddress(sin));
}

void Socket::bind(const Address &addr)
{
    if (addr.getType() != getProtocol()) throw Exception("Socket::bind()");
    
    if (::bind(_socket, addr.getAddr(), addr.getAddrLen()) < 0)
	throw Exception("Socket::bind()");
}

void Socket::setNonblocking()
{
    if (fcntl(_socket, F_SETFL, O_NONBLOCK) < 0)
	throw Exception("Socket::fcntl()");
}
    
bool Socket::isSameHost() const
{
    struct sockaddr_in peeraddr, myaddr;
    socklen_t addrlen = sizeof(struct sockaddr_in);
    
    // Retrieve the address of the remote and local hosts
    if (::getpeername(_socket, (struct sockaddr *) &peeraddr, &addrlen) < 0)
	throw Exception("Socket::getpeername()");
    if (::getsockname(_socket, (struct sockaddr *) &myaddr, &addrlen) < 0)
	throw Exception("Socket::getsockname()");
    
    // Return with a comparison of the two
    return (peeraddr.sin_addr.s_addr == myaddr.sin_addr.s_addr);
}
