/* ------------------------------------------------------------------------
 * $Id: Nobel.hh,v 1.6 2001/07/31 14:51:08 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-11-09 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Nobel_hh_
#define _Nobel_hh_

// -- 3Dwm Includes
#include "Nobel/CORBA.hh"
#include "Nobel/Types.hh"
#include "Nobel/Server.hh"
#include "Nobel/Client.hh"

// -- Class Declarations

namespace Nobelxx {
    
    // -- Forward Declarations
    class ClientHandleImpl;

    // -- Inline Functions

    /**
     * Retrieve the internal name of a CORBA interface.
     *
     * @@@ Does this rely on an omniORB-specific member variable?
     **/
    template <class T> const char *name() { return T::_PD_repoId; }

    /// X axis constant.
    const Nobel::Vertex3D xaxis = { 1.0f, 0.0f, 0.0f };

    /// Y axis constant.
    const Nobel::Vertex3D yaxis = { 0.0f, 1.0f, 0.0f };

    /// Z axis constant.
    const Nobel::Vertex3D zaxis = { 0.0f, 0.0f, 1.0f };

    /**
     * Construct a Nobel 3D vertex. Convenience function to avoid
     * having to do this manually.
     *
     * @param x x coordinate.
     * @param y y coordinate.
     * @param z z coordinate.
     * @return 3D vertex representing the given coordinates.
     **/
    Nobel::Vertex3D Vertex3D(float x, float y, float z) {
	Nobel::Vertex3D p;
	p.x = x; p.y = y; p.z = z;
	return p;
    }

    /**
     * Construct a Nobel texture coordinate. Convenience function to
     * avoid having to do this manually.
     *
     * @param u u coordinate.
     * @param v v coordinate.
     * @return texture coordinate representing the given coordinates.
     **/
    Nobel::TexCoord TexCoord(float u, float v) {
	Nobel::TexCoord p;
	p.u = u; p.v = v;
	return p;
    }

    /**
     * Construct a Nobel color. Convenience function to avoid having
     * to do this manually.
     *
     * @param r red component.
     * @param g green component.
     * @param b blue component.
     * @param a alpha component.
     * @return color with the given components.
     **/
    Nobel::Color Color(float r, float g, float b, float a = 1.0f) {
	Nobel::Color c;
	c.red = r; c.green = g; c.blue = b; c.alpha = a;
	return c;
    }

    /**
     * Nobel client. This entity may be used by C++ 3Dwm applications to
     * hide the details of the display server connection process.
     *
     * @@@ Move the C++ support code to a separate lib?
     **/
    class Client {
	
    public:
	
	/**
	 * Constructor. This will initialize the connection to the 3Dwm
	 * display server (see the client programmer's manual for the
	 * exact details). By passing in a non-NULL value for the "ior"
	 * parameter, the client will use the IOR instead of the CORBA
	 * naming service to connect to the server.
	 *
	 * @param name application name.
	 * @param argc number of command-line parameters.
	 * @param argv array of command-line parameters.
	 * @param ior server object reference (0 if not used).
	 **/
	Client(const char *name, int &argc, char **argv, const char *ior = 0);
	
	/**
	 * Destructor. This will shutdown the connection to the display
	 * server and deallocate all client-side entities.
	 **/
	~Client();
	
	/**
	 * Resolve a server-side interface and return a object reference
	 * to it.
	 * 
	 * @param name the name of the interface.
	 **/
	template <class T> 
	typename T::_ptr_type resolve(const char *name) {
	    
	    // Resolve the object in the server
	    CORBA::Object_ptr object;
	    try {
		object = _server->resolve(name);
	    }
	    catch(const CORBA::Exception &e) {
		std::cerr << "Failed to resolve " << name << ": " << e 
			  << std::endl;
		return T::_nil();
	    }
	    
	    // Narrow the reference to the correct type
	    typename T::_var_type reference;
	    try {
		reference = T::_narrow(object);
	    }
	    catch (const CORBA::Exception &e) {
		std::cerr << "Unable to narrow reference: " << e << std::endl;
		throw 0;
	    }
	    
	    // Make sure that the narrow operation was successful
	    if (CORBA::is_nil(reference)) {
		std::cerr << "Reference has incorrect type" << std::endl;
		throw 0;
	    }
	    
	    // Return the reference as a proper object (releasing it)
	    return reference._retn();    	
	}

	/**
	 * Resolve a server-side interface and return a object
	 * reference to it. This version of the resolve() call does
	 * not require a name string to identify the desired
	 * interface, it uses the default name taken from the
	 * interface itself.
	 **/
	template <class T> 
	typename T::_ptr_type resolve() {
	    return resolve(name<T>());
	}
	
	/**
	 * Retrieve local scene graph root.
	 *
	 * @@@ Subject to change!
	 **/
	Nobel::Node_ptr root() { return _server->root(); }
	
    private:
	
	CORBA::ORB_var _orb;
	ClientHandleImpl *_client;
	Nobel::ServerHandle_var _server;
    };
    
    /**
     * Client handle implementation. This CORBA object is designed to act
     * as a "stupid" echoing object that is pinged intermittently by the
     * server to detect crashing clients. It may be extended in the future
     * to include client data polling as well.
     **/
    class ClientHandleImpl : public virtual POA_Nobel::ClientHandle,
			     public virtual PortableServer::RefCountServantBase {
	
    public:
	
	/**
	 * Constructor.
	 **/
	ClientHandleImpl() { }
	
	/**
	 * Destructor.
	 **/
	~ClientHandleImpl() { }
	
	// IDL functions
	virtual void ping() { }
	
    };
    
};


#endif /* Nobel.hh */
    
