/* ------------------------------------------------------------------------
 * $Id: RuleSet.hh,v 1.2 2001/06/25 12:28:34 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _RuleSet_hh_
#define _RuleSet_hh_

// -- System Includes
#include <vector>

// -- Zorn Includes
#include "Instance.hh"

namespace Zorn {

    // -- Forward Declarations
    class AST;
    class ASTVisitor;

    // -- Class Declarations

    /**
     * Rule collection class.
     **/
    class RuleSet {
    public:

	/**
	 * Memory-management instance class.
	 **/
	ZORN_INSTANCE_CLASS(RuleSet);
    
	/**
	 * Singleton access method.
	 **/
	static inline RuleSet *instance() {
	    if (_instance == 0) _instance = new RuleSet(); 
	    return _instance; 
	}
    
	/**
	 * Add a Zorn rule to the rule set.
	 *
	 * @param rule abstract syntax tree representing the rule.
	 **/
	void addRule(AST *rule);

	/**
	 * Add Zorn initialization actions to the rule set. Inits are
	 * called once to initialize the store with any needed values.
	 *
	 * @param init abstract syntax tree representing the init.
	 **/
	void addInit(AST *init);

	/**
	 * Retrieve the number of rules in the rule set.
	 *
	 * @return number of rules.
	 **/
	int getRuleCount() const {
	    return _rules.size(); 
	}

	/**
	 * Visit all rules in the rule set using an AST visitor.
	 *
	 * @param visitor AST visitor instance to use for traversal.
	 **/
	void visitAllRules(ASTVisitor *visitor);

	/**
	 * Visit all initializations in the rule set using an AST
	 * visitor.
	 *
	 * @param visitor AST visitor instance to use for traversal.
	 **/
	void visitAllInits(ASTVisitor *visitor);

	/**
	 * Clear all rules from the rule set.
	 **/
	void clearAllRules() { 
	    deleteAll(_rules);
	}

	/**
	 * Clear all inits from the rule set.
	 **/
	void clearAllInits() {
	    deleteAll(_inits);
	}

    private:
	
	/**
	 * Visit all nodes in a list of AST nodes using an AST
	 * visitor.
	 *
	 * @param vs list of AST nodes to visit.
	 * @param visitor AST visitor instance to use for traversal.
	 **/
	void RuleSet::visitAll(const std::vector<AST *> &vs,
			       ASTVisitor *visitor);

	/**
	 * Delete all the nodes in a list of AST nodes. The pointers
	 * will be deallocated and the list itself will be cleared.
	 *
	 * @param vs list of AST nodes to delete.
	 **/
	void RuleSet::deleteAll(std::vector<AST *> &vs);

	/**
	 * Constructor (private, since this is a singleton).
	 **/
	RuleSet();

	/**
	 * Destructor (private, since this is a singleton).
	 **/
	~RuleSet();
    
	static RuleSet *_instance;
	std::vector<AST *> _rules;
	std::vector<AST *> _inits;
    };

};

#endif /* RuleSet.hh */
