/* ------------------------------------------------------------------------
 * $Id: View.hh,v 1.10 2001/06/27 12:48:55 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-11-22 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _View_hh_
#define _View_hh_

// -- 3Dwm Includes
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"
#include "Celsius/Quaternion.hh"
#include "Nobel/ViewSlot.hh"
#include "Polhem/TransformImpl.hh"

// -- Forward Declarations
class NodeImpl;
class RendererImpl;

// -- Class Declarations

/**
 * View class. Represents a view, similar to a camera, centered on a
 * specific viewslot in the scene graph. Note that the view itself is
 * *not* part of the scene graph.
 **/
class View {

public:

    /**
     * View parameters struct. 
     **/
    struct ViewParameters {
	int width, height;	// Viewport dimensions
	float far, near;	// Distance to far and near clipping plane
	float fov;		// Field of view
	float aspect;		// Aspect ratio
    };

    /**
     * Constructor.
     **/
    View(RendererImpl *r);
    
    /**
     * Destructor.
     **/
    ~View();
    
    /**
     * Attach to a given view slot.
     *
     * @param slot the new view slot to attach to.
     **/
    void attachTo(Nobel::ViewSlot_ptr slot);

    /**
     * Render the scene graph from this view.
     * 
     * @param root the scene graph root to use when rendering.
     **/ 
    void render(NodeImpl *root);

    /**
     * Compute the view transform from the current view slot transform
     * and the view's rotation and distance from the slot.
     *
     * @return view transform matrix.
     **/
    Matrix3D computeViewTransform();

    /**
     * Rotate the current viewslot.
     *
     * @param angle angle (in degrees) to rotate.
     * @param axis axis to rotate around.
     **/
    void rotate(float angle, const Nobel::Vertex3D &axis);
    
    /**
     * Translate the current viewslot.
     *
     * @param v vector to use for translation.
     **/
    void translate(const Nobel::Vertex3D &v);

    /**
     * Orbit the view around the current viewslot using an angle and
     * an axis.
     *
     * @param angle angle (in degrees) to rotate.
     * @param axis axis to rotate around.
     **/
    void orbit(float angle, const Vector3D &axis);

    /**
     * Dolly the view in relation to the current viewslot. This
     * modifies the view's distance from the viewslot.
     *
     * @param delta delta value to apply to the current distance.
     **/
    void dolly(float delta);

    /**
     * Zoom the view in and out. This modifies the user's field of
     * view (thus the minimum value is close to zero and the maximum
     * close to 180 degrees). The field of view is specified in
     * degrees.
     *
     * @param delta delta value to apply to the current field of view.
     **/
    void zoom(float delta);

    /**
     * Track the view slot (i.e. move it in the plane perpendicular to
     * the view direction). 
     *
     * @param v vector to use for tracking movement.
     **/
    void track(const Vector2D &v);

    /**
     * Return a pointer to the renderer used by this view
     **/
    //RendererImpl *getRenderer() { return _renderer; }

    /**
     * Retrieve the view parameters.
     **/
    ViewParameters getViewParameters() const;
    
private:
    
    /// Current view slot (this is the view's position in the scene graph)
    Nobel::ViewSlot_var _currentSlot;

    /// Renderer instance (used for traversing the scene graph)
    RendererImpl *_renderer;

    /**
     * Rotation quaternion (represents view rotation in relation to
     * the view slot).
     **/
    Quaternion _rotation;

    /// Translation of the view in relation to the viewslot
    Vector3D _translation;
    
    /// Field of view setting (used for zooming in and out)
    float _fov;
    bool _fov_changed;
};

#endif /* View.hh */
